package ru.yandex.solomon.core.db.model;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author albazh
 */
@ParametersAreNonnullByDefault
public class ViewHistory {

    public static final String URL_PREFIX = "/";
    public static final String DEFAULT_URL = URL_PREFIX;

    private static final String INVALID_URL_MESSAGE = "url should starts with " + StringUtils.wrap(URL_PREFIX, '\'');

    @Nullable
    private String id;

    private final String login;
    private final String url;
    @Nullable
    private final String pageTitle;
    private final boolean pinned;

    ViewHistory(
        String id,
        @Nullable String login,
        @Nullable String url,
        @Nullable String pageTitle,
        @Nullable Boolean pinned)
    {
        this(
            Objects.requireNonNull(id, "id must be non null"),
            Nullables.orEmpty(login),
            Nullables.orDefault(url, DEFAULT_URL),
            pageTitle,
            Nullables.orFalse(pinned)
        );
    }

    public ViewHistory(String login, String url, @Nullable String pageTitle, boolean pinned) {
        this(null, login, url, pageTitle, pinned);
    }

    /**
     * Default constructor
     */
    public ViewHistory(@Nullable String id, String login, String url, @Nullable String pageTitle, boolean pinned) {
        // id may be null in case of id generation after inserting
        this.id = id;
        Preconditions.checkArgument(url.startsWith(URL_PREFIX), INVALID_URL_MESSAGE);
        this.login = login;
        this.url = url;
        this.pageTitle = pageTitle;
        this.pinned = pinned;
    }

    @Nonnull
    public ViewHistory withId(@Nonnull String id) {
        return new ViewHistory(id, getLogin(), getUrl(), getPageTitle(), isPinned());
    }

    @Nonnull
    public ViewHistory withPin(boolean pin) {
        return new ViewHistory(getId(), getLogin(), getUrl(), getPageTitle(), pin);
    }

    @Nonnull
    public ViewHistory withPageTitle(@Nonnull String title) {
        return new ViewHistory(getId(), getLogin(), getUrl(), title, isPinned());
    }

    @Nonnull
    public ViewHistory withUrl(@Nonnull String url) {
        return new ViewHistory(getId(), getLogin(), url, getPageTitle(), isPinned());
    }

    @Nullable
    public String getId() {
        return id;
    }

    public String getLogin() {
        return login;
    }

    public String getUrl() {
        return url;
    }

    public boolean isPinned() {
        return pinned;
    }

    @Nullable
    public String getPageTitle() {
        return pageTitle;
    }

    public String toUserString() {
        return ToStringBuilder.reflectionToString(this, ToStringStyle.NO_CLASS_NAME_STYLE);
    }
}
