package ru.yandex.solomon.core.db.model.graph;

import java.time.Instant;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.ParametersAreNullableByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.model.AbstractAuditable;
import ru.yandex.solomon.core.db.model.AuditableBuilder;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class Graph extends AbstractAuditable {

    private final String id;
    private final String projectId;
    private final String folderId;
    private final String name;
    private final String description;
    private final Selector[] parameters;
    private final GraphElement[] elements;
    private final GraphMode graphMode;
    private final SecondaryGraphMode secondaryGraphMode;
    private final String min;
    private final String max;
    private final boolean normalize;
    private final ColorSchemeType colorScheme;
    private final Boolean dropNans;
    private final Boolean stack;
    private final AggregationType aggr;
    private final InterpolationType interpolate;
    private final ScaleType scale;
    private final String numberFormat;
    private final String green;
    private final String yellow;
    private final String red;
    private final String violet;
    private final String greenValue;
    private final String yellowValue;
    private final String redValue;
    private final String violetValue;
    private final String sortByLabel;
    private final boolean asc;
    private final String limit;
    private final OverLinesTransform overLinesTransform;
    private final String percentiles;
    private final String bucketLabel;
    private final boolean ignoreInf;
    private final FilterOrder filter;
    private final FilterFunction filterBy;
    private final String filterLimit;
    private final GraphTransform transform;
    private final String movingWindow;
    private final String movingPercentile;
    private final DownsamplingMode downsampling;
    private final DownsamplingAggregationType downsamplingAggr;
    private final DownsamplingFillType downsamplingFill;
    private final boolean ignoreMinStepMillis;
    private final String grid;
    private final int maxPoints;
    private final boolean hideNoData;
    private final Double threshold;
    private final String generatedId;

    public Graph(
        String id,
        String projectId,
        @Nullable String folderId,
        String name,
        @Nullable String description,
        @Nullable Selector[] parameters,
        @Nullable GraphElement[] elements,
        @Nullable GraphMode graphMode,
        @Nullable SecondaryGraphMode secondaryGraphMode,
        @Nullable String min,
        @Nullable String max,
        @Nullable Boolean normalize,
        @Nullable ColorSchemeType colorScheme,
        @Nullable Boolean dropNans,
        @Nullable Boolean stack,
        @Nullable AggregationType aggr,
        @Nullable InterpolationType interpolate,
        @Nullable ScaleType scale,
        @Nullable String numberFormat,
        @Nullable String green,
        @Nullable String yellow,
        @Nullable String red,
        @Nullable String violet,
        @Nullable String greenValue,
        @Nullable String yellowValue,
        @Nullable String redValue,
        @Nullable String violetValue,
        @Nullable String sortByLabel,
        @Nullable Boolean asc,
        @Nullable String limit,
        @Nullable OverLinesTransform overLinesTransform,
        @Nullable String percentiles,
        @Nullable String bucketLabel,
        @Nullable Boolean ignoreInf,
        @Nullable FilterOrder filter,
        @Nullable FilterFunction filterBy,
        @Nullable String filterLimit,
        @Nullable GraphTransform transform,
        @Nullable String movingWindow,
        @Nullable String movingPercentile,
        @Nullable DownsamplingMode downsampling,
        @Nullable DownsamplingAggregationType downsamplingAggr,
        @Nullable DownsamplingFillType downsamplingFill,
        @Nullable Boolean ignoreMinStepMillis,
        @Nullable String grid,
        @Nullable Integer maxPoints,
        @Nullable Boolean hideNoData,
        @Nullable Double threshold,
        @Nullable String generatedId,
        @Nullable Instant createdAt,
        @Nullable Instant updatedAt,
        @Nullable String createdBy,
        @Nullable String updatedBy,
        @Nullable Integer version)
    {
        super(createdAt, updatedAt, createdBy, updatedBy, Nullables.orZero(version));
        this.id = Objects.requireNonNull(id, "id must not be null");
        this.projectId = Objects.requireNonNull(projectId, "project id must not be null");
        this.folderId = Nullables.orEmpty(folderId);
        this.name = Objects.requireNonNull(name, "name must not be null");
        this.description = Nullables.orEmpty(description);
        this.parameters = Nullables.orDefault(parameters, new Selector[0]);
        this.elements = Nullables.orDefault(elements, new GraphElement[0]);
        this.graphMode = Nullables.orDefault(graphMode, GraphMode.GRAPH);
        this.secondaryGraphMode = Nullables.orDefault(secondaryGraphMode, SecondaryGraphMode.PIE);
        this.min = Nullables.orEmpty(min);
        this.max = Nullables.orEmpty(max);
        this.normalize = Nullables.orFalse(normalize);
        this.colorScheme = Nullables.orDefault(colorScheme, ColorSchemeType.AUTO);
        this.dropNans = dropNans;
        this.stack = stack;
        this.aggr = Nullables.orDefault(aggr, AggregationType.AVG);
        this.interpolate = Nullables.orDefault(interpolate, InterpolationType.LINEAR);
        this.scale = Nullables.orDefault(scale, ScaleType.NATURAL);
        this.numberFormat = Nullables.orEmpty(numberFormat);
        this.green = Nullables.orEmpty(green);
        this.yellow = Nullables.orEmpty(yellow);
        this.red = Nullables.orEmpty(red);
        this.violet = Nullables.orEmpty(violet);
        this.greenValue = Nullables.orEmpty(greenValue);
        this.yellowValue = Nullables.orEmpty(yellowValue);
        this.redValue = Nullables.orEmpty(redValue);
        this.violetValue = Nullables.orEmpty(violetValue);
        this.sortByLabel = Nullables.orEmpty(sortByLabel);
        this.asc = Nullables.orFalse(asc);
        this.limit = Nullables.orEmpty(limit);
        this.overLinesTransform = Nullables.orDefault(overLinesTransform, OverLinesTransform.NONE);
        this.percentiles = Nullables.orEmpty(percentiles);
        this.bucketLabel = Nullables.orEmpty(bucketLabel);
        this.ignoreInf = Nullables.orFalse(ignoreInf);
        this.filter = Nullables.orDefault(filter, FilterOrder.NONE);
        this.filterBy = Nullables.orDefault(filterBy, FilterFunction.AVG);
        this.filterLimit = Nullables.orEmpty(filterLimit);
        this.transform = Nullables.orDefault(transform, GraphTransform.NONE);
        this.movingWindow = Nullables.orEmpty(movingWindow);
        this.movingPercentile = Nullables.orEmpty(movingPercentile);
        this.downsampling = Nullables.orDefault(downsampling, DownsamplingMode.AUTO);
        this.downsamplingAggr = Nullables.orDefault(downsamplingAggr, DownsamplingAggregationType.DEFAULT_VALUE);
        this.downsamplingFill = Nullables.orDefault(downsamplingFill, DownsamplingFillType.NULL);
        this.ignoreMinStepMillis = Nullables.orFalse(ignoreMinStepMillis);
        this.grid = Nullables.orEmpty(grid);
        this.maxPoints = Nullables.orZero(maxPoints);
        this.hideNoData = Nullables.orFalse(hideNoData);
        this.threshold = threshold;
        this.generatedId = Nullables.orEmpty(generatedId);
    }

    public Graph(Builder builder) {
        this(
            builder.getId(),
            builder.getProjectId(),
            builder.getFolderId(),
            builder.getName(),
            builder.getDescription(),
            builder.getParameters(),
            builder.getElements(),
            builder.getGraphMode(),
            builder.getSecondaryGraphMode(),
            builder.getMin(),
            builder.getMax(),
            builder.getNormalize(),
            builder.getColorScheme(),
            builder.getDropNans(),
            builder.getStack(),
            builder.getAggr(),
            builder.getInterpolate(),
            builder.getScale(),
            builder.getNumberFormat(),
            builder.getGreen(),
            builder.getYellow(),
            builder.getRed(),
            builder.getViolet(),
            builder.getGreenValue(),
            builder.getYellowValue(),
            builder.getRedValue(),
            builder.getVioletValue(),
            builder.getSortByLabel(),
            builder.getAsc(),
            builder.getLimit(),
            builder.getOverLinesTransform(),
            builder.getPercentiles(),
            builder.getBucketLabel(),
            builder.isIgnoreInf(),
            builder.getFilter(),
            builder.getFilterBy(),
            builder.getFilterLimit(),
            builder.getTransform(),
            builder.getMovingWindow(),
            builder.getMovingPercentile(),
            builder.getDownsampling(),
            builder.getDownsamplingAggr(),
            builder.getDownsamplingFill(),
            builder.isIgnoreMinStepMillis(),
            builder.getGrid(),
            builder.getMaxPoints(),
            builder.getHideNoData(),
            builder.getThreshold(),
            builder.getGeneratedId(),
            builder.getCreatedAt(),
            builder.getUpdatedAt(),
            builder.getCreatedBy(),
            builder.getUpdatedBy(),
            builder.getVersion()
        );
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getId() {
        return id;
    }

    public String getProjectId() {
        return projectId;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public Selector[] getParameters() {
        return parameters;
    }

    public GraphElement[] getElements() {
        return elements;
    }

    public GraphMode getGraphMode() {
        return graphMode;
    }

    public SecondaryGraphMode getSecondaryGraphMode() {
        return secondaryGraphMode;
    }

    public String getMin() {
        return min;
    }

    public String getMax() {
        return max;
    }

    public boolean isNormalize() {
        return normalize;
    }

    public ColorSchemeType getColorScheme() {
        return colorScheme;
    }

    @Nullable
    public Boolean isDropNans() {
        return dropNans;
    }

    @Nullable
    public Boolean getStack() {
        return stack;
    }

    public AggregationType getAggr() {
        return aggr;
    }

    public InterpolationType getInterpolate() {
        return interpolate;
    }

    public OverLinesTransform getOverLinesTransform() {
        return overLinesTransform;
    }

    public ScaleType getScale() {
        return scale;
    }

    public String getNumberFormat() {
        return numberFormat;
    }

    public String getGreen() {
        return green;
    }

    public String getYellow() {
        return yellow;
    }

    public String getRed() {
        return red;
    }

    public String getViolet() {
        return violet;
    }

    public String getGreenValue() {
        return greenValue;
    }

    public String getYellowValue() {
        return yellowValue;
    }

    public String getRedValue() {
        return redValue;
    }

    public String getVioletValue() {
        return violetValue;
    }

    public String getSortByLabel() {
        return sortByLabel;
    }

    public boolean isAsc() {
        return asc;
    }

    public String getLimit() {
        return limit;
    }

    public String getPercentiles() {
        return percentiles;
    }

    public String getBucketLabel() {
        return bucketLabel;
    }

    public boolean isIgnoreInf() {
        return ignoreInf;
    }

    public FilterOrder getFilter() {
        return filter;
    }

    public FilterFunction getFilterBy() {
        return filterBy;
    }

    public String getFilterLimit() {
        return filterLimit;
    }

    public GraphTransform getTransform() {
        return transform;
    }

    public String getMovingWindow() {
        return movingWindow;
    }

    public String getMovingPercentile() {
        return movingPercentile;
    }

    public DownsamplingMode getDownsampling() {
        return downsampling;
    }

    public DownsamplingAggregationType getDownsamplingAggr() {
        return downsamplingAggr;
    }

    public DownsamplingFillType getDownsamplingFill() {
        return downsamplingFill;
    }

    public boolean isIgnoreMinStepMillis() {
        return ignoreMinStepMillis;
    }

    public String getGrid() {
        return grid;
    }

    public int getMaxPoints() {
        return maxPoints;
    }

    public boolean isHideNoData() {
        return hideNoData;
    }

    @Nullable
    public Double getThreshold() {
        return threshold;
    }

    public String getGeneratedId() {
        return generatedId;
    }

    @ParametersAreNullableByDefault
    public static final class Builder extends AuditableBuilder<Builder> {
        private String id;
        private String projectId;
        private String folderId = StringUtils.EMPTY;
        private String name;
        private String description = StringUtils.EMPTY;
        private Selector[] parameters = new Selector[0];
        private GraphElement[] elements = new GraphElement[0];
        private GraphMode graphMode = GraphMode.GRAPH;
        private SecondaryGraphMode secondaryGraphMode = SecondaryGraphMode.PIE;
        private String min = StringUtils.EMPTY;
        private String max = StringUtils.EMPTY;
        private boolean normalize = false;
        private ColorSchemeType colorScheme = ColorSchemeType.AUTO;
        private Boolean dropNans;
        private Boolean stack;
        private AggregationType aggr = AggregationType.AVG;
        private InterpolationType interpolate = InterpolationType.LINEAR;
        private ScaleType scale = ScaleType.NATURAL;
        private String numberFormat = StringUtils.EMPTY;
        private String green = StringUtils.EMPTY;
        private String yellow = StringUtils.EMPTY;
        private String red = StringUtils.EMPTY;
        private String violet = StringUtils.EMPTY;
        private String greenValue = StringUtils.EMPTY;
        private String yellowValue = StringUtils.EMPTY;
        private String redValue = StringUtils.EMPTY;
        private String violetValue = StringUtils.EMPTY;
        private String sortByLabel = StringUtils.EMPTY;
        private boolean asc = false;
        private String limit = StringUtils.EMPTY;
        private OverLinesTransform overLinesTransform = OverLinesTransform.NONE;
        private String percentiles = StringUtils.EMPTY;
        private String bucketLabel = StringUtils.EMPTY;
        private boolean ignoreInf = false;
        private FilterOrder filter = FilterOrder.NONE;
        private FilterFunction filterBy = FilterFunction.AVG;
        private String filterLimit = StringUtils.EMPTY;
        private GraphTransform transform = GraphTransform.NONE;
        private String movingWindow = StringUtils.EMPTY;
        private String movingPercentile = StringUtils.EMPTY;
        private DownsamplingMode downsampling = DownsamplingMode.AUTO;
        private DownsamplingAggregationType downsamplingAggr = DownsamplingAggregationType.DEFAULT_VALUE;
        private DownsamplingFillType downsamplingFill = DownsamplingFillType.NULL;
        private boolean ignoreMinStepMillis = false;
        private String grid = StringUtils.EMPTY;
        private Integer maxPoints = 0;
        private boolean hideNoData = false;
        private Double threshold = null;
        private String generatedId = StringUtils.EMPTY;

        private Builder() {
        }

        private Builder(@Nonnull Graph graph) {
            super(graph);
            this.id = graph.getId();
            this.projectId = graph.getProjectId();
            this.folderId = graph.getFolderId();
            this.name = graph.getName();
            this.description = graph.getDescription();
            this.parameters = graph.getParameters();
            this.elements = graph.getElements();
            this.graphMode = graph.getGraphMode();
            this.secondaryGraphMode = graph.getSecondaryGraphMode();
            this.min = graph.getMin();
            this.max = graph.getMax();
            this.normalize = graph.isNormalize();
            this.colorScheme = graph.getColorScheme();
            this.dropNans = graph.isDropNans();
            this.stack = graph.getStack();
            this.aggr = graph.getAggr();
            this.interpolate = graph.getInterpolate();
            this.scale = graph.getScale();
            this.numberFormat = graph.getNumberFormat();
            this.green = graph.getGreen();
            this.yellow = graph.getYellow();
            this.red = graph.getRed();
            this.violet = graph.getViolet();
            this.greenValue = graph.getGreenValue();
            this.yellowValue = graph.getYellowValue();
            this.redValue = graph.getRedValue();
            this.violetValue = graph.getVioletValue();
            this.sortByLabel = graph.getSortByLabel();
            this.asc = graph.isAsc();
            this.limit = graph.getLimit();
            this.overLinesTransform = graph.getOverLinesTransform();
            this.percentiles = graph.getPercentiles();
            this.bucketLabel = graph.getBucketLabel();
            this.ignoreInf = graph.isIgnoreInf();
            this.filter = graph.getFilter();
            this.filterBy = graph.getFilterBy();
            this.filterLimit = graph.getFilterLimit();
            this.transform = graph.getTransform();
            this.movingWindow = graph.getMovingWindow();
            this.movingPercentile = graph.getMovingPercentile();
            this.downsampling = graph.getDownsampling();
            this.downsamplingAggr = graph.getDownsamplingAggr();
            this.downsamplingFill = graph.getDownsamplingFill();
            this.ignoreMinStepMillis = graph.isIgnoreMinStepMillis();
            this.grid = graph.getGrid();
            this.maxPoints = graph.getMaxPoints();
            this.hideNoData = graph.isHideNoData();
            this.threshold = graph.getThreshold();
            this.generatedId = graph.getGeneratedId();
        }

        @Nonnull
        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        @Nonnull
        public Builder setProjectId(String projectId) {
            this.projectId = projectId;
            return this;
        }

        @Nonnull
        public Builder setFolderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        @Nonnull
        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        @Nonnull
        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        @Nonnull
        public Builder setParameters(Selector[] parameters) {
            this.parameters = parameters;
            return this;
        }

        @Nonnull
        public Builder setElements(GraphElement[] elements) {
            this.elements = elements;
            return this;
        }

        @Nonnull
        public Builder setGraphMode(GraphMode graphMode) {
            this.graphMode = graphMode;
            return this;
        }

        @Nonnull
        public Builder setSecondaryGraphMode(SecondaryGraphMode secondaryGraphMode) {
            this.secondaryGraphMode = secondaryGraphMode;
            return this;
        }

        @Nonnull
        public Builder setMin(String min) {
            this.min = min;
            return this;
        }

        @Nonnull
        public Builder setMax(String max) {
            this.max = max;
            return this;
        }

        @Nonnull
        public Builder setNormalize(boolean normalize) {
            this.normalize = normalize;
            return this;
        }

        @Nonnull
        public Builder setColorScheme(ColorSchemeType colorScheme) {
            this.colorScheme = colorScheme;
            return this;
        }

        @Nonnull
        public Builder setDropNans(Boolean dropNans) {
            this.dropNans = dropNans;
            return this;
        }

        @Nonnull
        public Builder setStack(Boolean stack) {
            this.stack = stack;
            return this;
        }

        @Nonnull
        public Builder setAggr(AggregationType aggr) {
            this.aggr = aggr;
            return this;
        }

        @Nonnull
        public Builder setInterpolate(InterpolationType interpolate) {
            this.interpolate = interpolate;
            return this;
        }

        @Nonnull
        public Builder setOverLinesTransform(OverLinesTransform overLinesTransform) {
            this.overLinesTransform = overLinesTransform;
            return this;
        }

        @Nonnull
        public Builder setScale(ScaleType scale) {
            this.scale = scale;
            return this;
        }

        @Nonnull
        public Builder setNumberFormat(String numberFormat) {
            this.numberFormat = numberFormat;
            return this;
        }

        @Nonnull
        public Builder setGreen(String green) {
            this.green = green;
            return this;
        }

        @Nonnull
        public Builder setYellow(String yellow) {
            this.yellow = yellow;
            return this;
        }

        @Nonnull
        public Builder setRed(String red) {
            this.red = red;
            return this;
        }

        @Nonnull
        public Builder setViolet(String violet) {
            this.violet = violet;
            return this;
        }

        @Nonnull
        public Builder setGreenValue(String greenValue) {
            this.greenValue = greenValue;
            return this;
        }

        @Nonnull
        public Builder setYellowValue(String yellowValue) {
            this.yellowValue = yellowValue;
            return this;
        }

        @Nonnull
        public Builder setRedValue(String redValue) {
            this.redValue = redValue;
            return this;
        }

        @Nonnull
        public Builder setVioletValue(String violetValue) {
            this.violetValue = violetValue;
            return this;
        }

        @Nonnull
        public Builder setSortByLabel(String sortByLabel) {
            this.sortByLabel = sortByLabel;
            return this;
        }

        @Nonnull
        public Builder setAsc(boolean asc) {
            this.asc = asc;
            return this;
        }

        @Nonnull
        public Builder setLimit(String limit) {
            this.limit = limit;
            return this;
        }

        @Nonnull
        public Builder setPercentiles(String percentiles) {
            this.percentiles = percentiles;
            return this;
        }

        @Nonnull
        public Builder setBucketLabel(String bucketLabel) {
            this.bucketLabel = bucketLabel;
            return this;
        }

        @Nonnull
        public Builder setIgnoreInf(boolean ignoreInf) {
            this.ignoreInf = ignoreInf;
            return this;
        }

        @Nonnull
        public Builder setFilter(FilterOrder filter) {
            this.filter = filter;
            return this;
        }

        @Nonnull
        public Builder setFilterBy(FilterFunction filterBy) {
            this.filterBy = filterBy;
            return this;
        }

        @Nonnull
        public Builder setFilterLimit(String filterLimit) {
            this.filterLimit = filterLimit;
            return this;
        }

        @Nonnull
        public Builder setTransform(GraphTransform transform) {
            this.transform = transform;
            return this;
        }

        @Nonnull
        public Builder setMovingWindow(String movingWindow) {
            this.movingWindow = movingWindow;
            return this;
        }

        @Nonnull
        public Builder setMovingPercentile(String movingPercentile) {
            this.movingPercentile = movingPercentile;
            return this;
        }

        @Nonnull
        public Builder setDownsampling(DownsamplingMode downsampling) {
            this.downsampling = downsampling;
            return this;
        }

        @Nonnull
        public Builder setDownsamplingAggr(DownsamplingAggregationType downsamplingAggr) {
            this.downsamplingAggr = downsamplingAggr;
            return this;
        }

        @Nonnull
        public Builder setDownsamplingFill(DownsamplingFillType downsamplingFill) {
            this.downsamplingFill = downsamplingFill;
            return this;
        }

        @Nonnull
        public Builder setIgnoreMinStepMillis(boolean ignoreMinStepMillis) {
            this.ignoreMinStepMillis = ignoreMinStepMillis;
            return this;
        }

        @Nonnull
        public Builder setGrid(String grid) {
            this.grid = grid;
            return this;
        }

        @Nonnull
        public Builder setMaxPoints(Integer maxPoints) {
            this.maxPoints = maxPoints;
            return this;
        }

        @Nonnull
        public Builder setHideNoData(boolean hideNoData) {
            this.hideNoData = hideNoData;
            return this;
        }

        @Nonnull
        public Builder setThreshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        @Nonnull
        public Builder setGeneratedId(String generatedId) {
            this.generatedId = generatedId;
            return this;
        }

        public String getId() {
            return id;
        }

        public String getProjectId() {
            return projectId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getName() {
            return name;
        }

        public String getDescription() {
            return description;
        }

        public Selector[] getParameters() {
            return parameters;
        }

        public GraphElement[] getElements() {
            return elements;
        }

        public GraphMode getGraphMode() {
            return graphMode;
        }

        public SecondaryGraphMode getSecondaryGraphMode() {
            return secondaryGraphMode;
        }

        public String getMin() {
            return min;
        }

        public String getMax() {
            return max;
        }

        public Boolean getNormalize() {
            return normalize;
        }

        public ColorSchemeType getColorScheme() {
            return colorScheme;
        }

        public Boolean getDropNans() {
            return dropNans;
        }

        public Boolean getStack() {
            return stack;
        }

        public AggregationType getAggr() {
            return aggr;
        }

        public InterpolationType getInterpolate() {
            return interpolate;
        }

        public ScaleType getScale() {
            return scale;
        }

        public String getNumberFormat() {
            return numberFormat;
        }

        public String getGreen() {
            return green;
        }

        public String getYellow() {
            return yellow;
        }

        public String getRed() {
            return red;
        }

        public String getViolet() {
            return violet;
        }

        public String getGreenValue() {
            return greenValue;
        }

        public String getYellowValue() {
            return yellowValue;
        }

        public String getRedValue() {
            return redValue;
        }

        public String getVioletValue() {
            return violetValue;
        }

        public String getSortByLabel() {
            return sortByLabel;
        }

        public Boolean getAsc() {
            return asc;
        }

        public String getLimit() {
            return limit;
        }

        public OverLinesTransform getOverLinesTransform() {
            return overLinesTransform;
        }

        public String getPercentiles() {
            return percentiles;
        }

        public String getBucketLabel() {
            return bucketLabel;
        }

        public Boolean isIgnoreInf() {
            return ignoreInf;
        }

        public FilterOrder getFilter() {
            return filter;
        }

        public FilterFunction getFilterBy() {
            return filterBy;
        }

        public String getFilterLimit() {
            return filterLimit;
        }

        public GraphTransform getTransform() {
            return transform;
        }

        public String getMovingWindow() {
            return movingWindow;
        }

        public String getMovingPercentile() {
            return movingPercentile;
        }

        public DownsamplingMode getDownsampling() {
            return downsampling;
        }

        public DownsamplingAggregationType getDownsamplingAggr() {
            return downsamplingAggr;
        }

        public DownsamplingFillType getDownsamplingFill() {
            return downsamplingFill;
        }

        public boolean isIgnoreMinStepMillis() {
            return ignoreMinStepMillis;
        }

        public String getGrid() {
            return grid;
        }

        public Integer getMaxPoints() {
            return maxPoints;
        }

        public Boolean getHideNoData() {
            return hideNoData;
        }

        public Double getThreshold() {
            return threshold;
        }

        public String getGeneratedId() {
            return generatedId;
        }

        @Nonnull
        public Graph build() {
            return new Graph(this);
        }
    }
}
