package ru.yandex.solomon.core.kikimr;

import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.net.HostAndPort;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.kikimr.client.discovery.Discovery;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
class SolomonDiscovery implements Discovery, AutoCloseable {
    private final DiscoveryService discovery;
    private final List<String> targets;
    private final PingActorRunner actor;
    private volatile Set<HostAndPort> addresses = Set.of();
    private final AtomicReference<CompletableFuture<Void>> onChange = new AtomicReference<>(new CompletableFuture<>());

    public SolomonDiscovery(DiscoveryService discovery, List<String> targets, Executor executor, ScheduledExecutorService timer) {
        this.discovery = discovery;
        this.targets = targets;
        this.actor = PingActorRunner.newBuilder()
                .onPing(this::act)
                .executor(executor)
                .timer(timer)
                .operation("discovery_ydb_endpoints_by_solomon-discovery")
                .pingInterval(Duration.ofMinutes(1))
                .backoffMaxDelay(Duration.ofMinutes(5))
                .backoffDelay(Duration.ofSeconds(5))
                .build();
        actor.forcePing();
    }

    public Set<HostAndPort> addresses() {
        return addresses;
    }

    public CompletionStage<Void> forceUpdate() {
        return actor.forcePing();
    }

    public CompletionStage<Void> onChange() {
        return onChange.get();
    }

    private CompletableFuture<Void> act(int attempt) {
        return discovery.resolve(targets)
                .thenAccept(result -> {
                    var addresses = Set.copyOf(result);
                    if (!addresses.equals(this.addresses)) {
                        this.addresses = addresses;
                        onChange.getAndSet(new CompletableFuture<>()).complete(null);
                    }
                });
    }

    @Override
    public void close() throws Exception {
        actor.close();
    }
}
