package ru.yandex.solomon.core.urlStatus;

import java.io.IOException;
import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.UnknownHostException;
import java.util.concurrent.TimeoutException;

import com.fasterxml.jackson.core.JsonParseException;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.monlib.metrics.encode.ParseException;
import ru.yandex.monlib.metrics.encode.spack.SpackException;
import ru.yandex.solomon.proto.UrlStatusType;


/**
* @author checat
*/
public class UrlStatusTypes {

    public static UrlStatusType classifyError(Throwable t) {
        if (t == null) {
            return UrlStatusType.OK;
        }
        if (t instanceof UrlStatusTypeAwareException) {
            return ((UrlStatusTypeAwareException) t).urlStatusType();
        }
        if (t instanceof ConnectException || isConnectionResetByPeer(t)) {
            return UrlStatusType.CONNECT_FAILURE;
        }
        if (t instanceof TimeoutException) {
            return UrlStatusType.TIMEOUT;
        }
        if (t instanceof JsonParseException || t.getCause() instanceof JsonParseException) {
            return UrlStatusType.JSON_ERROR;
        }
        if (t instanceof SpackException) {
            return UrlStatusType.SPACK_ERROR;
        }
        if (t instanceof UnknownHostException || t instanceof NoRouteToHostException) {
            return UrlStatusType.UNKNOWN_HOST;
        }
        if (t instanceof ParseException) {
            return UrlStatusType.PARSE_ERROR;
        }
        if (t instanceof StatusRuntimeException) {
            var status = ((StatusRuntimeException) t).getStatus();
            if (status.getCode() == Status.Code.DEADLINE_EXCEEDED) {
                return UrlStatusType.TIMEOUT;
            }
        }
        Throwable cause = t.getCause();
        if (cause != null) {
            return classifyError(cause);
        } else {
            return UrlStatusType.UNKNOWN_ERROR;
        }
    }

    private static boolean isConnectionResetByPeer(Throwable t) {
        // unfortunately not all errors mapped by Netty's native epoll transport to an appropriate exception class,
        // see io.netty.channel.unix.Errors.NativeIoException
        return t instanceof IOException && t.getMessage() != null && t.getMessage().contains("Connection reset by peer");
    }
}
