package ru.yandex.solomon.core.validators;

import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.solomon.core.exceptions.BadRequestException;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class IdValidator {

    public static final int MAX_ID_LENGTH = 55;
    public static final int MAX_PROJECT_ID_LENGTH = 80;
    public static final int MAX_SHARD_ID_LENGTH = 180;
    public static final int MAX_CLUSTER_ID_LENGTH = 101;
    public static final int MAX_SERVICE_ID_LENGTH = 101;

    private static final Pattern ID_PATTERN = Pattern.compile("[a-zA-Z0-9_.-]+");

    public static void ensureValid(@Nullable String id, String whichId) {
        ensureValid(id, whichId, MAX_ID_LENGTH);
    }

    public static void ensureValid(@Nullable String id, String whichId, int maxLength) {
        String message = validate(id, whichId, maxLength);
        if (message != null) {
            throw new BadRequestException(message);
        }
    }

    public static void ensureProjectIdValid(@Nullable String id) {
        ensureValid(id, "project", MAX_PROJECT_ID_LENGTH);
    }

    public static void ensureClusterIdValid(@Nullable String id, String whichId) {
        String message = validate(id, whichId, MAX_CLUSTER_ID_LENGTH);
        if (message != null) {
            throw new BadRequestException(message);
        }
    }

    public static void ensureServiceIdValid(@Nullable String id, String whichId) {
        String message = validate(id, whichId, MAX_SERVICE_ID_LENGTH);
        if (message != null) {
            throw new BadRequestException(message);
        }
    }

    public static void ensureShardIdValid(@Nullable String id, String whichId) {
        String message = validate(id, whichId, MAX_SHARD_ID_LENGTH);
        if (message != null) {
            throw new BadRequestException(message);
        }
    }

    private static String validate(@Nullable String id, String whichId, int maxLength) {
        if (StringUtils.isBlank(id)) {
            return whichId + " id cannot be blank";
        }
        if ("new".equals(id)) {
            return  whichId + " id cannot be \"new\"";
        }
        if ("protoNew".equals(id)) {
            return whichId + " id cannot be \"protoNew\"";
        }
        if (id.length() > maxLength) {
            return String.format("%s id must have length fewer than %s characters", whichId, maxLength);
        }
        if (!ID_PATTERN.matcher(id).matches()) {
            return whichId + " id must contain latin letters, digits, dots, hyphens or underscores";
        }
        String message = StrictValidator.validateValue(id);
        if (message != null) {
            return "invalid " + whichId + " id: " + message;
        }

        return null;
    }
}
