package ru.yandex.solomon.core.conf;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;

import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonConfWithContextTest {
    @Test
    public void notUniqueClusterId() {
        var alice = Shard.newBuilder()
            .setProjectId("alice")
            .setId("alice_prod_sys")
            .setNumId(1)
            .setClusterId("prod")
            .setClusterName("prod")
            .setServiceId("alice_sys")
            .setServiceName("alice_sys")
            .build();

        var bob =  Shard.newBuilder()
            .setProjectId("bob")
            .setId("bob_prod_sys")
            .setNumId(2)
            .setClusterId("prod")
            .setClusterName("production")
            .setServiceId("bob_sys")
            .setServiceName("bob_sys")
            .build();

        var conf = create(alice, bob);
        {
            var shardConf = conf.getShardByNumId(alice.getNumId());
            assertTrue(shardConf.isCorrect());
            assertEquals(alice, shardConf.getRaw());
            assertEquals("prod", shardConf.getConfOrThrow().getCluster().getName());
            assertEquals("prod", shardConf.getConfOrThrow().getCluster().getId());
        }
        {
            var shardConf = conf.getShardByNumId(bob.getNumId());
            assertTrue(shardConf.isCorrect());
            assertEquals(bob, shardConf.getRaw());
            assertEquals("production", shardConf.getConfOrThrow().getCluster().getName());
            assertEquals("prod", shardConf.getConfOrThrow().getCluster().getId());
        }
    }

    @Test
    public void notUniqueServiceId() {
        var alice = Shard.newBuilder()
            .setProjectId("alice")
            .setId("alice_prod_sys")
            .setNumId(1)
            .setClusterId("alice_prod")
            .setClusterName("prod")
            .setServiceId("sys")
            .setServiceName("sys")
            .build();

        var bob =  Shard.newBuilder()
            .setProjectId("bob")
            .setId("bob_prod_sys")
            .setNumId(2)
            .setClusterId("bob_prod")
            .setClusterName("prod")
            .setServiceId("sys")
            .setServiceName("system")
            .build();

        var conf = create(alice, bob);
        {
            var shardConf = conf.getShardByNumId(alice.getNumId());
            assertTrue(shardConf.isCorrect());
            assertEquals(alice, shardConf.getRaw());
            assertEquals("sys", shardConf.getConfOrThrow().getService().getName());
            assertEquals("sys", shardConf.getConfOrThrow().getService().getId());
        }
        {
            var shardConf = conf.getShardByNumId(bob.getNumId());
            assertTrue(shardConf.isCorrect());
            assertEquals(bob, shardConf.getRaw());
            assertEquals("system", shardConf.getConfOrThrow().getService().getName());
            assertEquals("sys", shardConf.getConfOrThrow().getService().getId());
        }
    }

    @Test
    public void notUniqueShardId() {
        var alice = Shard.newBuilder()
                .setProjectId("alice")
                .setId("sys")
                .setNumId(1)
                .setClusterId("alice_prod")
                .setClusterName("prod")
                .setServiceId("alice_sys")
                .setServiceName("sys")
                .build();

        var bob =  Shard.newBuilder()
                .setProjectId("bob")
                .setId("sys")
                .setNumId(2)
                .setClusterId("bob_prod")
                .setClusterName("prod")
                .setServiceId("bob_sys")
                .setServiceName("system")
                .build();

        var conf = create(alice, bob);
        {
            var shardConf = conf.getShardByNumId(alice.getNumId());
            assertTrue(shardConf.isCorrect());
            assertEquals(alice, shardConf.getRaw());
        }
        {
            var shardConf = conf.getShardByNumId(bob.getNumId());
            assertTrue(shardConf.isCorrect());
            assertEquals(bob, shardConf.getRaw());
        }
    }

    @Test
    public void notExistsServiceProvider() {
        var project = project("alice").build();
        var cluster = cluster("alice", "alice_prod", "prod").build();
        var service = service("alice", "alice_sys", "sys").setServiceProvider("not_exist").build();
        var shard = Shard.newBuilder()
                .setProjectId("alice")
                .setId("sys")
                .setNumId(1)
                .setClusterId("alice_prod")
                .setClusterName("prod")
                .setServiceId("alice_sys")
                .setServiceName("sys")
                .build();

        var rawConf = new SolomonRawConf(List.of(), List.of(project), List.of(cluster), List.of(service), List.of(shard));
        var context = SolomonConfWithContext.create(rawConf);

        var result = context.getShardByNumId(1);
        assertNotNull(result.getThrowable());
        assertFalse(result.isCorrect());
    }

    @Test
    public void serviceProvider() {
        var project = project("alice").build();
        var cluster = cluster("alice", "alice_prod", "prod").build();
        var serviceProvider = serviceProvider("sys").build();
        var service = service("alice", "alice_sys", "sys").setServiceProvider("sys").build();
        var shard = Shard.newBuilder()
                .setProjectId("alice")
                .setId("sys")
                .setNumId(1)
                .setClusterId("alice_prod")
                .setClusterName("prod")
                .setServiceId("alice_sys")
                .setServiceName("sys")
                .build();

        var rawConf = new SolomonRawConf(List.of(serviceProvider), List.of(project), List.of(cluster), List.of(service), List.of(shard));
        var context = SolomonConfWithContext.create(rawConf);

        var result = context.getShardByNumId(1).getConfOrThrow();
        assertEquals(serviceProvider, result.getService().getServiceProvider());
    }

    private SolomonConfWithContext create(Shard... shards) {
        List<Project> projects = Arrays.stream(shards)
            .map(s -> project(s.getProjectId()).build())
            .distinct()
            .collect(Collectors.toList());

        List<Service> services = Arrays.stream(shards)
            .map(s -> service(s.getProjectId(), s.getServiceId(), s.getServiceName()).build())
            .distinct()
            .collect(Collectors.toList());

        List<Cluster> clusters = Arrays.stream(shards)
            .map(s -> cluster(s.getProjectId(), s.getClusterId(), s.getClusterName()).build())
            .distinct()
            .collect(Collectors.toList());

        var rawConf = new SolomonRawConf(List.of(), projects, clusters, services, Arrays.asList(shards));
        return SolomonConfWithContext.create(rawConf);
    }

    private Project.Builder project(String id) {
        return Project.newBuilder()
                .setId(id)
                .setName(id)
                .setOwner("gordiychuk");
    }

    private Service.Builder service(String projectId, String id, String name) {
        return Service.newBuilder()
                .setProjectId(projectId)
                .setId(id)
                .setName(name);
    }

    private ServiceProvider.Builder serviceProvider(String id) {
        return ServiceProvider.newBuilder().setId(id);
    }

    private Cluster.Builder cluster(String projectId, String id, String name) {
        return Cluster.newBuilder()
                .setProjectId(projectId)
                .setId(id)
                .setName(name);
    }
}
