package ru.yandex.solomon.core.conf.aggr;

import java.util.Set;

import javax.annotation.Nullable;

import org.junit.Test;

import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.labels.shard.ShardKey;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class AggrRuleConfTest {

    @Test
    public void hostCluster() {
        var conf = parse("host=*", "host=cluster");
        assertEquals(Set.of("host"), conf.getConditionLabelNames());
        assertEquals(Set.of(), conf.getSubstitutionLabelNames());
        assertFalse(conf.isHostRequired());
    }

    @Test
    public void hostDc() {
        var conf = parse("host=*", "host={{DC}}");
        assertEquals(Set.of("host"), conf.getConditionLabelNames());
        assertEquals(Set.of("DC"), conf.getSubstitutionLabelNames());
        assertFalse(conf.isHostRequired());
    }

    @Test
    public void totalOnHost() {
        var conf = parse("account=*", "account=total");
        assertEquals(Set.of("account"), conf.getConditionLabelNames());
        assertEquals(Set.of(), conf.getSubstitutionLabelNames());
        assertTrue(conf.isHostRequired());
    }

    @Test
    public void dcTohost() {
        var conf = parse("DC=*", "host={{DC}}");
        assertEquals(Set.of("DC"), conf.getConditionLabelNames());
        assertEquals(Set.of("DC"), conf.getSubstitutionLabelNames());
        assertFalse(conf.isHostRequired());
    }

    @Test
    public void rmHost() {
        var conf = parse("host=*", "host=-");
        assertEquals(Set.of("host"), conf.getConditionLabelNames());
        assertEquals(Set.of(), conf.getSubstitutionLabelNames());
        assertFalse(conf.isHostRequired());
    }

    @Test
    public void rmWorker() {
        var conf = parse("worker=*", "worker=-");
        assertEquals(Set.of("worker"), conf.getConditionLabelNames());
        assertEquals(Set.of(), conf.getSubstitutionLabelNames());
        assertTrue(conf.isHostRequired());
    }

    @Test
    public void partHostMatch() {
        var conf = parse("host=reactor-back-processing-*", "host=reactor-back-processing");
        assertEquals(Set.of("host"), conf.getConditionLabelNames());
        assertEquals(Set.of(), conf.getSubstitutionLabelNames());
        assertTrue(conf.isHostRequired());
    }

    @Test
    public void extendHost() {
        var conf = parse("host=*", "host={{host}}-{{DC}}");
        assertEquals(Set.of("host"), conf.getConditionLabelNames());
        assertEquals(Set.of("host", "DC"), conf.getSubstitutionLabelNames());
        assertTrue(conf.isHostRequired());
    }

    @Test
    public void hostAndClientIdClusterTotal() {
        var conf = parse("host=*, clientId=*", "host=cluster, clientId=total");
        assertEquals(Set.of("host", "clientId"), conf.getConditionLabelNames());
        assertEquals(Set.of(), conf.getSubstitutionLabelNames());
        assertFalse(conf.isHostRequired());
    }

    @Test
    public void defaultAggregationSum() {
        var conf = parse("host=*", "host=cluster", null);
        assertEquals(MetricAggregation.SUM, conf.getAggregation());
    }

    @Test
    public void aggregationToLast() {
        var conf = parse("host=*", "host=cluster", MetricAggregation.LAST);
        assertEquals(MetricAggregation.LAST, conf.getAggregation());
    }

    private AggrRuleConf parse(String condition, String target) {
        return parse(condition, target, null);
    }

    private AggrRuleConf parse(String condition, String target, @Nullable MetricAggregation aggregation) {
        var shard = new ShardKey("project", "cluster", "service");
        var rule = AggrRule.of(condition, target, aggregation);
        return new AggrRuleConf(rule, shard);
    }
}
