package ru.yandex.solomon.core.conf.aggr;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author Stepan Koltsov
 */
public class LabelValueExprTest {

    private void testEvalImpl(@Nonnull String expr, @Nullable String expected, Map<String, String> env) {
        Assert.assertEquals(expected, LabelValueExpr.parse(expr).eval(Labels.of(env), Labels.of()));
    }

    private void testEvalImpl(String expr, String expected) {
        testEvalImpl(expr, expected, Collections.emptyMap());
    }

    private void testEvalImpl(String expr, String expected, String k1, String v1) {
        testEvalImpl(expr, expected, Collections.singletonMap(k1, v1));
    }

    private void testEvalImpl(String expr, String expected, String k1, String v1, String k2, String v2) {
        HashMap<String, String> env = new HashMap<>();
        env.put(k1, v1);
        env.put(k2, v2);
        testEvalImpl(expr, expected, env);
    }


    @Test
    public void eval() {
        testEvalImpl("aaa", "aaa");
        testEvalImpl("aaa", "aaa", "bbb", "ccc");

        testEvalImpl("{{aaa}}", null);
        testEvalImpl("{{aaa}}", null, "bbb", "ccc");
        testEvalImpl("{{aaa}}", "ccc", "aaa", "ccc");

        testEvalImpl("aaa{{bbb}}ccc", null);
        testEvalImpl("aaa{{bbb}}ccc", null, "ccc", "ddd");
        testEvalImpl("aaa{{bbb}}ccc", "aaadddccc", "bbb", "ddd");

        testEvalImpl("aaa{{bbb}}", null);
        testEvalImpl("aaa{{bbb}}", null, "ccc", "ddd");
        testEvalImpl("aaa{{bbb}}", "aaaddd", "bbb", "ddd");

        testEvalImpl("{{aaa}}bbb", null);
        testEvalImpl("{{aaa}}bbb", null, "ccc", "ddd");
        testEvalImpl("{{aaa}}bbb", "dddbbb", "aaa", "ddd");

        testEvalImpl("{{aaa}}bbb{{ccc}}", null);
        testEvalImpl("{{aaa}}bbb{{ccc}}", null, "aaa", "eee");
        testEvalImpl("{{aaa}}bbb{{ccc}}", "eeebbbfff", "aaa", "eee", "ccc", "fff");
    }

    private void testBindImpl(String expr, String expected, Map<String, String> partial) {
        Assert.assertEquals(expected, LabelValueExpr.parse(expr).bind(partial).toString());
    }

    private void testBindImpl(String expr, String expected) {
        testBindImpl(expr, expected, Collections.emptyMap());
    }

    private void testBindImpl(String expr, String expected, String k1, String v1) {
        testBindImpl(expr, expected, Collections.singletonMap(k1, v1));
    }

    private void testBindImpl(String expr, String expected, String k1, String v1, String k2, String v2) {
        HashMap<String, String> env = new HashMap<>();
        env.put(k1, v1);
        env.put(k2, v2);
        testBindImpl(expr, expected, env);
    }

    @Test
    public void bind() {
        testBindImpl("aaa", "aaa");
        testBindImpl("{{cluster}}-{{service}}", "{{cluster}}-{{service}}");
        testBindImpl("{{cluster}}-{{service}}", "main-{{service}}", "cluster", "main");
        testBindImpl("xx-{{cluster}}-{{service}}", "xx-main-hen", "cluster", "main", "service", "hen");
    }

    @Test
    public void isValid() {
        assertTrue(LabelValueExpr.isValid("-"));
        assertTrue(LabelValueExpr.isValid("aaa"));
        assertTrue(LabelValueExpr.isValid("{{bbb}}"));
        assertTrue(LabelValueExpr.isValid("aaa{{bbb}}"));
        assertTrue(LabelValueExpr.isValid("{{bbb}}ccc"));
        assertTrue(LabelValueExpr.isValid("aaa{{bbb}}ccc"));
        assertTrue(LabelValueExpr.isValid("aaa{{bbb}}ccc{{ddd}}"));
        assertTrue(LabelValueExpr.isValid("a*"));
        assertTrue(LabelValueExpr.isValid("a?"));
        assertTrue(LabelValueExpr.isValid("a?b*c"));
        assertTrue(LabelValueExpr.isValid("a*b?c"));
        assertTrue(LabelValueExpr.isValid("!OK"));
        assertTrue(LabelValueExpr.isValid("OK|ERROR"));
        assertTrue(LabelValueExpr.isValid("-|ERROR"));
        assertTrue(LabelValueExpr.isValid("OK|-"));
        assertTrue(LabelValueExpr.isValid("abc-{{def}}-ghi"));

        Assert.assertFalse(LabelValueExpr.isValid("{{"));
        Assert.assertFalse(LabelValueExpr.isValid("{{-}}"));
        Assert.assertFalse(LabelValueExpr.isValid("{{aaa"));
        Assert.assertFalse(LabelValueExpr.isValid("|"));
        Assert.assertFalse(LabelValueExpr.isValid("x|"));
        Assert.assertFalse(LabelValueExpr.isValid("|y"));
        Assert.assertFalse(LabelValueExpr.isValid("x||y"));
    }

    @Test
    public void hostRequired() {
        assertTrue(LabelValueExpr.parse("{{host}}").isHostRequired());
        assertTrue(LabelValueExpr.parse("{{host}}-").isHostRequired());
        assertTrue(LabelValueExpr.parse("-{{host}}").isHostRequired());
        assertTrue(LabelValueExpr.parse("-{{host}}-").isHostRequired());
        assertTrue(LabelValueExpr.parse("{{DC}}-{{host}}").isHostRequired());
        assertTrue(LabelValueExpr.parse("{{host}}-{{DC}}").isHostRequired());
    }

    @Test
    public void hostNotRequired() {
        assertFalse(LabelValueExpr.parse("{{DC}}").isHostRequired());
        assertFalse(LabelValueExpr.parse("host").isHostRequired());
        assertFalse(LabelValueExpr.parse("cluster").isHostRequired());
        assertFalse(LabelValueExpr.parse("total").isHostRequired());
        assertFalse(LabelValueExpr.parse("{{DC}}-{{worker}}").isHostRequired());
        assertFalse(LabelValueExpr.parse("{{worker}}").isHostRequired());
    }
}
