package ru.yandex.solomon.core.conf.flags;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.conf.ConfigNotInitialized;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsConfig;
import ru.yandex.solomon.util.file.SimpleFileStorage;

import static ru.yandex.solomon.flags.FeatureFlagsMatchers.assertHasFlag;
import static ru.yandex.solomon.flags.FeatureFlagsMatchers.assertHasNotFlag;
import static ru.yandex.solomon.flags.FeatureFlagsMatchers.assertIsDefined;
import static ru.yandex.solomon.flags.FeatureFlagsMatchers.assertNotEmpty;

/**
 * @author Vladimir Gordiychuk
 */
public class FeatureFlagsHolderImplTest {
    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();
    Path storage;
    FeatureFlagsHolderImpl holder;

    @Before
    public void setUp() throws Exception {
        String name = testName.getMethodName();
        storage = tmp.newFolder(name).toPath();
        holder = createFeatureFlagsHolder();
    }

    @Test(expected = ConfigNotInitialized.class)
    public void projectFlagsNotInitialized() {
        holder.flags("projectId");
    }

    @Test(expected = ConfigNotInitialized.class)
    public void shardFlagsNotInitialized() {
        holder.flags(123);
    }

    @Test
    public void projectFlagNotDefined() {
        var alice = shard(42, "alice");
        holder.onConfigurationLoad(solomonConf(alice));

        holder.onConfigurationLoad(new FeatureFlagsConfig());
        var flags = holder.flags(alice.getProjectId());
        assertHasNotFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void shardFlagNotDefined() {
        var alice = shard(42, "alice");
        holder.onConfigurationLoad(solomonConf(alice));

        holder.onConfigurationLoad(new FeatureFlagsConfig());
        var flags = holder.flags(alice.getNumId());
        assertHasNotFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void projectFlagUseDefault() {
        var bob = shard(43, "bob");
        holder.onConfigurationLoad(solomonConf(bob));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags(bob.getProjectId());
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void shardFlagUseDefault() {
        var bob = shard(43, "bob");
        holder.onConfigurationLoad(solomonConf(bob));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags(bob.getNumId());
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void shardFlagUseServiceProvider() {
        var serviceProvider = serviceProvider("compute");
        var service  = service("bob")
                .toBuilder()
                .setName(serviceProvider.getId())
                .setServiceProvider(serviceProvider.getId())
                .build();
        var bob = shard(44, "bob").toBuilder()
                .setServiceName("compute")
                .setServiceId(service.getId())
                .setServiceName(service.getName())
                .build();
        holder.onConfigurationLoad(solomonConf(serviceProvider, service, bob));

        var config = new FeatureFlagsConfig();
        config.addServiceProviderFlag(serviceProvider.getId(), FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags(bob.getNumId());
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void defaultAndProjectFlagsMerged() {
        var alice = shard(42, "alice");
        holder.onConfigurationLoad(solomonConf(alice));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, true);
        config.addProjectFlag(alice.getProjectId(), FeatureFlag.EXAMPLE, true);
        holder.onConfigurationLoad(config);

        {
            var flags = holder.flags(alice.getNumId());

            assertIsDefined(flags, FeatureFlag.TEST);
            assertIsDefined(flags, FeatureFlag.EXAMPLE);
            assertHasFlag(flags, FeatureFlag.TEST);
            assertHasFlag(flags, FeatureFlag.EXAMPLE);
        }
        {
            var flags = holder.flags(alice.getProjectId());

            assertIsDefined(flags, FeatureFlag.TEST);
            assertIsDefined(flags, FeatureFlag.EXAMPLE);
            assertHasFlag(flags, FeatureFlag.TEST);
            assertHasFlag(flags, FeatureFlag.EXAMPLE);
        }
    }

    @Test
    public void projectFlag() {
        var alice = shard(42, "alice");
        holder.onConfigurationLoad(solomonConf(alice));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, false);
        config.addProjectFlag(alice.getProjectId(), FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags(alice.getProjectId());
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void shardFlag() {
        var alice = shard(42, "alice");
        holder.onConfigurationLoad(solomonConf(alice));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, false);
        config.getProjectFlags(alice.getProjectId()).addShardFlag(alice.getId(), FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags(alice.getNumId());
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void serviceProviderFlag() {
        var serviceProvider = serviceProvider("compute");
        var service  = service("alice")
                .toBuilder()
                .setName(serviceProvider.getId())
                .setServiceProvider(serviceProvider.getId())
                .build();
        var alice = shard(42, "alice").toBuilder()
                .setServiceName("compute")
                .setServiceId(service.getId())
                .setServiceName(service.getName())
                .build();
        holder.onConfigurationLoad(solomonConf(serviceProvider, service, alice));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, false);
        config.addServiceProviderFlag(serviceProvider.getId(), FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags(alice.getNumId());
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void configUpdate() {
        var alice = shard(42, "alice");
        var bob = shard(43, "bob");

        {
            // empty config
            holder.onConfigurationLoad(solomonConf());
            holder.onConfigurationLoad(new FeatureFlagsConfig());

            assertHasNotFlag(holder.flags("test"), FeatureFlag.TEST);
        }
        {
            // one shard without flag
            holder.onConfigurationLoad(solomonConf(alice));
            assertHasNotFlag(holder.flags(alice.getNumId()), FeatureFlag.TEST);
        }
        {
            // default flag for all projects
            var config = new FeatureFlagsConfig();
            config.addDefaultFlag(FeatureFlag.TEST, true);
            holder.onConfigurationLoad(config);
            assertHasFlag(holder.flags(alice.getNumId()), FeatureFlag.TEST);
            assertHasFlag(holder.flags(alice.getProjectId()), FeatureFlag.TEST);
        }
        {
            // new shard added to config
            holder.onConfigurationLoad(solomonConf(alice, bob));
            assertHasFlag(holder.flags(bob.getNumId()), FeatureFlag.TEST);
            assertHasFlag(holder.flags(bob.getProjectId()), FeatureFlag.TEST);
        }
        {
            // override flag for bob
            var config = new FeatureFlagsConfig();
            config.addDefaultFlag(FeatureFlag.TEST, true);
            config.addProjectFlag(bob.getProjectId(), FeatureFlag.TEST, false);
            holder.onConfigurationLoad(config);

            assertHasNotFlag(holder.flags(bob.getNumId()), FeatureFlag.TEST);
            assertHasNotFlag(holder.flags(bob.getProjectId()), FeatureFlag.TEST);
            // alice still has it
            assertHasFlag(holder.flags(alice.getNumId()), FeatureFlag.TEST);
            assertHasFlag(holder.flags(alice.getProjectId()), FeatureFlag.TEST);
        }
        {
            // delete override flag
            var config = new FeatureFlagsConfig();
            config.addDefaultFlag(FeatureFlag.TEST, true);
            holder.onConfigurationLoad(config);

            assertHasFlag(holder.flags(bob.getNumId()), FeatureFlag.TEST);
            assertHasFlag(holder.flags(bob.getProjectId()), FeatureFlag.TEST);
            assertHasFlag(holder.flags(alice.getNumId()), FeatureFlag.TEST);
            assertHasFlag(holder.flags(alice.getProjectId()), FeatureFlag.TEST);
        }
    }

    @Test
    public void concurrentConfigUpdate() {
        var shards = IntStream.range(1, 10000)
                .mapToObj(i -> shard(i, "shardId-" + i))
                .toArray(Shard[]::new);
        var config = solomonConf(shards);
        var flagsConfig = new FeatureFlagsConfig();
        flagsConfig.addProjectFlag("projectId", FeatureFlag.TEST, true);
        for (var shard : shards) {
            var project = flagsConfig.getProjectFlags(shard.getProjectId());
            project.addShardFlag(shard.getId(), FeatureFlag.TEST, ThreadLocalRandom.current().nextBoolean());
        }

        var sync = new CyclicBarrier(2);
        var one = CompletableFuture.runAsync(() -> {
            try {
                sync.await();
                holder.onConfigurationLoad(config);
            } catch (Throwable e) {
                throw new RuntimeException(e);
            }
        });
        var two = CompletableFuture.runAsync(() -> {
            try {
                sync.await();
                holder.onConfigurationLoad(flagsConfig);
            } catch (Throwable e) {
                throw new RuntimeException(e);
            }
        });

        one.join();
        two.join();
        assertHasFlag(holder.flags("projectId"), FeatureFlag.TEST);
    }

    @Test
    public void persistFeatureFlagsState() {
        var alice = shard(42, "alice");
        holder.onConfigurationLoad(solomonConf(alice));

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, false);
        config.addProjectFlag(alice.getProjectId(), FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        {
            var flags = holder.flags(alice.getProjectId());
            assertHasFlag(flags, FeatureFlag.TEST);
        }

        // reload it
        holder = createFeatureFlagsHolder();
        {
            var flags = holder.flags(alice.getProjectId());
            assertHasFlag(flags, FeatureFlag.TEST);
        }

        holder.onConfigurationLoad(solomonConf(alice));
        {
            var flags = holder.flags(alice.getNumId());
            assertHasFlag(flags, FeatureFlag.TEST);
        }
        {
            var flags = holder.flags("test");
            assertHasNotFlag(flags, FeatureFlag.TEST);
            assertNotEmpty(flags);
        }
    }

    @Test
    public void serviceProviderFlagShardNotExist() {
        holder.onConfigurationLoad(solomonConf());

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, false);
        config.addServiceProviderFlag("compute", FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags("alice", "folder", "compute");
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    @Test
    public void projectFlagShardNotExist() {
        holder.onConfigurationLoad(solomonConf());

        var config = new FeatureFlagsConfig();
        config.addDefaultFlag(FeatureFlag.TEST, false);
        config.addProjectFlag("alice", FeatureFlag.TEST, true);
        holder.onConfigurationLoad(config);

        var flags = holder.flags("alice", "folder", "compute");
        assertHasFlag(flags, FeatureFlag.TEST);
    }

    private Shard shard(int numId, String id) {
        return Shard.newBuilder()
                .setId(id)
                .setNumId(numId)
                .setProjectId("projectId-" + id)
                .setServiceId("serviceId-" + id)
                .setServiceName("serviceName-" + id)
                .setClusterId("clusterId-"+id)
                .setClusterName("clusterName-"+id)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        0))
                .build();
    }

    private ServiceProvider serviceProvider(String name) {
        return ServiceProvider.newBuilder().setId(name).build();
    }

    private Service service(String id) {
        return Service.newBuilder()
                .setProjectId("projectId-" + id)
                .setId("serviceId-" + id)
                .setName("serviceName-" + id)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        11))
                .build();
    }

    private static SolomonConfWithContext solomonConf(Shard... shards) {
        List<Project> projects = Arrays.stream(shards)
                .map(s -> Project.newBuilder()
                        .setId(s.getProjectId())
                        .setName(s.getProjectId())
                        .setOwner("jamel")
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Service> services = Arrays.stream(shards)
                .map(s -> Service.newBuilder()
                        .setId(s.getServiceId())
                        .setName(s.getServiceName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Cluster> clusters = Arrays.stream(shards)
                .map(s -> Cluster.newBuilder()
                        .setId(s.getClusterId())
                        .setName(s.getClusterName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        SolomonRawConf rawConf = new SolomonRawConf(List.of(), projects, clusters, services, Arrays.asList(shards));
        return SolomonConfWithContext.create(rawConf);
    }

    private static SolomonConfWithContext solomonConf(ServiceProvider serviceProvider, Service service, Shard shard) {
        List<Project> projects = Stream.of(shard)
                .map(s -> Project.newBuilder()
                        .setId(s.getProjectId())
                        .setName(s.getProjectId())
                        .setOwner("jamel")
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Cluster> clusters = Stream.of(shard)
                .map(s -> Cluster.newBuilder()
                        .setId(s.getClusterId())
                        .setName(s.getClusterName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        SolomonRawConf rawConf = new SolomonRawConf(List.of(serviceProvider), projects, clusters, List.of(service), List.of(shard));
        return SolomonConfWithContext.create(rawConf);
    }

    private FeatureFlagsHolderImpl createFeatureFlagsHolder() {
        return new FeatureFlagsHolderImpl(new SimpleFileStorage(storage), new MetricRegistry());
    }
}
