package ru.yandex.solomon.core.conf.flags;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.junit.Test;

import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;

import static org.junit.Assert.assertEquals;


/**
 * @author Vladimir Gordiychuk
 */
public class FeatureFlagsOperationTest {

    private static SolomonConfWithContext solomonConf(Shard... shards) {
        List<Project> projects = Arrays.stream(shards)
                .map(s -> Project.newBuilder()
                        .setId(s.getProjectId())
                        .setName(s.getProjectId())
                        .setOwner("jamel")
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Service> services = Arrays.stream(shards)
                .map(s -> Service.newBuilder()
                        .setId(s.getServiceId())
                        .setName(s.getServiceName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Cluster> clusters = Arrays.stream(shards)
                .map(s -> Cluster.newBuilder()
                        .setId(s.getClusterId())
                        .setName(s.getClusterName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        SolomonRawConf rawConf = new SolomonRawConf(List.of(), projects, clusters, services, Arrays.asList(shards));
        return SolomonConfWithContext.create(rawConf);
    }

    private static List<String> apply(String projectId, String clusterId, String serviceId, String shardId, String serviceProvider, Shard... shards) {
        var config = solomonConf(shards);
        var result = Collections.synchronizedList(new ArrayList<String>());
        var op = new FeatureFlagsOperation(config) {
            @Override
            protected CompletableFuture<?> changeProject(String projectId) {
                return CompletableFuture.supplyAsync(() -> {
                    result.add("project:" + projectId);
                    return null;
                });
            }

            @Override
            protected CompletableFuture<?> changeServiceProvider(String serviceProvider) {
                return CompletableFuture.supplyAsync(() -> {
                    result.add("serviceProvider:" + serviceProvider);
                    return null;
                });
            }

            @Override
            protected CompletableFuture<?> changeCluster(String projectId, String clusterId) {
                return CompletableFuture.supplyAsync(() -> {
                    result.add("cluster:" + projectId + "/" + clusterId);
                    return null;
                });
            }

            @Override
            protected CompletableFuture<?> changeService(String projectId, String serviceId) {
                return CompletableFuture.supplyAsync(() -> {
                    result.add("service:" + projectId + "/" + serviceId);
                    return null;
                });
            }

            @Override
            protected CompletableFuture<?> changeShard(String projectId, String shardId) {
                return CompletableFuture.supplyAsync(() -> {
                    result.add("shard:" + projectId + "/" + shardId);
                    return null;
                });
            }
        };
        op.apply(projectId, clusterId, serviceId, shardId, serviceProvider).join();
        return result;
    }

    private static Shard shard(String id) {
        return Shard.newBuilder()
                .setId(id)
                .setNumId(id.hashCode())
                .setProjectId("projectId-" + id)
                .setServiceId("serviceId-" + id)
                .setServiceName("serviceName-" + id)
                .setClusterId("clusterId-" + id)
                .setClusterName("clusterName-" + id)
                .build();
    }

    @Test
    public void emptyAll() {
        var result = apply("", "", "", "", "");
        assertEquals(List.of("project:"), result);
    }

    @Test
    public void applyOnServiceProvider() {
        var result = apply("", "", "", "", "compute");
        assertEquals(List.of("serviceProvider:compute"), result);
    }

    @Test
    public void expectProject() {
        var result = apply("myProjectId", "", "", "", "");
        assertEquals(List.of("project:myProjectId"), result);
    }

    @Test
    public void globProject() {
        var alice = shard("alice");
        var bob = shard("bob");
        var result = apply("projectId-a*", "", "", "","", alice, bob);
        assertEquals(List.of("shard:projectId-alice/alice"), result);
    }

    @Test
    public void expectCluster() {
        var result = apply("myProjectId", "myCluster", "", "", "");
        assertEquals(List.of("cluster:myProjectId/myCluster"), result);
    }

    @Test
    public void globCluster() {
        var alice = shard("alice");
        var bob = shard("bob");
        assertEquals(List.of("shard:projectId-alice/alice"), apply("projectId-alice", "clusterId-a*", "", "", "", alice, bob));
        assertEquals(List.of(), apply("projectId-alice", "clusterId-b*", "", "", "", alice, bob));
        assertEquals(List.of("shard:projectId-alice/alice"), apply("", "clusterId-a*", "", "", "", alice, bob));
    }

    @Test
    public void exactService() {
        var result = apply("myProjectId", "", "myService", "", "");
        assertEquals(List.of("service:myProjectId/myService"), result);
    }

    @Test
    public void globService() {
        var alice = shard("alice");
        var bob = shard("bob");
        assertEquals(List.of("shard:projectId-alice/alice"), apply("projectId-alice", "", "serviceId-a*", "", "", alice, bob));
        assertEquals(List.of(), apply("projectId-alice", "", "serviceId-b*", "", "", alice, bob));
        assertEquals(List.of("shard:projectId-alice/alice"), apply("", "", "serviceId-a*", "", "", alice, bob));
    }

    @Test
    public void exactShard() {
        var result = apply("myProjectId", "", "", "myShardId", "");
        assertEquals(List.of("shard:myProjectId/myShardId"), result);
    }

    @Test
    public void globShard() {
        var alice = shard("alice");
        var bob = shard("bob");
        assertEquals(List.of("shard:projectId-alice/alice"), apply("projectId-alice", "", "", "a*", "", alice, bob));
        assertEquals(List.of(), apply("projectId-alice", "", "", "b*", "", alice, bob));
        assertEquals(List.of("shard:projectId-alice/alice"), apply("", "", "", "a*", "", alice, bob));
    }

    @Test
    public void exactAll() {
        var alice = shard("alice");
        var bob = shard("bob");
        assertEquals(List.of("shard:projectId-alice/alice"), apply(alice.getProjectId(), alice.getClusterId(), alice.getServiceId(), alice.getId(), "", alice, bob));
        assertEquals(List.of(), apply(alice.getProjectId(), alice.getClusterId(), alice.getServiceId(), bob.getId(), "", alice, bob));
        assertEquals(List.of(), apply(alice.getProjectId(), alice.getClusterId(), bob.getServiceId(), alice.getId(), "", alice, bob));
        assertEquals(List.of(), apply(alice.getProjectId(), bob.getClusterId(), alice.getServiceId(), alice.getId(), "", alice, bob));
        assertEquals(List.of(), apply(bob.getProjectId(), alice.getClusterId(), alice.getServiceId(), alice.getId(), "", alice, bob));
        assertEquals(List.of(), apply(bob.getProjectId(), alice.getClusterId(), alice.getServiceId(), alice.getId(), bob.getServiceName(), alice, bob));
    }
}
