package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.util.List;

import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.Agent;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Max Sherbakov
 */
@YaExternal
public abstract class AbstractAgentDaoTest {
    protected abstract AgentDao getAgentDao();

    protected Agent.Builder defaultAgent() {
        return Agent.newBuilder()
                .setProvider("myProvider")
                .setHostname("my.lovely.host")
                .setDataPort(3443)
                .setManagementPort(10050)
                .setVersion("1.0")
                .setDescription("I'm an agent");
    }

    private void insertOrUpdateSync(Agent agent) {
        join(getAgentDao().insertOrUpdate(agent));
    }

    private void deleteByProviderSync(String provider) {
        join(getAgentDao().deleteByProvider(provider));
    }

    private void deleteOneSync(String provider, String hostname) {
        join(getAgentDao().deleteOne(provider, hostname));
    }

    private Integer deleteObsoleteSync(Instant instant) {
        return join(getAgentDao().deleteObsolete(instant));
    }

    public List<Agent> findAllSync() {
        return join(getAgentDao().findAll());
    }

    public List<Agent> findByProviderSync(String provider) {
        PagedResult<Agent> pagedResult = join(getAgentDao().findByProvider(provider, PageOptions.ALL));
        return pagedResult.getResult();
    }

    @Test
    public void insertAndFind() {
        Agent.Builder builder = defaultAgent().setLastSeen(Instant.ofEpochSecond(1));
        Agent agent = builder.build();
        insertOrUpdateSync(agent);
        List<Agent> agents = findAllSync();
        assertEquals(agents.size(), 1);
        assertEquals(agents.get(0), agent);
    }

    @Test
    public void findByProvider() {
        Agent.Builder builder = defaultAgent().setLastSeen(Instant.ofEpochSecond(1));
        Agent agent = builder.build();
        insertOrUpdateSync(agent);
        List<Agent> agents = findByProviderSync("myProvider");
        assertEquals(1, agents.size());
        assertEquals(agent, agents.get(0));

        agents = findByProviderSync("notMyProvider");
        assertTrue(agents.isEmpty());
    }

    @Test
    public void deleteByProvider() {
        Agent.Builder builder = defaultAgent();
        Agent first = builder.setHostname("host.name").build();
        Agent second = builder
                .setProvider("differentProvider")
                .setHostname("host2.name")
                .build();

        insertOrUpdateSync(first);
        insertOrUpdateSync(second);
        deleteByProviderSync("differentProvider");

        List<Agent> agents = findAllSync();
        assertEquals(agents.size(), 1);
        assertEquals(agents.get(0), first);
    }

    @Test
    public void update() {
        Agent.Builder builder = defaultAgent();
        Agent agent = builder.build();
        insertOrUpdateSync(agent);
        builder.setLastSeen(Instant.ofEpochSecond(42));
        agent = builder.build();
        insertOrUpdateSync(agent);

        List<Agent> agents = findAllSync();
        assertEquals(1, agents.size());
        assertEquals(agent, agents.get(0));

        builder.setDescription("new description");
        agent = builder.build();
        insertOrUpdateSync(agent);
        agents = findAllSync();

        assertEquals(1, agents.size());
        assertEquals(agent, agents.get(0));

        builder
                .setManagementPort(agent.getManagementPort() + 1)
                .setDataPort(agent.getDataPort() + 1);

        agent = builder.build();
        insertOrUpdateSync(agent);

        agents = findAllSync();
        assertEquals(1, agents.size());
        assertEquals(agent, agents.get(0));
    }

    @Test
    public void deleteOne() {
        Agent.Builder builder = defaultAgent();
        Agent agent = builder.build();
        insertOrUpdateSync(agent);
        List<Agent> agents = findAllSync();
        assertEquals(1, agents.size());
        assertEquals(agent, agents.get(0));

        deleteOneSync(agent.getProvider(), agent.getHostname());
        assertTrue(findAllSync().isEmpty());
    }

    @Test
    public void deleteObsolete() {
        Agent.Builder builder = defaultAgent();
        Agent recent = builder.setLastSeen(Instant.ofEpochSecond(2)).build();
        Agent old = builder.setHostname("my.host2").setLastSeen(Instant.ofEpochSecond(0)).build();
        Agent oneMore = builder.setProvider("differentProvider").setLastSeen(Instant.ofEpochSecond(0)).build();

        insertOrUpdateSync(recent);
        insertOrUpdateSync(old);
        insertOrUpdateSync(oneMore);
        Integer removed = deleteObsoleteSync(Instant.ofEpochSecond(1));
        assertEquals(1, removed.intValue());
        List<Agent> agents = findAllSync();
        assertEquals(1, agents.size());
        assertEquals(recent, agents.get(0));
    }
}
