package ru.yandex.solomon.core.db.dao;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.commons.lang3.RandomUtils;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.ClusterConductorGroupConf;
import ru.yandex.solomon.core.db.model.ClusterConductorTagConf;
import ru.yandex.solomon.core.db.model.ClusterHostListConf;
import ru.yandex.solomon.core.db.model.ClusterHostUrlConf;
import ru.yandex.solomon.core.db.model.ClusterNannyGroupConf;
import ru.yandex.solomon.core.db.model.ClusterNetworkConf;
import ru.yandex.solomon.core.db.model.ClusterQloudGroupConf;
import ru.yandex.solomon.core.db.model.ClusterYpConf;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.NannyEnv;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.byId;


/**
 * @author Sergey Polovko
 */
@YaExternal
public abstract class AbstractClustersDaoTest {

    private static final ClusterHostListConf FIRST_CLUSTER_HOST_LIST =
        ClusterHostListConf.of("url_pattern1", "0-10 20-30", "MAN", new String[]{ "label1", "label2" });
    private static final ClusterHostListConf SECOND_CLUSTER_HOST_LIST =
        ClusterHostListConf.of("url_pattern2", "0-10 20-30", "SAS", new String[]{ "label3", "label4" });
    private static final ClusterHostUrlConf FIRST_CLUSTER_HOST_URL =
        ClusterHostUrlConf.of("http://url1:9090", new String[]{ "one", "two" }, false);
    private static final ClusterHostUrlConf SECOND_CLUSTER_HOST_URL =
        ClusterHostUrlConf.of("http://url2:9190", new String[]{ "one", "two", "three" }, true);

    protected abstract ClustersDao getClustersDao();

    private boolean insertSync(Cluster cluster) {
        return join(getClustersDao().insert(cluster));
    }

    private Optional<Cluster> findOneSync(String projectId, String clusterId) {
        return join(getClustersDao().findOne(projectId, "", clusterId));
    }

    private Optional<Cluster> findOneSync(String projectId, String folderId, String clusterId) {
        return join(getClustersDao().findOne(projectId, "", clusterId));
    }

    private PagedResult<Cluster> findByProjectIdSync(String projectId, PageOptions pageOpts, String text) {
        return join(getClustersDao().findByProjectId(projectId, "", pageOpts, text));
    }

    private TokenBasePage<Cluster> findByProjectIdPagedSync(String projectId, int pageSize, String pageToken, String text) {
        return join(getClustersDao().findByProjectIdPaged(projectId, "", pageSize, pageToken, text));
    }

    private PagedResult<Cluster> findByFolderIdSync(String projectId, String folderId, PageOptions pageOpts, String text) {
        return join(getClustersDao().findByProjectId(projectId, folderId, pageOpts, text));
    }

    private Optional<Cluster> partialUpdateSync(Cluster cluster) {
        return join(getClustersDao().partialUpdate(cluster));
    }

    private boolean existsSync(String projectId, String clusterId) {
        return join(getClustersDao().exists(projectId, "", clusterId));
    }

    private boolean existsSync(String projectId, String folderId, String clusterId) {
        return join(getClustersDao().exists(projectId, folderId, clusterId));
    }

    private void deleteByProjectIdSync(String projectId) {
        join(getClustersDao().deleteByProjectId(projectId, ""));
    }

    private void deleteByFolderIdSync(String projectId, String folderId) {
        join(getClustersDao().deleteByProjectId(projectId, folderId));
    }

    @Test
    public void insertAndFind() {
        Cluster.Builder builder = simpleCluster();
        Cluster cluster = builder.build();
        assertTrue(insertSync(cluster));
        List<Cluster> all = join(getClustersDao().findAll());
        assertFalse(all.isEmpty());
        Cluster actualCluster = all.get(0);
        assertEquals(cluster, actualCluster);

        Cluster foundCluster = findOneSync(cluster.getProjectId(), cluster.getId()).orElseThrow(AssertionError::new);
        assertEquals(actualCluster, foundCluster);
    }

    @Test
    public void insertAndFindWithFolder() {
        Cluster cluster = simpleCluster().setFolderId("folder").build();
        assertTrue(insertSync(cluster));
        List<Cluster> all = join(getClustersDao().findAll());
        assertFalse(all.isEmpty());
        Cluster actualCluster = all.get(0);
        assertEquals(cluster, actualCluster);

        Cluster foundCluster = findOneSync(cluster.getProjectId(), cluster.getFolderId(), cluster.getId()).orElseThrow(AssertionError::new);
        assertEquals(actualCluster, foundCluster);
    }

    @Test
    public void insertDuplicate() {
        Cluster cluster = simpleCluster().setId("test-id").build();
        assertTrue(insertSync(cluster));
        assertFalse(insertSync(cluster));
    }

    @Test
    public void insertDuplicateWithFolder() {
        Cluster cluster = simpleCluster().setId("test-id").setFolderId("folder").build();
        assertTrue(insertSync(cluster));
        assertFalse(insertSync(cluster));
    }

    @Test
    public void findByProjectIdWithoutPaging() {
        Cluster.Builder builder = simpleCluster();
        Cluster solomonA = builder.setId("SA").setProjectId("Solomon").build();
        Cluster solomonB = builder.setId("SB").setProjectId("Solomon").build();
        Cluster otherX = builder.setId("X").setProjectId("Other").build();
        insertSync(solomonA);
        insertSync(solomonB);
        insertSync(otherX);

        PagedResult<Cluster> pagedResult = findByProjectIdSync("Solomon", PageOptions.ALL, "*");
        assertEquals(2, pagedResult.getTotalCount());
        List<Cluster> result = pagedResult.getResult();
        assertEquals(2, result.size());
        Map<String, Cluster> byId = byId(result, Cluster::getId);
        assertEquals(2, byId.size());
        assertTrue(byId.containsKey("SA"));
        assertTrue(byId.containsKey("SB"));
        assertPartialEqual(solomonA, byId.get("SA"));
        assertPartialEqual(solomonB, byId.get("SB"));
    }

    @Test
    public void findByFolderIdWithoutPaging() {
        Cluster.Builder builder = simpleCluster();
        Cluster folderA = builder.setId("FA").setProjectId("solomon").setFolderId("folder").build();
        Cluster folderB = builder.setId("FB").setProjectId("solomon").setFolderId("folder").build();
        Cluster other = builder.setId("X").setProjectId("solomon").setFolderId("other").build();
        insertSync(folderA);
        insertSync(folderB);
        insertSync(other);

        PagedResult<Cluster> pagedResult = findByFolderIdSync("solomon", "folder", PageOptions.ALL, "*");
        assertEquals(2, pagedResult.getTotalCount());
        List<Cluster> result = pagedResult.getResult();
        assertEquals(2, result.size());
        Map<String, Cluster> byId = byId(result, Cluster::getId);
        assertEquals(2, byId.size());
        assertTrue(byId.containsKey("FA"));
        assertTrue(byId.containsKey("FB"));
        assertPartialEqual(folderA, byId.get("FA"));
        assertPartialEqual(folderB, byId.get("FB"));
    }

    @Test
    public void findByProjectIdPaged() {
        Cluster.Builder builder = simpleCluster();
        Cluster solomonA = builder.setId("SA").setProjectId("Solomon").build();
        Cluster solomonB = builder.setId("SB").setProjectId("Solomon").build();
        Cluster otherX = builder.setId("X").setProjectId("Other").build();
        insertSync(solomonA);
        insertSync(solomonB);
        insertSync(otherX);

        PagedResult<Cluster> firstPage = findByProjectIdSync("Solomon", new PageOptions(1, 0), "*");
        assertEquals(2, firstPage.getTotalCount());
        List<Cluster> result = firstPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(solomonA, result.get(0));

        PagedResult<Cluster> nextPage = findByProjectIdSync("Solomon", new PageOptions(1, 1), "*");
        assertEquals(2, nextPage.getTotalCount());
        result = nextPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(solomonB, result.get(0));

        PagedResult<Cluster> afterPage = findByProjectIdSync("Solomon", new PageOptions(2, 1), "*");
        assertEquals(2, afterPage.getTotalCount());
        result = afterPage.getResult();
        assertTrue(result.isEmpty());
    }

    @Test
    public void findByProjectIdWithTokenBasedPage() {
        Cluster.Builder builder = simpleCluster();
        Cluster solomonA = builder.setId("SA").setProjectId("Solomon").build();
        Cluster solomonB = builder.setId("SB").setProjectId("Solomon").build();
        Cluster otherX = builder.setId("X").setProjectId("Other").build();
        insertSync(solomonA);
        insertSync(solomonB);
        insertSync(otherX);

        TokenBasePage<Cluster> firstPage = findByProjectIdPagedSync("Solomon", 1, "", "*");
        List<Cluster> result = firstPage.getItems();
        assertEquals(1, result.size());
        assertEquals(solomonA, result.get(0));
        assertEquals("1", firstPage.getNextPageToken());

        TokenBasePage<Cluster> nextPage = findByProjectIdPagedSync("Solomon", 1, firstPage.getNextPageToken(), "*");
        result = nextPage.getItems();
        assertEquals(1, result.size());
        assertEquals(solomonB, result.get(0));
        assertEquals("", nextPage.getNextPageToken());

        TokenBasePage<Cluster> afterPage = findByProjectIdPagedSync("Solomon", 1, "2", "*");
        result = afterPage.getItems();
        assertTrue(result.isEmpty());
    }

    @Test
    public void findByFolderIdPaged() {
        Cluster.Builder builder = simpleCluster();
        Cluster folderA = builder.setId("FA").setProjectId("solomon").setFolderId("folder").build();
        Cluster folderB = builder.setId("FB").setProjectId("solomon").setFolderId("folder").build();
        Cluster otherX = builder.setId("X").setProjectId("solomon").setFolderId("other").build();
        insertSync(folderA);
        insertSync(folderB);
        insertSync(otherX);

        PagedResult<Cluster> firstPage = findByFolderIdSync("solomon", "folder", new PageOptions(1, 0), "*");
        assertEquals(2, firstPage.getTotalCount());
        List<Cluster> result = firstPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(folderA, result.get(0));

        PagedResult<Cluster> nextPage = findByFolderIdSync("solomon", "folder", new PageOptions(1, 1), "*");
        assertEquals(2, nextPage.getTotalCount());
        result = nextPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(folderB, result.get(0));

        PagedResult<Cluster> afterPage = findByFolderIdSync("solomon", "folder", new PageOptions(2, 1), "*");
        assertEquals(2, afterPage.getTotalCount());
        result = afterPage.getResult();
        assertTrue(result.isEmpty());
    }

    @Test
    public void findByProjectIdWithRegexpPaged() {
        Cluster.Builder builder = simpleCluster();
        Cluster solomonD = builder.setId("SolomonD").setName("Кластер для Solomon.D").setProjectId("Solomon").build();
        Cluster solomonF = builder.setId("SolomonF").setName("Кластер для Solomon.F").setProjectId("Solomon").build();
        Cluster solomonX = builder.setId("SolomonX").setName("Кластер X для Избранных").setProjectId("Solomon").build();
        Cluster otherX = builder.setId("X").setName("Кластер для Other").setProjectId("Other").build();
        insertSync(solomonF);
        insertSync(solomonX);
        insertSync(solomonD);
        insertSync(otherX);

        PagedResult<Cluster> firstPage = findByProjectIdSync("Solomon", new PageOptions(1, 0), "для");
        assertEquals(3, firstPage.getTotalCount());
        List<Cluster> result = firstPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(solomonD, result.get(0));

        firstPage = findByProjectIdSync("Solomon", new PageOptions(4, 0), "для");
        assertEquals(3, firstPage.getTotalCount());
        result = firstPage.getResult();
        assertEquals(3, result.size());
        assertPartialEqual(solomonD, result.get(0));
        assertPartialEqual(solomonF, result.get(1));
        assertPartialEqual(solomonX, result.get(2));

        PagedResult<Cluster> secondPage = findByProjectIdSync("Solomon", new PageOptions(1, 1), "для Solomon.");
        assertEquals(2, secondPage.getTotalCount());
        result = secondPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(solomonF, result.get(0));

        PagedResult<Cluster> thirdPage = findByProjectIdSync("Solomon", new PageOptions(1, 2), "Solomon*");
        assertEquals(3, thirdPage.getTotalCount());
        result = thirdPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(solomonX, result.get(0));
    }

    @Test
    public void findByFolderIdWithRegexpPaged() {
        Cluster.Builder builder = simpleCluster();
        Cluster folderD = builder.setId("SolomonD").setName("Кластер для Solomon.D").setProjectId("solomon").setFolderId("folder").build();
        Cluster folderF = builder.setId("SolomonF").setName("Кластер для Solomon.F").setProjectId("solomon").setFolderId("folder").build();
        Cluster folderX = builder.setId("SolomonX").setName("Кластер X для Избранных").setProjectId("solomon").setFolderId("folder").build();
        Cluster otherX = builder.setId("X").setName("Кластер для Other").setProjectId("solomon").setFolderId("other").build();
        insertSync(folderF);
        insertSync(folderX);
        insertSync(folderD);
        insertSync(otherX);

        PagedResult<Cluster> firstPage = findByFolderIdSync("solomon", "folder", new PageOptions(1, 0), "для");
        assertEquals(3, firstPage.getTotalCount());
        List<Cluster> result = firstPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(folderD, result.get(0));

        firstPage = findByFolderIdSync("solomon", "folder", new PageOptions(4, 0), "для");
        assertEquals(3, firstPage.getTotalCount());
        result = firstPage.getResult();
        assertEquals(3, result.size());
        assertPartialEqual(folderD, result.get(0));
        assertPartialEqual(folderF, result.get(1));
        assertPartialEqual(folderX, result.get(2));

        PagedResult<Cluster> secondPage = findByFolderIdSync("solomon", "folder", new PageOptions(1, 1), "для Solomon.");
        assertEquals(2, secondPage.getTotalCount());
        result = secondPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(folderF, result.get(0));

        PagedResult<Cluster> thirdPage = findByFolderIdSync("solomon", "folder", new PageOptions(1, 2), "Solomon*");
        assertEquals(3, thirdPage.getTotalCount());
        result = thirdPage.getResult();
        assertEquals(1, result.size());
        assertPartialEqual(folderX, result.get(0));
    }


    @Test
    public void partialUpdateClusterNotFound() {
        Cluster.Builder builder = simpleCluster();
        Cluster clusterA = builder.setId("A").setProjectId("Project").build();
        Cluster clusterB = builder.setId("B").setProjectId("Project").build();
        insertSync(clusterA);
        insertSync(clusterB);

        Cluster unknownC = builder.setId("C").setProjectId("Project").build();
        assertFalse(partialUpdateSync(unknownC).isPresent());
        assertFalse(findOneSync("Project", "C").isPresent());

        Cluster unknownA = builder.setId("A").setProjectId("Unknown").build();
        assertFalse(partialUpdateSync(unknownA).isPresent());
        assertFalse(findOneSync("Unknown", "A").isPresent());
    }

    @Test
    public void partialUpdateClusterNotFoundWithFolder() {
        Cluster.Builder builder = simpleCluster();
        Cluster clusterA = builder.setId("A").setProjectId("Project").setFolderId("folder").build();
        Cluster clusterB = builder.setId("B").setProjectId("Project").setFolderId("folder").build();
        insertSync(clusterA);
        insertSync(clusterB);

        Cluster unknownC = builder.setId("C").setProjectId("Project").setFolderId("folder").build();
        assertFalse(partialUpdateSync(unknownC).isPresent());
        assertFalse(findOneSync("Project", "folder", "C").isPresent());

        Cluster unknownA = builder.setId("A").setProjectId("Unknown").setFolderId("folder").build();
        assertFalse(partialUpdateSync(unknownA).isPresent());
        assertFalse(findOneSync("Unknown", "folder", "A").isPresent());
    }

    @Test
    public void partialUpdateClusterStaleVersion() {
        Cluster.Builder builder = simpleCluster();
        builder.setVersion(10);
        Cluster cluster = builder.setId("A").setProjectId("Project").build();
        insertSync(cluster);
        builder.setVersion(9);
        assertFalse(partialUpdateSync(builder.build()).isPresent());
        findOneSync(cluster.getProjectId(), cluster.getId())
            .map(cluster::equals)
            .orElseThrow(AssertionError::new);
    }

    @Test
    public void partialUpdateClusterStaleVersionWithFolder() {
        Cluster.Builder builder = simpleCluster();
        builder.setVersion(10);
        Cluster cluster = builder.setId("A").setProjectId("Project").setFolderId("folder").build();
        insertSync(cluster);
        builder.setVersion(9);
        assertFalse(partialUpdateSync(builder.build()).isPresent());
        findOneSync(cluster.getProjectId(), cluster.getFolderId(), cluster.getId())
            .map(cluster::equals)
            .orElseThrow(AssertionError::new);
    }

    @Test
    public void partialUpdate() {
        Cluster.Builder builder = simpleCluster().setId("A").setProjectId("Project").setShardSettings(simpleShardSettings());
        Cluster cluster = builder.build();
        insertSync(cluster);
        Cluster.Builder update = randomCluster().setId("A").setProjectId("Project");
        Cluster afterUpdate = partialUpdateSync(update.build()).orElseThrow(AssertionError::new);
        update
            .setVersion(update.getVersion() + 1)
            .setCreatedBy(cluster.getCreatedBy())
            .setCreatedAt(cluster.getCreatedAt());
        assertEquals(update.build(), afterUpdate);
        Cluster foundCluster =
            findOneSync(afterUpdate.getProjectId(), afterUpdate.getId()).orElseThrow(AssertionError::new);
        assertEquals(afterUpdate, foundCluster);
    }

    @Test
    public void partialUpdateWithoutVersion() {
        Cluster.Builder builder = simpleCluster().setId("A").setProjectId("Project").setShardSettings(simpleShardSettings());
        Cluster cluster = builder.build();
        insertSync(cluster);
        Cluster update = randomCluster()
                .setId("A")
                .setProjectId("Project")
                .setVersion(-1)
                .setLabels(Map.of("label1", "value1"))
                .build();
        Cluster afterUpdate = partialUpdateSync(update).orElseThrow(AssertionError::new);
        Cluster expectedCluster = update.toBuilder()
                .setVersion(1)
                .setCreatedBy(cluster.getCreatedBy())
                .setCreatedAt(cluster.getCreatedAt())
                .build();
        assertEquals(expectedCluster, afterUpdate);
    }

    @Test
    public void partialUpdateWithFolder() {
        Cluster cluster = simpleCluster()
            .setId("A")
            .setProjectId("Project")
            .setFolderId("folder")
            .setShardSettings(simpleShardSettings())
            .build();
        insertSync(cluster);

        Cluster.Builder update = randomCluster().setId("A").setProjectId("Project").setFolderId("folder2");
        Cluster afterUpdate = partialUpdateSync(update.build()).orElseThrow(AssertionError::new);
        update
            .setVersion(update.getVersion() + 1)
            .setCreatedBy(cluster.getCreatedBy())
            .setCreatedAt(cluster.getCreatedAt());
        assertEquals(update.build(), afterUpdate);

        Cluster foundCluster = findOneSync(afterUpdate.getProjectId(), afterUpdate.getFolderId(), afterUpdate.getId())
            .orElseThrow(AssertionError::new);
        assertEquals(afterUpdate, foundCluster);
    }

    @Test
    public void insertExistsDelete() {
        Cluster.Builder builder = simpleCluster();
        insertSync(builder.setId("A").setProjectId("Solomon").build());
        insertSync(builder.setId("B").setProjectId("Solomon").build());
        insertSync(builder.setId("C").setProjectId("Solomon").build());
        insertSync(builder.setId("X").setProjectId("Other").build());

        assertTrue(existsSync("Solomon", "B"));
        assertTrue(existsSync("Solomon", "A"));
        assertTrue(existsSync("Solomon", "C"));
        assertFalse(existsSync("Solomon", "D"));
        assertTrue(existsSync("Other", "X"));
        assertFalse(existsSync("Project", "X"));

        assertTrue(join(getClustersDao().deleteOne("Solomon", "", "A")));
        assertFalse(existsSync("Solomon", "A"));
        assertTrue(existsSync("Solomon", "B"));
        assertTrue(existsSync("Solomon", "C"));

        deleteByProjectIdSync("Solomon");
        assertFalse(existsSync("Solomon", "B"));
        assertFalse(existsSync("Solomon", "C"));
        assertTrue(existsSync("Other", "X"));

        deleteByProjectIdSync("Other");
        assertFalse(existsSync("Other", "X"));
    }

    @Test
    public void insertExistsDeleteWithFolder() {
        Cluster.Builder builder = simpleCluster();
        insertSync(builder.setId("A").setProjectId("Solomon").setFolderId("folder").build());
        insertSync(builder.setId("B").setProjectId("Solomon").setFolderId("folder").build());
        insertSync(builder.setId("C").setProjectId("Solomon").setFolderId("folder").build());
        insertSync(builder.setId("X").setProjectId("Solomon").setFolderId("other").build());

        assertTrue(existsSync("Solomon", "folder", "B"));
        assertTrue(existsSync("Solomon", "folder", "A"));
        assertTrue(existsSync("Solomon", "folder", "C"));
        assertFalse(existsSync("Solomon", "folder", "D"));
        assertTrue(existsSync("Solomon", "other", "X"));
        assertFalse(existsSync("Project", "folder", "X"));

        assertTrue(join(getClustersDao().deleteOne("Solomon", "folder", "A")));
        assertFalse(existsSync("Solomon", "folder",  "A"));
        assertTrue(existsSync("Solomon", "folder", "B"));
        assertTrue(existsSync("Solomon", "folder",  "C"));

        deleteByFolderIdSync("Solomon", "folder");
        assertFalse(existsSync("Solomon", "folder", "B"));
        assertFalse(existsSync("Solomon", "folder", "C"));
        assertTrue(existsSync("Solomon", "other", "X"));

        deleteByFolderIdSync("Solomon", "other");
        assertFalse(existsSync("Solomon", "other", "X"));
    }

    @Test
    public void findAllPaged() {
        Cluster.Builder builder = simpleCluster();
        insertSync(builder.setId("cluster4").setName("Cluster4").setProjectId("solomon").setFolderId("folder1").build());
        insertSync(builder.setId("cluster3").setName("Cluster3").setProjectId("solomon").setFolderId("folder2").build());
        insertSync(builder.setId("cluster2").setName("Cluster2").setProjectId("air").build());
        insertSync(builder.setId("cluster1").setName("Cluster1").setProjectId("air").build());

        List<Cluster> clusters = join(getClustersDao().findAll(new PageOptions(2, 0), "")).getResult();
        List<String> clusterIds = clusters.stream()
            .map(Cluster::getId)
            .collect(Collectors.toList());

        Assert.assertEquals(clusterIds, List.of("cluster1", "cluster2"));
    }

    @Test
    public void findAllPagedWithText() {
        Cluster.Builder builder = simpleCluster();
        insertSync(builder.setId("cluster4").setName("Cluster4").setProjectId("solomon").build());
        insertSync(builder.setId("cluster3").setName("Cluster3").setProjectId("solomon").build());
        insertSync(builder.setId("cluster2").setName("Cluster2").setProjectId("solomon").build());
        insertSync(builder.setId("cluster1").setName("Cluster1").setProjectId("solomon").build());

        List<Cluster> clusters = join(getClustersDao().findAll(new PageOptions(2, 0), "1")).getResult();
        List<String> clusterIds = clusters.stream()
            .map(Cluster::getId)
            .collect(Collectors.toList());

        Assert.assertEquals(clusterIds, List.of("cluster1"));
    }

    private static void assertPartialEqual(Cluster expected, Cluster actual) {
        assertEquals(expected.getId(), actual.getId());
        assertEquals(expected.getName(), actual.getName());
        assertEquals(expected.getProjectId(), actual.getProjectId());
    }

    private static Cluster.Builder simpleCluster() {
        return Cluster.newBuilder()
            .setId("one")
            .setProjectId("projectx")
            .setName("some cluster")
            .setDescription("some description")
            .setHosts(allHost())
            .setHostUrls(allHostUrls())
            .setConductorGroups(simpleConductorGroups())
            .setConductorTags(simpleConductorTags())
            .setNannyGroups(simpleNannyGroups())
            .setQloudGroups(simpleQloudGroups())
            .setNetworks(simpleNetworkGroup())
            .setYpClusters(simpleYp())
            .setCreatedAt(Instant.ofEpochMilli(System.currentTimeMillis()))
            .setShardSettings(ShardSettings.of(ShardSettings.Type.PULL,
                    ShardSettings.PullSettings.newBuilder().setHostLabelPolicy(ShardSettings.HostLabelPolicy.FULL_HOSTNAME).build(),
                    0,
                    73,
                    DecimPolicy.UNDEFINED,
                    ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                    0))
            .setCreatedBy("snoop")
            .setLabels(Map.of("label1", "value1", "label2", "value2"))
            .setFolderId(RandomUtils.nextBoolean() ? "" : "some_folder");
    }

    private static Cluster.Builder randomCluster() {
        String id = UUID.randomUUID().toString();
        Cluster.Builder builder = Cluster.newBuilder()
            .setId(id)
            .setProjectId("Project-" + id)
            .setName("Name-" + id)
            .setDescription("Description for " + id)
            .setShardSettings(simpleShardSettings())
            .setLabels(Map.of("label1", "value1", "label2", "value2"))
            .setHosts(Collections.singletonList(FIRST_CLUSTER_HOST_LIST))
            .setHostUrls(Collections.singletonList(FIRST_CLUSTER_HOST_URL))
            .setConductorGroups(simpleConductorGroups())
            .setConductorTags(simpleConductorTags())
            .setNannyGroups(simpleNannyGroups())
            .setQloudGroups(simpleQloudGroups())
            .setNetworks(simpleNetworkGroup())
            .setYpClusters(simpleYp());
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        Instant createdAt = now.minus(Duration.ofDays(RandomUtils.nextInt(1, 10)));
        Instant updatedAt = now.plusSeconds(RandomUtils.nextInt(1, 10));
        builder.setCreatedAt(createdAt).setCreatedBy("other").setUpdatedBy("updater").setUpdatedAt(updatedAt);
        return builder;
    }

    private static List<ClusterQloudGroupConf> simpleQloudGroups() {
        return Arrays.asList(ClusterQloudGroupConf.of("compponen1", "environment1", "app1", "project1", "deploy1",
            new String[]{ "l1" }));
    }

    private static List<ClusterNannyGroupConf> simpleNannyGroups() {
        return Arrays.asList(
            ClusterNannyGroupConf.of("service1", new String[]{ "label1" }, true, NannyEnv.PRODUCTION, 80, new String[]{ "cfg1", "cfg2" }));
    }

    private static List<ClusterConductorTagConf> simpleConductorTags() {
        return Arrays.asList(ClusterConductorTagConf.of("имя", new String[]{ "метка1", "метка2" }));
    }

    private static List<ClusterConductorGroupConf> simpleConductorGroups() {
        return Arrays.asList(ClusterConductorGroupConf.of("first", new String[]{ "раз", "два" }));
    }

    private static List<ClusterNetworkConf> simpleNetworkGroup() {
        return Arrays.asList(ClusterNetworkConf.of("127.0.0.1", 6666, new String[]{"адын", "адынадын["}));
    }

    private static List<ClusterYpConf> simpleYp() {
        return Arrays.asList(ClusterYpConf.of(
            "f.man.yp-c.yandex.net", "monitoring-charts-prod.charts", "cluster", new String[]{"k1=t1", "k2=t2"}, "100500", "/my/label"));
    }

    private static ShardSettings simpleShardSettings() {
        var pullSettings = ShardSettings.PullSettings.newBuilder()
                .setPort(10)
                .setPath("/metrics")
                .setAddTsArgs(true)
                .setProtocol(ShardSettings.PullProtocol.HTTP)
                .setHostLabelPolicy(ShardSettings.HostLabelPolicy.SHORT_HOSTNAME)
                .setTvmDestinationId("tvm")
                .build();

        var aggregationSettings = ShardSettings.AggregationSettings.of(false,
                new ServiceMetricConf.AggrRule[0], false);

        return ShardSettings.of(
                ShardSettings.Type.PULL,
                pullSettings,
                5,
                31,
                DecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS,
                aggregationSettings,
                15
        );
    }

    private static List<ClusterHostUrlConf> allHostUrls() {
        return Arrays.asList(FIRST_CLUSTER_HOST_URL, SECOND_CLUSTER_HOST_URL);
    }

    private static List<ClusterHostListConf> allHost() {
        return Arrays.asList(FIRST_CLUSTER_HOST_LIST, SECOND_CLUSTER_HOST_LIST);
    }

}
