package ru.yandex.solomon.core.db.dao;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.DashboardPanel;
import ru.yandex.solomon.core.db.model.DashboardRow;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static junit.framework.TestCase.assertEquals;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.equalsByString;


/**
 * @author Oleg Baryshnikov
 */
@YaExternal
public abstract class AbstractDashboardsDaoTest {

    protected abstract DashboardsDao getDashboardsDao();

    @Test
    public void insertOneDashboard() {
        Dashboard dashboard = new Dashboard(
            "dashboard",
            "project",
            "",
            "Dashboard #1",
            "description",
            1,
            new Selector[]{ new Selector("host", "*") },
            new DashboardRow[]{
                new DashboardRow(
                    new DashboardPanel[]{
                        new DashboardPanel(DashboardPanel.Type.IFRAME, "title", "",
                            "https://yandex.ru", "", 0, 0)
                    }
                ),
            },
            "mon123456",
            Instant.parse("2017-10-10T07:48:00Z"),
            Instant.parse("2017-10-10T07:48:00Z"),
            "user",
            "user",
            0
        );

        Assert.assertTrue(insertSync(dashboard));

        Optional<Dashboard> fromDb = findOneSync(dashboard.getProjectId(), dashboard.getId());
        Assert.assertTrue(fromDb.isPresent());
        Assert.assertEquals(dashboard, fromDb.get());
    }

    @Test
    public void insertOneDashboardWithFolder() {
        Dashboard dashboard = new Dashboard(
            "dashboard",
            "project",
            "folder",
            "Dashboard #1",
            "description",
            1,
            new Selector[]{ new Selector("host", "*") },
            new DashboardRow[]{
                new DashboardRow(
                    new DashboardPanel[]{
                        new DashboardPanel(DashboardPanel.Type.IFRAME, "title", "",
                            "https://yandex.ru", "", 0, 0)
                    }
                ),
            },
            "mon123456",
            Instant.parse("2017-10-10T07:48:00Z"),
            Instant.parse("2017-10-10T07:48:00Z"),
            "user",
            "user",
            0
        );

        Assert.assertTrue(insertSync(dashboard));

        Optional<Dashboard> fromDb = findOneSync(dashboard.getProjectId(), dashboard.getFolderId(), dashboard.getId());
        Assert.assertTrue(fromDb.isPresent());
        Assert.assertEquals(dashboard, fromDb.get());
    }

    @Test
    public void findByProjectIdShorted() {
        List<Dashboard> dashboards = Arrays.asList(
            Dashboard.newBuilder()
                .setId("dashboard1")
                .setProjectId("junk")
                .setName("Junk dashboard #1")
                .setParameters(new Selector[]{ new Selector("host", "cluster")})
                    .setGeneratedId("mon123451")
                .build(),
            Dashboard.newBuilder()
                .setId("dashboard2")
                .setProjectId("test")
                .setName("Junk dashboard #2")
                .setGeneratedId("mon123452")
                .build()
        );

        List<ShortGraphOrDashboardConf> expectedShortConfs =
            dashboards.stream()
                .filter(dash -> "junk".equals(dash.getProjectId()))
                .map(AbstractDashboardsDaoTest::toShortDashboardConf)
                .collect(Collectors.toList());

        for (Dashboard dashboard : dashboards) {
            Assert.assertTrue(insertSync(dashboard));
        }

        List<ShortGraphOrDashboardConf> shortConfs = join(getDashboardsDao().findByProjectIdShorted("junk", ""));

        Assert.assertFalse(shortConfs.isEmpty());
        Assert.assertEquals(expectedShortConfs, shortConfs);
    }

    @Test
    public void findByFolderIdShorted() {
        List<Dashboard> dashboards = Arrays.asList(
            Dashboard.newBuilder()
                .setId("dashboard1")
                .setProjectId("junk")
                .setFolderId("folder1")
                .setName("Junk dashboard #1")
                .setParameters(new Selector[]{ new Selector("host", "cluster")})
                    .setGeneratedId("mon123451")
                .build(),
            Dashboard.newBuilder()
                .setId("dashboard2")
                .setProjectId("junk")
                .setFolderId("folder2")
                .setName("Junk dashboard #2")
                .setGeneratedId("mon123452")
                .build()
        );

        List<ShortGraphOrDashboardConf> expectedShortConfs =
            dashboards.stream()
                .filter(dash -> "folder1".equals(dash.getFolderId()))
                .map(AbstractDashboardsDaoTest::toShortDashboardConf)
                .collect(Collectors.toList());

        for (Dashboard dashboard : dashboards) {
            Assert.assertTrue(insertSync(dashboard));
        }

        List<ShortGraphOrDashboardConf> shortConfs = join(getDashboardsDao().findByProjectIdShorted("junk", "folder1"));

        Assert.assertFalse(shortConfs.isEmpty());
        Assert.assertEquals(expectedShortConfs, shortConfs);
    }

    private static ShortGraphOrDashboardConf toShortDashboardConf(Dashboard dashboard) {
        return new ShortGraphOrDashboardConf(
            dashboard.getId(),
            dashboard.getProjectId(),
            dashboard.getName(),
            dashboard.getParameters()
        );
    }

    @Test
    public void insertSeveralDashboards() {
        Dashboard dashboard = makeSimpleDashboard("project", "", "dashboard", "Dashboard");
        Assert.assertTrue(insertSync(dashboard));
        Assert.assertFalse(insertSync(dashboard));
        Assert.assertTrue(insertSync(makeSimpleDashboard("project", "", "dashboard2", "Dashboard2")));
    }

    @Test
    public void insertSeveralDashboardsWithFolder() {
        Dashboard dashboard = makeSimpleDashboard("project", "folder", "dashboard", "Dashboard");
        Assert.assertTrue(insertSync(dashboard));
        Assert.assertFalse(insertSync(dashboard));
        Assert.assertTrue(insertSync(makeSimpleDashboard("project", "folder", "dashboard2", "Dashboard2")));
    }

    @Test
    public void findByProjectIdWithAllPageSize() {
        List<Dashboard> dashboards = fillDashboardCollection();
        List<Dashboard> exprectedDashboards = dashboards.stream()
            .filter(s -> "solomon".equals(s.getProjectId()))
            .collect(Collectors.toList());

        PagedResult<Dashboard> solomonDashboardsFromDb = findByProjectIdSync("solomon", PageOptions.ALL, "");
        assertPartialEquality(exprectedDashboards, solomonDashboardsFromDb.getResult());

        Assert.assertEquals(0, solomonDashboardsFromDb.getCurrentPage());
        Assert.assertEquals(1, solomonDashboardsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonDashboardsFromDb.getTotalCount());
    }

    @Test
    public void findByFolderIdWithAllPageSize() {
        List<Dashboard> dashboards = fillDashboardCollection();
        List<Dashboard> exprectedDashboards = dashboards.stream()
            .filter(s -> "solomon".equals(s.getFolderId()))
            .collect(Collectors.toList());

        PagedResult<Dashboard> solomonDashboardsFromDb = findByFolderIdSync("solomon", "solomon", PageOptions.ALL, "");
        assertPartialEquality(exprectedDashboards, solomonDashboardsFromDb.getResult());

        Assert.assertEquals(0, solomonDashboardsFromDb.getCurrentPage());
        Assert.assertEquals(1, solomonDashboardsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonDashboardsFromDb.getTotalCount());
    }

    @Test
    public void findByProjectIdWithPageSize() {
        List<Dashboard> dashboards = fillDashboardCollection();
        List<Dashboard> expectedDashboards = dashboards.stream()
            .filter(s -> "solomon".equals(s.getProjectId()))
            .limit(2)
            .collect(Collectors.toList());

        PagedResult<Dashboard> solomonDashboardsFromDb = findByProjectIdSync("solomon", new PageOptions(2, 0), "");
        assertPartialEquality(expectedDashboards, solomonDashboardsFromDb.getResult());

        Assert.assertEquals(0, solomonDashboardsFromDb.getCurrentPage());
        Assert.assertEquals(2, solomonDashboardsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonDashboardsFromDb.getTotalCount());
    }

    @Test
    public void findByFolderIdWithPageSize() {
        List<Dashboard> dashboards = fillDashboardCollection();
        List<Dashboard> expectedDashboards = dashboards.stream()
            .filter(s -> "solomon".equals(s.getFolderId()))
            .limit(2)
            .collect(Collectors.toList());

        PagedResult<Dashboard> solomonDashboardsFromDb = findByFolderIdSync("solomon", "solomon", new PageOptions(2, 0), "");
        assertPartialEquality(expectedDashboards, solomonDashboardsFromDb.getResult());

        Assert.assertEquals(0, solomonDashboardsFromDb.getCurrentPage());
        Assert.assertEquals(2, solomonDashboardsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonDashboardsFromDb.getTotalCount());
    }

    @Test
    public void findByProjectIdAndText() {
        List<Dashboard> dashboards = Arrays.asList(
            makeSimpleDashboard("solomon", "", "d1", "dashboard 1"),
            makeSimpleDashboard("solomon", "", "d2", "Dashboard 2"),
            makeSimpleDashboard("solomon", "", "dashboard3", "d3"),
            makeSimpleDashboard("solomon", "", "d4", "d4")
        );

        for (Dashboard dashboard : dashboards) {
            Assume.assumeTrue(insertSync(dashboard));
        }

        List<Dashboard> expectedDashboards = dashboards.stream()
            .filter(s -> StringUtils.containsIgnoreCase(s.getId(), "dash")
                    || StringUtils.containsIgnoreCase(s.getName(), "dash"))
            .collect(Collectors.toList());

        PagedResult<Dashboard> solomonDashboardsFromDb = findByProjectIdSync("solomon", PageOptions.ALL, "dashboard");
        assertPartialEquality(expectedDashboards, solomonDashboardsFromDb.getResult());
    }

    @Test
    public void findByFolderIdAndText() {
        List<Dashboard> dashboards = Arrays.asList(
            makeSimpleDashboard("solomon", "solomon", "d1", "dashboard 1"),
            makeSimpleDashboard("solomon", "solomon", "d2", "Dashboard 2"),
            makeSimpleDashboard("solomon", "solomon", "dashboard3", "d3"),
            makeSimpleDashboard("solomon", "solomon", "d4", "d4")
        );

        for (Dashboard dashboard : dashboards) {
            Assume.assumeTrue(insertSync(dashboard));
        }

        List<Dashboard> expectedDashboards = dashboards.stream()
            .filter(s -> StringUtils.containsIgnoreCase(s.getId(), "dash")
                    || StringUtils.containsIgnoreCase(s.getName(), "dash"))
            .collect(Collectors.toList());

        PagedResult<Dashboard> solomonDashboardsFromDb = findByFolderIdSync("solomon", "solomon", PageOptions.ALL, "dashboard");
        assertPartialEquality(expectedDashboards, solomonDashboardsFromDb.getResult());
    }

    private List<Dashboard> fillDashboardCollection() {
        List<String> projectIds = List.of("kikimr", "yql", "solomon");
        List<Dashboard> dashboards = new ArrayList<>(projectIds.size() * 4);

        for (String projectId : projectIds) {
            dashboards.add(makeSimpleDashboard(projectId, projectId, projectId + "_d1", "dashboard 1"));
            dashboards.add(makeSimpleDashboard(projectId, projectId, projectId + "_d2", "dashboard 2"));
            dashboards.add(makeSimpleDashboard(projectId, projectId, projectId + "_d3", "d3"));
            dashboards.add(makeSimpleDashboard(projectId, projectId, projectId + "_d4", "dashboard 4"));
        }

        for (Dashboard dashboard : dashboards) {
            Assume.assumeTrue(insertSync(dashboard));
        }
        return dashboards;
    }

    private void assertPartialEquality(List<Dashboard> expected, List<Dashboard> actual) {
        Assert.assertEquals(expected.size(), actual.size());
        expected.sort(Comparator.comparing(Dashboard::getProjectId).thenComparing(Comparator.comparing(Dashboard::getName)));
        actual.sort(Comparator.comparing(Dashboard::getProjectId).thenComparing(Comparator.comparing(Dashboard::getName)));

        for (int i = 0; i < expected.size(); ++i) {
            Dashboard expectedDashboard = expected.get(i);
            Dashboard actualDashboard = actual.get(i);
            Assert.assertEquals(expectedDashboard.getProjectId(), actualDashboard.getProjectId());
            Assert.assertEquals(expectedDashboard.getId(), actualDashboard.getId());
            Assert.assertEquals(expectedDashboard.getName(), actualDashboard.getName());
        }
    }

    @Test
    public void partialUpdate() {
        Dashboard dashboard = new Dashboard(
            "dashboard",
            "project",
                "", "Dashboard",
            "description",
            1,
            new Selector[]{ new Selector("host", "*") },
            new DashboardRow[]{
                new DashboardRow(
                    new DashboardPanel[]{
                        new DashboardPanel(DashboardPanel.Type.IFRAME, "title", "subtitle", "https://yandex.ru", "",0, 0)
                    }
                ),
            },
            "mon123456",
            Instant.parse("2017-10-10T07:48:00Z"),
            Instant.parse("2017-10-10T07:48:00Z"),
            "user",
            "user",
            0
        );

        Dashboard dashboardToUpdate = new Dashboard(
            "dashboard",
            "project",
                "", "Dashboard (updated)",
            "description",
            1.25,
            new Selector[]{ new Selector("host", "*"), new Selector("sensor", "*") },
            new DashboardRow[]{
                new DashboardRow(
                    new DashboardPanel[]{
                        new DashboardPanel(DashboardPanel.Type.IFRAME, "title", "subtitle", "https://wiki.yandex-team.ru", "", 0, 0),
                        new DashboardPanel(DashboardPanel.Type.MARKDOWN, "title", "subtitle", "https://my.at.yandex-team.ru", "", 0, 0),
                    }
                ),
            },
            "mon123457",
            Instant.parse("2017-10-10T07:48:00Z"),
            Instant.parse("2017-10-10T07:57:00Z"),
            "user",
            "user_2",
            0
        );

        Assume.assumeTrue(insertSync(dashboard));

        Optional<Dashboard> updatedDashboardOpt = join(getDashboardsDao().partialUpdate(dashboardToUpdate));
        Assert.assertTrue(updatedDashboardOpt.isPresent());

        Dashboard updatedDashboard = updatedDashboardOpt.get();
        Dashboard expectedDashboard = updatedDashboard.toBuilder()
                .setVersion(1)
                .setGeneratedId(dashboard.getGeneratedId())
                .build();
        Assert.assertEquals(expectedDashboard, updatedDashboard);
    }

    @Test
    public void partialUpdateWithFolder() {
        Dashboard dashboard = new Dashboard(
            "dashboard",
            "project",
            "folder",
            "Dashboard",
            "description",
            1,
            new Selector[]{ new Selector("host", "*") },
            new DashboardRow[]{
                new DashboardRow(
                    new DashboardPanel[]{
                        new DashboardPanel(DashboardPanel.Type.IFRAME, "title", "subtitle", "https://yandex.ru", "",0, 0)
                    }
                ),
            },
            "mon123456",
            Instant.parse("2017-10-10T07:48:00Z"),
            Instant.parse("2017-10-10T07:48:00Z"),
            "user",
            "user",
            0
        );

        Dashboard dashboardToUpdate = new Dashboard(
            "dashboard",
            "project",
            "folder",
            "Dashboard (updated)",
            "description",
            1.25,
            new Selector[]{ new Selector("host", "*"), new Selector("sensor", "*") },
            new DashboardRow[]{
                new DashboardRow(
                    new DashboardPanel[]{
                        new DashboardPanel(DashboardPanel.Type.IFRAME, "title", "subtitle", "https://wiki.yandex-team.ru", "", 0, 0),
                        new DashboardPanel(DashboardPanel.Type.MARKDOWN, "title", "subtitle", "https://my.at.yandex-team.ru", "", 0, 0),
                    }
                ),
            },
            "mon123457",
            Instant.parse("2017-10-10T07:48:00Z"),
            Instant.parse("2017-10-10T07:57:00Z"),
            "user",
            "user_2",
            0
        );

        Assume.assumeTrue(insertSync(dashboard));

        Optional<Dashboard> updatedDashboardOpt = join(getDashboardsDao().partialUpdate(dashboardToUpdate));
        Assert.assertTrue(updatedDashboardOpt.isPresent());

        Dashboard updatedDashboard = updatedDashboardOpt.get();
        Dashboard expectedDashboard = updatedDashboard.toBuilder()
                .setVersion(1)
                .setGeneratedId(dashboard.getGeneratedId())
                .build();
        Assert.assertEquals(expectedDashboard, updatedDashboard);
    }

    @Test
    public void deleteOne() {
        Dashboard dashboard = makeSimpleDashboard("project", "", "dashboard", StringUtils.capitalize("dashboard"));

        Assume.assumeTrue(insertSync(dashboard));

        Assert.assertTrue(deleteOneSync(dashboard.getProjectId(), dashboard.getId()));
        Assume.assumeFalse(findOneSync(dashboard.getProjectId(), dashboard.getId()).isPresent());

        Assert.assertFalse(deleteOneSync("project", "dashboard_2"));
        Assert.assertFalse(deleteOneSync("project_2", "dashboard"));
    }

    @Test
    public void deleteOneWithFolder() {
        Dashboard dashboard = makeSimpleDashboard("project", "folder", "dashboard", StringUtils.capitalize("dashboard"));

        Assume.assumeTrue(insertSync(dashboard));

        Assert.assertTrue(deleteOneSync(dashboard.getProjectId(), dashboard.getFolderId(), dashboard.getId()));
        Assume.assumeFalse(findOneSync(dashboard.getProjectId(), dashboard.getFolderId(), dashboard.getId()).isPresent());

        Assert.assertFalse(deleteOneSync("project", "folder", "dashboard_2"));
        Assert.assertFalse(deleteOneSync("project_2", "folder_2", "dashboard"));
    }

    @Test
    public void exists() {
        Dashboard dashboard = makeSimpleDashboard("project", "", "dashboard", StringUtils.capitalize("dashboard"));

        Assume.assumeTrue(insertSync(dashboard));

        Assert.assertTrue(existsSync("project", "dashboard"));
        Assert.assertFalse(existsSync("project", "dashboard2"));
    }

    @Test
    public void existsWithFolder() {
        Dashboard dashboard = makeSimpleDashboard("project", "folder", "dashboard", StringUtils.capitalize("dashboard"));

        Assume.assumeTrue(insertSync(dashboard));

        Assert.assertTrue(existsSync("project", "folder", "dashboard"));
        Assert.assertFalse(existsSync("project", "folder", "dashboard2"));
    }

    @Test
    public void insertFindAndDelete() {
        Dashboard.Builder builder = simpleDashboard();
        Dashboard solomonA = builder.setId("A").setProjectId("Solomon").build();
        Dashboard solomonB = builder.setId("B").setProjectId("Solomon").build();
        Dashboard solomonC = builder.setId("C").setProjectId("Solomon").build();
        Dashboard otherX = builder.setId("X").setProjectId("Other").build();

        Assert.assertTrue(insertSync(solomonA));
        Assert.assertTrue(insertSync(solomonB));
        Assert.assertTrue(insertSync(solomonC));
        Assert.assertTrue(insertSync(otherX));
        Assert.assertFalse(insertSync(solomonA));

        Assert.assertTrue(existsSync("Solomon", "A"));
        Dashboard foundA = findOneSync("Solomon", "A").orElseThrow(AssertionError::new);
        equalsByString(solomonA, foundA);

        Assert.assertFalse(existsSync("Solomon", "Y"));
        Assert.assertFalse(findOneSync("Solomon", "Y").isPresent());
        Assert.assertFalse(existsSync("A", "Other"));
        Assert.assertFalse(findOneSync("A", "Other").isPresent());

        Map<String, Dashboard> byId =
            DaoTestFixture.byId(findAllSync(), Dashboard::getId);
        assertEquals(4, byId.size());
        equalsByString(solomonA, byId.get("A"));
        equalsByString(solomonB, byId.get("B"));
        equalsByString(solomonC, byId.get("C"));
        equalsByString(otherX, byId.get("X"));

        Assert.assertTrue(deleteOneSync("Solomon", "A"));
        Assert.assertFalse(deleteOneSync("Solomon", "A"));

        byId = DaoTestFixture.byId(findAllSync(), Dashboard::getId);
        assertEquals(3, byId.size());
        equalsByString(solomonB, byId.get("B"));
        equalsByString(solomonC, byId.get("C"));
        equalsByString(otherX, byId.get("X"));

        join(getDashboardsDao().deleteByProjectId("Solomon", ""));

        List<Dashboard> dashboards = findAllSync();
        assertEquals(1, dashboards.size());
        equalsByString(otherX, dashboards.get(0));
    }

    @Test
    public void insertFindAndDeleteWithFolder() {
        Dashboard.Builder builder = simpleDashboard();
        Dashboard folderA = builder.setId("A").setProjectId("Solomon").setFolderId("folder").build();
        Dashboard folderB = builder.setId("B").setProjectId("Solomon").setFolderId("folder").build();
        Dashboard folderC = builder.setId("C").setProjectId("Solomon").setFolderId("folder").build();
        Dashboard otherX = builder.setId("X").setProjectId("Solomon").setFolderId("other").build();

        Assert.assertTrue(insertSync(folderA));
        Assert.assertTrue(insertSync(folderB));
        Assert.assertTrue(insertSync(folderC));
        Assert.assertTrue(insertSync(otherX));
        Assert.assertFalse(insertSync(folderA));

        Assert.assertTrue(existsSync("Solomon", "folder", "A"));
        Dashboard foundA = findOneSync("Solomon", "folder", "A").orElseThrow(AssertionError::new);
        equalsByString(folderA, foundA);

        Assert.assertFalse(existsSync("Solomon", "folder", "Y"));
        Assert.assertFalse(findOneSync("Solomon", "folder", "Y").isPresent());
        Assert.assertFalse(existsSync("A", "folder", "Other"));
        Assert.assertFalse(findOneSync("A", "folder", "Other").isPresent());

        Map<String, Dashboard> byId =
            DaoTestFixture.byId(findAllSync(), Dashboard::getId);
        assertEquals(4, byId.size());
        equalsByString(folderA, byId.get("A"));
        equalsByString(folderB, byId.get("B"));
        equalsByString(folderC, byId.get("C"));
        equalsByString(otherX, byId.get("X"));

        Assert.assertTrue(deleteOneSync("Solomon", "folder", "A"));
        Assert.assertFalse(deleteOneSync("Solomon", "folder", "A"));

        byId = DaoTestFixture.byId(findAllSync(), Dashboard::getId);
        assertEquals(3, byId.size());
        equalsByString(folderB, byId.get("B"));
        equalsByString(folderC, byId.get("C"));
        equalsByString(otherX, byId.get("X"));

        join(getDashboardsDao().deleteByProjectId("Solomon", "folder"));

        List<Dashboard> dashboards = findAllSync();
        assertEquals(1, dashboards.size());
        equalsByString(otherX, dashboards.get(0));
    }

    private List<Dashboard> findAllSync() {
        return join(getDashboardsDao().findAll());
    }

    private Optional<Dashboard> findOneSync(String projectId, String dashboardId) {
        return join(getDashboardsDao().findOne(projectId, "", dashboardId));
    }

    private Optional<Dashboard> findOneSync(String projectId, String folderId, String dashboardId) {
        return join(getDashboardsDao().findOne(projectId, folderId, dashboardId));
    }

    private boolean deleteOneSync(String projectId, String dashboardId) {
        return join(getDashboardsDao().deleteOne(projectId, "", dashboardId));
    }

    private boolean deleteOneSync(String projectId, String folderId, String dashboardId) {
        return join(getDashboardsDao().deleteOne(projectId, folderId, dashboardId));
    }

    private boolean insertSync(Dashboard dashboard) {
        return join(getDashboardsDao().insert(dashboard));
    }

    private boolean existsSync(String projectId, String dashboardId) {
        return join(getDashboardsDao().exists(projectId, "", dashboardId));
    }

    private boolean existsSync(String projectId, String folderId, String dashboardId) {
        return join(getDashboardsDao().exists(projectId, folderId, dashboardId));
    }

    private PagedResult<Dashboard> findByProjectIdSync(String projectId, PageOptions pageOpts, String text) {
        return join(getDashboardsDao().findByProjectId(projectId, "", pageOpts, text));
    }

    private PagedResult<Dashboard> findByFolderIdSync(String projectId, String folderId, PageOptions pageOpts, String text) {
        return join(getDashboardsDao().findByProjectId(projectId, folderId, pageOpts, text));
    }

    private static Dashboard.Builder simpleDashboard() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return Dashboard
            .newBuilder()
            .setId("Dashboard1")
            .setProjectId("Solomon1")
            .setName("Name1")
            .setDescription("Desc1")
            .setHeightMultiplier(123.456)
            .setParameters(simpleParameters().toArray(new Selector[0]))
            .setRows(simpleRows().toArray(new DashboardRow[0]))
            .setCreatedAt(now)
            .setCreatedBy("user1")
            .setUpdatedAt(now.plus(Duration.ofHours(2)))
            .setUpdatedBy("user2")
            .setVersion(1);
    }

    private static List<DashboardRow> simpleRows() {
        return Arrays.asList(new DashboardRow(simplePanels().toArray(new DashboardPanel[0])));
    }

    private static List<DashboardPanel> simplePanels() {
        return Arrays.asList(new DashboardPanel(DashboardPanel.Type.IFRAME, "Title1", "Subtitle1", "url1", "", 2, 3));
    }

    private static List<Selector> simpleParameters() {
        return Arrays.asList(
            new Selector("Name1", "Value1"),
            new Selector("Name2", "Value2"));
    }

    private static Dashboard makeSimpleDashboard(
        String projectId,
        String folderId,
        String id,
        String name)
    {
        return Dashboard.newBuilder()
            .setId(id)
            .setProjectId(projectId)
            .setFolderId(folderId)
            .setName(name)
            .setUpdatedBy("user")
            .setGeneratedId("mon123456")
            .build();
    }
}
