package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.util.Optional;
import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.FolderMenu;
import ru.yandex.solomon.core.db.model.MenuItem;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.decrement;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.equalsByString;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.increment;


/**
 * @author Oleg Baryshnikov
 */
@YaExternal
public abstract class AbstractFolderMenuDaoTest {

    protected abstract FolderMenuDao getFolderMenuDao();

    @Test
    public void deleteById() {
        FolderMenu folderMenu1 = FolderMenu.newBuilder()
                .setProjectId("solomon")
                .setId("folder1")
                .setItems(new MenuItem[0])
                .setCreatedAt(Instant.EPOCH)
                .setUpdatedAt(Instant.EPOCH)
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .setVersion(0)
                .build();

        FolderMenu folderMenu2 = FolderMenu.newBuilder()
                .setProjectId("solomon")
                .setId("folder2")
                .setItems(new MenuItem[0])
                .setCreatedAt(Instant.EPOCH)
                .setUpdatedAt(Instant.EPOCH)
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .setVersion(0)
                .build();

        assertTrue(upsertSync(folderMenu1).isPresent());
        assertTrue(upsertSync(folderMenu2).isPresent());

        deleteByIdSync("solomon", "folder1");

        Assert.assertFalse(findByIdSync("solomon", "folder1").isPresent());
        assertTrue(findByIdSync("solomon", "folder2").isPresent());
    }

    @Test
    public void updateThenFind() {
        FolderMenu.Builder builder = menu();
        FolderMenu folderMenu = builder.build();
        upsertSync(folderMenu).orElseThrow(AssertionError::new);

        MenuItem[] menuUpdate = { item("Simple", new MenuItem[0]) };
        increment(builder).setItems(menuUpdate).setUpdatedBy("x");
        upsertSync(builder.build());

        FolderMenu foundMenu = findByIdSync(folderMenu.getProjectId(), folderMenu.getId()).orElseThrow(AssertionError::new);
        FolderMenu expectedMenu = increment(builder).build();
        equalsByString(foundMenu, expectedMenu);

        assertFalse(upsertSync(decrement(builder).setUpdatedBy("yyy").build()).isPresent());
        foundMenu = findByIdSync(folderMenu.getProjectId(), folderMenu.getId()).orElseThrow(AssertionError::new);
        equalsByString(foundMenu, expectedMenu);
    }

    @Test
    public void extraDeleteById() {
        FolderMenu.Builder builder = menu();
        FolderMenu menuA = upsertSync(builder.setId("MenuA").build()).orElseThrow(AssertionError::new);
        FolderMenu menuB = upsertSync(builder.setId("MenuB").build()).orElseThrow(AssertionError::new);
        FolderMenu menuC = upsertSync(builder.setId("MenuC").build()).orElseThrow(AssertionError::new);

        deleteByIdSync("", "MenuD");

        findByIdSync(menuA.getProjectId(), menuA.getId()).orElseThrow(AssertionError::new);
        findByIdSync(menuB.getProjectId(), menuB.getId()).orElseThrow(AssertionError::new);
        findByIdSync(menuC.getProjectId(), menuC.getId()).orElseThrow(AssertionError::new);

        deleteByIdSync(menuB.getProjectId(), menuB.getId());
        findByIdSync(menuA.getProjectId(), menuA.getId()).orElseThrow(AssertionError::new);
        assertFalse(findByIdSync(menuB.getProjectId(), menuB.getId()).isPresent());
        findByIdSync(menuC.getProjectId(), menuC.getId()).orElseThrow(AssertionError::new);
    }

    private boolean insertSync(FolderMenu menu) {
        return join(getFolderMenuDao().insert(menu));
    }

    private Optional<FolderMenu> upsertSync(FolderMenu menu) {
        return join(getFolderMenuDao().upsert(menu));
    }

    private Optional<FolderMenu> findByIdSync(String projectId, String id) {
        return join(getFolderMenuDao().findById(projectId, id));
    }

    private void deleteByIdSync(String projectId, String id) {
        join(getFolderMenuDao().deleteById(projectId, id));
    }

    private static FolderMenu.Builder menu() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        String login = "snoop";
        return FolderMenu
                .newBuilder()
                .setProjectId("project")
                .setId("menu1")
                .setItems(menuItems())
                .setCreatedAt(now)
                .setCreatedBy(login)
                .setUpdatedAt(now)
                .setUpdatedBy(login)
                .setVersion(1);
    }

    private static MenuItem[] menuItems() {
        return new MenuItem[]{
                menuItem("A", 0),
                menuItem("B", 1),
                menuItem("C", 2) };
    }

    private static MenuItem menuItem(String title, int childrenCount) {
        MenuItem[] children = IntStream
                .rangeClosed(1, childrenCount)
                .boxed()
                .map(index -> menuItem(title + "." + index, childrenCount - 1))
                .toArray(MenuItem[]::new);
        return item(title, children);
    }

    private static MenuItem item(String title, MenuItem[] children) {
        return new MenuItem(title, "http://localhost/", children, "selectors");
    }
}
