package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.db.model.ShortGraphOrDashboardConf;
import ru.yandex.solomon.core.db.model.graph.AggregationType;
import ru.yandex.solomon.core.db.model.graph.ColorSchemeType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingAggregationType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingFillType;
import ru.yandex.solomon.core.db.model.graph.DownsamplingMode;
import ru.yandex.solomon.core.db.model.graph.ElementTransform;
import ru.yandex.solomon.core.db.model.graph.FilterFunction;
import ru.yandex.solomon.core.db.model.graph.FilterOrder;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.core.db.model.graph.GraphElement;
import ru.yandex.solomon.core.db.model.graph.GraphElementType;
import ru.yandex.solomon.core.db.model.graph.GraphTransform;
import ru.yandex.solomon.core.db.model.graph.InterpolationType;
import ru.yandex.solomon.core.db.model.graph.OverLinesTransform;
import ru.yandex.solomon.core.db.model.graph.ScaleType;
import ru.yandex.solomon.core.db.model.graph.YaxisPosition;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeFalse;
import static org.junit.Assume.assumeTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;


/**
 * @author Oleg Baryshnikov
 */
@YaExternal
public abstract class AbstractGraphsDaoTest {

    protected abstract GraphsDao getGraphsDao();

    @Test
    public void insertOneGraph() {
        Graph graph = makeSimpleGraph("project", "", "graph", "Graph");

        assertTrue(insertSync(graph));

        Optional<Graph> fromDb = findOneSync(graph);
        assertTrue(fromDb.isPresent());
        Assert.assertEquals(graph, fromDb.get());
    }

    @Test
    public void insertOneGraphWithFolder() {
        Graph graph = makeSimpleGraph("project", "folder", "graph", "Graph");

        assertTrue(insertSync(graph));

        Optional<Graph> fromDb = findOneSync(graph);
        assertTrue(fromDb.isPresent());
        Assert.assertEquals(graph, fromDb.get());
    }

    @Test
    public void insertSeveralGraphs() {
        Graph graph = makeSimpleGraph("project", "", "graph", "Graph");
        assertTrue(insertSync(graph));
        assertFalse(insertSync(graph));
        assertTrue(insertSync(makeSimpleGraph("project", "", "graph2", "Graph2")));
    }

    @Test
    public void insertSeveralGraphsWithFolder() {
        Graph graph = makeSimpleGraph("project", "folder", "graph", "Graph");
        assertTrue(insertSync(graph));
        assertFalse(insertSync(graph));
        assertTrue(insertSync(makeSimpleGraph("project", "folder", "graph2", "Graph2")));
    }

    @Test
    public void findByProjectIdWithAllPageSize() {
        List<Graph> graphs = fillGraphCollection();
        List<Graph> exprectedGraphs = graphs.stream()
            .filter(s -> "solomon".equals(s.getProjectId()))
            .collect(Collectors.toList());

        PagedResult<Graph> solomonGraphsFromDb = findByProjectIdSync("solomon", PageOptions.ALL, "");

        assertPartialEquality(exprectedGraphs, solomonGraphsFromDb.getResult());

        Assert.assertEquals(0, solomonGraphsFromDb.getCurrentPage());
        Assert.assertEquals(1, solomonGraphsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonGraphsFromDb.getTotalCount());
    }

    @Test
    public void findByFolderIdWithAllPageSize() {
        List<Graph> graphs = fillGraphCollection();
        List<Graph> exprectedGraphs = graphs.stream()
            .filter(s -> "solomon".equals(s.getFolderId()))
            .collect(Collectors.toList());

        PagedResult<Graph> solomonGraphsFromDb = findByFolderIdSync("solomon", "solomon", PageOptions.ALL, "");

        assertPartialEquality(exprectedGraphs, solomonGraphsFromDb.getResult());

        Assert.assertEquals(0, solomonGraphsFromDb.getCurrentPage());
        Assert.assertEquals(1, solomonGraphsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonGraphsFromDb.getTotalCount());
    }

    @Test
    public void findByProjectIdWithPageSize() {
        List<Graph> graphs = fillGraphCollection();
        List<Graph> expectedGraphs = graphs.stream()
            .filter(s -> "solomon".equals(s.getProjectId()))
            .limit(2)
            .collect(Collectors.toList());

        PagedResult<Graph> solomonGraphsFromDb = findByProjectIdSync("solomon", new PageOptions(2, 0), "");

        assertPartialEquality(expectedGraphs, solomonGraphsFromDb.getResult());

        Assert.assertEquals(0, solomonGraphsFromDb.getCurrentPage());
        Assert.assertEquals(2, solomonGraphsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonGraphsFromDb.getTotalCount());
    }

    @Test
    public void findByFolderIdWithPageSize() {
        List<Graph> graphs = fillGraphCollection();
        List<Graph> expectedGraphs = graphs.stream()
            .filter(s -> "solomon".equals(s.getFolderId()))
            .limit(2)
            .collect(Collectors.toList());

        PagedResult<Graph> solomonGraphsFromDb = findByFolderIdSync("solomon", "solomon", new PageOptions(2, 0), "");

        assertPartialEquality(expectedGraphs, solomonGraphsFromDb.getResult());

        Assert.assertEquals(0, solomonGraphsFromDb.getCurrentPage());
        Assert.assertEquals(2, solomonGraphsFromDb.getPagesCount());
        Assert.assertEquals(4, solomonGraphsFromDb.getTotalCount());
    }

    @Test
    public void findByProjectIdAndText() {
        List<Graph> graphs = Arrays.asList(
            makeSimpleGraph("solomon", "", "d1", "graph 1"),
            makeSimpleGraph("solomon", "", "d2", "Graph 2"),
            makeSimpleGraph("solomon", "", "graph3", "d3"),
            makeSimpleGraph("solomon", "", "d4", "d4")
        );

        for (Graph graph : graphs) {
            assumeTrue(insertSync(graph));
        }

        List<Graph> expectedGraphs = graphs.stream()
            .filter(s -> StringUtils.containsIgnoreCase(s.getId(), "graph")
                    || StringUtils.containsIgnoreCase(s.getName(), "graph"))
            .collect(Collectors.toList());

        PagedResult<Graph> solomonGraphsFromDb = findByProjectIdSync("solomon", PageOptions.ALL, "graph");
        assertPartialEquality(expectedGraphs, solomonGraphsFromDb.getResult());
    }

    @Test
    public void findByFolderIdAndText() {
        List<Graph> graphs = Arrays.asList(
            makeSimpleGraph("solomon", "folder", "d1", "graph 1"),
            makeSimpleGraph("solomon", "folder", "d2", "Graph 2"),
            makeSimpleGraph("solomon", "folder", "graph3", "d3"),
            makeSimpleGraph("solomon", "folder", "d4", "d4")
        );

        for (Graph graph : graphs) {
            assumeTrue(insertSync(graph));
        }

        List<Graph> expectedGraphs = graphs.stream()
            .filter(s -> StringUtils.containsIgnoreCase(s.getId(), "graph")
                    || StringUtils.containsIgnoreCase(s.getName(), "graph"))
            .collect(Collectors.toList());

        PagedResult<Graph> solomonGraphsFromDb = findByFolderIdSync("solomon", "folder", PageOptions.ALL, "graph");
        assertPartialEquality(expectedGraphs, solomonGraphsFromDb.getResult());
    }

    private List<Graph> fillGraphCollection() {
        List<String> projectIds = List.of("kikimr", "yql", "solomon");
        List<Graph> graphs = new ArrayList<>(projectIds.size() * 4);

        for (String projectId : projectIds) {
            graphs.add(makeSimpleGraph(projectId, projectId, projectId + "_d1", "graph 1"));
            graphs.add(makeSimpleGraph(projectId, projectId, projectId + "_d2", "graph 2"));
            graphs.add(makeSimpleGraph(projectId, projectId, projectId + "_d3", "d3"));
            graphs.add(makeSimpleGraph(projectId, projectId, projectId + "_d4", "graph 4"));
        }

        for (Graph graph : graphs) {
            assumeTrue(insertSync(graph));
        }
        return graphs;
    }

    private void assertPartialEquality(List<Graph> expected, List<Graph> actual) {
        Assert.assertEquals(expected.size(), actual.size());

        for (int i = 0; i < expected.size(); ++i) {
            Graph expectedGraph = expected.get(i);
            Graph actualGraph = actual.get(i);
            Assert.assertEquals(expectedGraph.getProjectId(), actualGraph.getProjectId());
            Assert.assertEquals(expectedGraph.getId(), actualGraph.getId());
            Assert.assertEquals(expectedGraph.getName(), actualGraph.getName());
        }
    }

    @Test
    public void partialUpdate() {
        Graph graph = makeSimpleGraph("project", "", "graph", "Graph");

        Graph graphToUpdate = graph.toBuilder()
            .setDescription("description #2")
            .setParameters(new Selector[]{ new Selector("label3", "*") })
            .setElements(new GraphElement[]{
                new GraphElement(
                    "title", GraphElementType.EXPRESSION, null,
                    "{label3=value3}", "", true, "stack", true, "red",
                    YaxisPosition.RIGHT, ElementTransform.INTEGRATE
                ),
            })
            .setMin("")
            .setMax("")
            .setNormalize(true)
            .setColorScheme(ColorSchemeType.GRADIENT)
            .setDropNans(true)
            .setStack(true)
            .setAggr(AggregationType.SUM)
            .setInterpolate(InterpolationType.NONE)
            .setOverLinesTransform(OverLinesTransform.PERCENTILE)
            .setScale(ScaleType.LOG)
            .setNumberFormat("4_M")
            .setGreen("10")
            .setYellow("20")
            .setRed("30")
            .setViolet("40")
            .setGreenValue("10")
            .setYellowValue("20")
            .setRedValue("30")
            .setVioletValue("40")
            .setSortByLabel("label")
            .setAsc(true)
            .setLimit("100")
            .setPercentiles("50,90,99")
            .setBucketLabel("bucket")
            .setFilter(FilterOrder.TOP)
            .setFilterBy(FilterFunction.MAX)
            .setFilterLimit("10")
            .setTransform(GraphTransform.MOVING_PERCENTILE)
            .setMovingWindow("10m")
            .setMovingPercentile("90")
            .setDownsampling(DownsamplingMode.BY_INTERVAL)
            .setDownsamplingAggr(DownsamplingAggregationType.LAST)
            .setDownsamplingFill(DownsamplingFillType.NONE)
            .setIgnoreMinStepMillis(true)
            .setGrid("1m")
            .setMaxPoints(100)
            .setThreshold(15d)
            .setGeneratedId("mon123457")
            .setUpdatedAt(Instant.parse("2017-10-10T07:57:00Z"))
            .setCreatedBy("user")
            .setUpdatedBy("user_2")
            .setVersion(0)
            .build();

        assumeTrue(insertSync(graph));

        Optional<Graph> updatedGraphOpt = join(getGraphsDao().partialUpdate(graphToUpdate));
        assertTrue(updatedGraphOpt.isPresent());

        Graph updatedGraph = updatedGraphOpt.get();
        Graph expectedGraph = graphToUpdate.toBuilder()
            .setVersion(1)
            .setGeneratedId(graph.getGeneratedId())
            .build();
        Assert.assertEquals(expectedGraph, updatedGraph);
    }

    @Test
    public void deleteOne() {
        Graph graph = makeSimpleGraph("project", "", "graph", "Graph");

        assumeTrue(insertSync(graph));

        assertTrue(deleteOneSync(graph.getProjectId(), graph.getId()));
        assumeFalse(findOneSync(graph).isPresent());

        assertFalse(deleteOneSync("project", "graph_2"));
        assertFalse(deleteOneSync("project_2", "graph"));

        graph = makeSimpleGraph("project", "", "graph", "Graph");
        assumeTrue(insertSync(graph));
        assertFalse(deleteOneSync("project", "xyz"));
        Graph found = findOneSync(graph).orElseThrow(AssertionError::new);
        assertTrue(EqualsBuilder.reflectionEquals(graph, found));
        assertTrue(deleteOneSync(graph.getProjectId(), graph.getId()));
        assumeFalse(findOneSync(graph).isPresent());
    }

    @Test
    public void deleteOneWithFolder() {
        Graph graph = makeSimpleGraph("project", "folder", "graph", "Graph");

        assumeTrue(insertSync(graph));

        assertTrue(deleteOneSync(graph.getProjectId(), graph.getFolderId(), graph.getId()));
        assumeFalse(findOneSync(graph).isPresent());

        assertFalse(deleteOneSync("project", "folder", "graph_2"));
        assertFalse(deleteOneSync("project_2", "folder_2",  "graph"));

        graph = makeSimpleGraph("project", "folder", "graph", "Graph");
        assumeTrue(insertSync(graph));
        assertFalse(deleteOneSync("project", "folder", "xyz"));
        Graph found = findOneSync(graph).orElseThrow(AssertionError::new);
        assertTrue(EqualsBuilder.reflectionEquals(graph, found));
        assertTrue(deleteOneSync(graph.getProjectId(), graph.getId()));
        assumeFalse(findOneSync(graph).isPresent());
    }

    @Test
    public void deleteByProject() {
        Graph graphA = makeSimpleGraph("project1", "", "graphA", "graphA");
        Graph graphB = makeSimpleGraph("project1", "", "graphB", "graphB");
        Graph graphX = makeSimpleGraph("project2", "", "graphX", "graphX");

        assumeTrue(insertSync(graphA));
        assumeTrue(insertSync(graphB));
        assumeTrue(insertSync(graphX));

        join(getGraphsDao().deleteByProjectId("project1", ""));

        assertFalse(existsSync(graphA.getProjectId(), graphA.getId()));
        assertFalse(existsSync(graphB.getProjectId(), graphB.getId()));
        assertTrue(existsSync(graphX.getProjectId(), graphX.getId()));
    }

    @Test
    public void deleteByFolder() {
        Graph graphA = makeSimpleGraph("project1", "folder1", "graphA", "graphA");
        Graph graphB = makeSimpleGraph("project1", "folder1", "graphB", "graphB");
        Graph graphX = makeSimpleGraph("project2", "folder2", "graphX", "graphX");

        assumeTrue(insertSync(graphA));
        assumeTrue(insertSync(graphB));
        assumeTrue(insertSync(graphX));

        join(getGraphsDao().deleteByProjectId("project1", "folder1"));

        assertFalse(existsSync(graphA.getProjectId(), graphA.getId()));
        assertFalse(existsSync(graphB.getProjectId(), graphB.getId()));
        assertTrue(existsSync(graphX.getProjectId(), graphX.getId()));
    }

    @Test
    public void exists() {
        Graph graph = makeSimpleGraph("project", "", "graph", "Graph");

        assumeTrue(insertSync(graph));

        assertTrue(existsSync("project", "graph"));
        assertFalse(existsSync("project", "graph2"));
        assertTrue(existsSync(graph.getProjectId(), graph.getId()));
        assertFalse(existsSync("project", "graph2"));
    }

    @Test
    public void existsWithFolder() {
        Graph graph = makeSimpleGraph("project", "folder", "graph", "Graph");

        assumeTrue(insertSync(graph));

        assertTrue(existsSync("project", "folder", "graph"));
        assertFalse(existsSync("project", "folder", "graph2"));
        assertTrue(existsSync(graph.getProjectId(), graph.getFolderId(), graph.getId()));
        assertFalse(existsSync("project", "folder", "graph2"));
    }

    @Test
    public void findAllShorted() {
        List<Graph> graphs = Arrays.asList(
            Graph.newBuilder()
                .setId("graph1")
                .setProjectId("junk")
                .setName("Junk graph #1")
                .setParameters(new Selector[]{ new Selector("host", "cluster") })
                .setGeneratedId("mon123451")
                .build(),
            Graph.newBuilder()
                .setId("graph2")
                .setProjectId("other_junk")
                .setName("Other junk graph #2")
                .setGeneratedId("mon123452")
                .build()
        );

        List<ShortGraphOrDashboardConf> expectedShortConfs =
            graphs.stream()
                .filter(x -> "junk".equals(x.getProjectId()))
                .map(AbstractGraphsDaoTest::toShortGraphConf)
                .collect(Collectors.toList());

        for (Graph graph : graphs) {
            assertTrue(insertSync(graph));
        }

        List<ShortGraphOrDashboardConf> shortConfs = join(getGraphsDao().findByProjectIdShorted("junk", ""));

        assertFalse(shortConfs.isEmpty());
        Assert.assertEquals(expectedShortConfs, shortConfs);
    }

    private Optional<Graph> findOneSync(Graph graph) {
        return join(getGraphsDao().findOne(graph.getProjectId(), graph.getFolderId(), graph.getId()));
    }

    private boolean insertSync(Graph graph) {
        return join(getGraphsDao().insert(graph));
    }

    private boolean deleteOneSync(String projectId, String graphId) {
        return join(getGraphsDao().deleteOne(projectId, "", graphId));
    }

    private boolean deleteOneSync(String projectId, String folderId, String graphId) {
        return join(getGraphsDao().deleteOne(projectId, folderId, graphId));
    }

    private PagedResult<Graph> findByProjectIdSync(String projectId, PageOptions pageOpts, String text) {
        return join(getGraphsDao().findByProjectId(projectId, "", pageOpts, text));
    }

    private PagedResult<Graph> findByFolderIdSync(String projectId, String folderId, PageOptions pageOpts, String text) {
        return join(getGraphsDao().findByProjectId(projectId, folderId, pageOpts, text));
    }

    private boolean existsSync(String projectId, String graphId) {
        return join(getGraphsDao().exists(projectId, "", graphId));
    }

    private boolean existsSync(String projectId, String folderId, String graphId) {
        return join(getGraphsDao().exists(projectId, folderId, graphId));
    }

    private static ShortGraphOrDashboardConf toShortGraphConf(Graph graph) {
        return new ShortGraphOrDashboardConf(
            graph.getId(),
            graph.getProjectId(),
            graph.getName(),
            graph.getParameters()
        );
    }

    private static Graph makeSimpleGraph(
        String projectId,
        String folderId,
        String id,
        String name)
    {
        return Graph.newBuilder()
            .setId(id)
            .setProjectId(projectId)
            .setFolderId(folderId)
            .setName(name)
            .setDescription("description")
            .setParameters(new Selector[]{ new Selector("host", "*") })
            .setElements(new GraphElement[]{
                new GraphElement(
                    "title",
                    GraphElementType.SELECTORS,
                    new Selector[]{
                        new Selector("label1", "value1"),
                        new Selector("label2", "value2"),
                    },
                    "{label1=value1, label2=value2}",
                    "",
                    true,
                    "stack",
                    false,
                    "red",
                    YaxisPosition.LEFT,
                    ElementTransform.NONE
                ),
            })
            .setGeneratedId("mon123456")
            .setCreatedBy("user")
            .setUpdatedBy("user")
            .build();
    }
}
