package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.util.Optional;
import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.MenuItem;
import ru.yandex.solomon.core.db.model.ProjectMenu;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.decrement;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.equalsByString;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.increment;


/**
 * @author Oleg Baryshnikov
 */
@YaExternal
public abstract class AbstractProjectMenuDaoTest {

    protected abstract ProjectMenuDao getProjectMenuDao();

    @Test
    public void insert() {
        MenuItem[] menuItems = new MenuItem[]{
            new MenuItem("Item 1", "/?project=solomon", null, null),
            new MenuItem(
                "Item 2",
                null,
                new MenuItem[]{
                    new MenuItem("Subitem 1", "/?host=cluster", null, null),
                    new MenuItem("Subitem 2", "/?host={{host}}", null, "host={{host}}"),
                    new MenuItem("Subitem 3", "/?path={{path}}", null, null)
                },
                null
            ),
        };

        ProjectMenu projectMenu = new ProjectMenu(
            "solomon",
            menuItems,
            0,
            Instant.EPOCH,
            Instant.EPOCH,
            "user",
            "user"
        );

        Optional<ProjectMenu> projectMenuUpserted0 = upsertSync(projectMenu);
        assertTrue(projectMenuUpserted0.isPresent());

        ProjectMenu projectMenuUpserted = projectMenuUpserted0.get();
        assertEquals("solomon", projectMenuUpserted.getId());
        assertArrayEquals(menuItems, projectMenuUpserted.getItems());
        assertEquals(1, projectMenuUpserted.getVersion());
        assertEquals(Instant.EPOCH, projectMenuUpserted.getUpdatedAt());
        assertEquals("user", projectMenuUpserted.getUpdatedBy());

        assertFalse(insertSync(projectMenu));

        projectMenu = new ProjectMenu(
            "other",
            menuItems,
            1,
            Instant.EPOCH,
            Instant.EPOCH,
            "user",
            "user"
        );
        assertTrue(insertSync(projectMenu));
        ProjectMenu found = findByIdSync(projectMenu.getId()).orElseThrow(AssertionError::new);
        DaoTestFixture.equalsByString(projectMenu, found);
    }

    @Test
    public void deleteById() {
        ProjectMenu projectMenu1 = new ProjectMenu(
            "solomon",
            new MenuItem[0],
            0,
            Instant.EPOCH,
            Instant.EPOCH,
            "user",
            "user"
        );

        ProjectMenu projectMenu2 = new ProjectMenu(
            "junk",
            new MenuItem[0],
            0,
            Instant.EPOCH,
            Instant.EPOCH,
            "user",
            "user"
        );

        assertTrue(upsertSync(projectMenu1).isPresent());
        assertTrue(upsertSync(projectMenu2).isPresent());

        deleteByIdSync("solomon");

        Assert.assertFalse(findByIdSync("solomon").isPresent());
        assertTrue(findByIdSync("junk").isPresent());
    }

    @Test
    public void insertThenFind() {
        ProjectMenu.Builder builder = menu();
        ProjectMenu projectMenu = builder.build();
        Optional<ProjectMenu> newProjectMenu = upsertSync(projectMenu);
        ProjectMenu actualMenu = newProjectMenu.orElseThrow(AssertionError::new);

        increment(builder);
        ProjectMenu expectedMenu = builder.build();
        equalsByString(actualMenu, expectedMenu);

        ProjectMenu foundMenu = findByIdSync(builder.getId()).orElseThrow(AssertionError::new);
        equalsByString(foundMenu, expectedMenu);
    }

    @Test
    public void updateThenFind() {
        ProjectMenu.Builder builder = menu();
        ProjectMenu projectMenu = builder.build();
        upsertSync(projectMenu).orElseThrow(AssertionError::new);

        MenuItem[] menuUpdate = { item("Simple", new MenuItem[0]) };
        increment(builder).setItems(menuUpdate).setUpdatedBy("x");
        upsertSync(builder.build());

        ProjectMenu foundMenu = findByIdSync(builder.getId()).orElseThrow(AssertionError::new);
        ProjectMenu expectedMenu = increment(builder).build();
        equalsByString(foundMenu, expectedMenu);

        assertFalse(upsertSync(decrement(builder).setUpdatedBy("yyy").build()).isPresent());
        foundMenu = findByIdSync(builder.getId()).orElseThrow(AssertionError::new);
        equalsByString(foundMenu, expectedMenu);
    }

    @Test
    public void extraDeleteById() {
        ProjectMenu.Builder builder = menu();
        ProjectMenu menuA = upsertSync(builder.setId("MenuA").build()).orElseThrow(AssertionError::new);
        ProjectMenu menuB = upsertSync(builder.setId("MenuB").build()).orElseThrow(AssertionError::new);
        ProjectMenu menuC = upsertSync(builder.setId("MenuC").build()).orElseThrow(AssertionError::new);

        deleteByIdSync("MenuD");

        findByIdSync(menuA.getId()).orElseThrow(AssertionError::new);
        findByIdSync(menuB.getId()).orElseThrow(AssertionError::new);
        findByIdSync(menuC.getId()).orElseThrow(AssertionError::new);

        deleteByIdSync(menuB.getId());
        findByIdSync(menuA.getId()).orElseThrow(AssertionError::new);
        assertFalse(findByIdSync(menuB.getId()).isPresent());
        findByIdSync(menuC.getId()).orElseThrow(AssertionError::new);
    }

    private boolean insertSync(ProjectMenu projectMenu) {
        return join(getProjectMenuDao().insert(projectMenu));
    }

    private Optional<ProjectMenu> upsertSync(ProjectMenu projectMenu1) {
        return join(getProjectMenuDao().upsert(projectMenu1));
    }

    private Optional<ProjectMenu> findByIdSync(String id) {
        return join(getProjectMenuDao().findById(id));
    }

    private void deleteByIdSync(String id) {
        join(getProjectMenuDao().deleteById(id));
    }

    private static ProjectMenu.Builder menu() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        String login = "snoop";
        return ProjectMenu
            .newBuilder()
            .setId("menu1")
            .setItems(menuItems())
            .setCreatedAt(now)
            .setCreatedBy(login)
            .setUpdatedAt(now)
            .setUpdatedBy(login)
            .setVersion(1);
    }

    private static MenuItem[] menuItems() {
        return new MenuItem[]{
            menuItem("A", 0),
            menuItem("B", 1),
            menuItem("C", 2) };
    }

    private static MenuItem menuItem(String title, int childrenCount) {
        MenuItem[] children = IntStream
            .rangeClosed(1, childrenCount)
            .boxed()
            .map(index -> menuItem(title + "." + index, childrenCount - 1))
            .toArray(MenuItem[]::new);
        return item(title, children);
    }

    private static MenuItem item(String title, MenuItem[] children) {
        return new MenuItem(title, "http://localhost/", children, "selectors");
    }
}
