package ru.yandex.solomon.core.db.dao;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.ProjectSettings;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public abstract class AbstractProjectSettingsDaoTest {

    @Test
    public void findAll() {
        List<ProjectSettings> projectSettingsList = new ArrayList<>();

        for (int i = 0; i < 5000; ++i) {
            ProjectSettings projectSettings = new ProjectSettings("project_" + i, Map.of());
            projectSettingsList.add(projectSettings);
        }

        upsertAllSync(projectSettingsList);

        List<ProjectSettings> foundProjectSettingsList = findAllSync();

        foundProjectSettingsList.sort(Comparator.comparing(ProjectSettings::getId, TextWithNumbersComparator.instance));

        assertEquals(projectSettingsList, foundProjectSettingsList);
    }

    @Test
    public void findKnownProject() {
        ProjectSettings projectSettings = new ProjectSettings("solomon", Map.of("option1", "value1"));

        upsertSync(projectSettings);

        ProjectSettings result = findSync("solomon");
        assertEquals(projectSettings, result);
    }

    @Test
    public void findUnknownProject() {
        ProjectSettings projectSettings = new ProjectSettings("solomon", Map.of("option1", "value1"));

        upsertSync(projectSettings);

        ProjectSettings result = findSync("other_project");
        assertEquals(new ProjectSettings("other_project", Map.of()), result);
    }

    @Test
    public void upsertNewProjectSettings() {
        ProjectSettings projectSettings = new ProjectSettings("solomon", Map.of("option1", "value1"));
        upsertSync(projectSettings);
        assertEquals(projectSettings, findSync("solomon"));
    }

    @Test
    public void upsertExistingProjectSettings() {
        upsertSync(new ProjectSettings("solomon", Map.of()));

        ProjectSettings projectSettings = new ProjectSettings("solomon", Map.of("option1", "value1"));

        upsertSync(projectSettings);

        assertEquals(projectSettings, findSync("solomon"));
    }

    @Test
    public void deleteKnownProject() {
        upsertSync(new ProjectSettings("solomon", Map.of()));
        assertTrue(deleteSync("solomon"));
        assertTrue(findAllSync().isEmpty());
    }

    @Test
    public void deleteUnknownProject() {
        upsertSync(new ProjectSettings("solomon", Map.of()));
        assertFalse(deleteSync("other_project"));
    }

    protected abstract ProjectSettingsDao getProjectSettingsDao();

    private ProjectSettings findSync(String id) {
        return join(getProjectSettingsDao().find(id));
    }

    private ProjectSettings upsertSync(ProjectSettings projectSettings) {
        return join(getProjectSettingsDao().upsert(projectSettings));
    }

    private void upsertAllSync(List<ProjectSettings> projectSettingsList) {
        for (List<ProjectSettings> batch : Lists.partition(projectSettingsList, 50)) {
            join(upsertBatch(batch));
        }
    }

    private CompletableFuture<Void> upsertBatch(List<ProjectSettings> batch) {
        ProjectSettingsDao dao = getProjectSettingsDao();
        final int size = batch.size();
        final CompletableFuture<?>[] futures = new CompletableFuture<?>[size];
        for (int i = 0; i < size; i++) {
            futures[i] = dao.upsert(batch.get(i));
        }
        return CompletableFuture.allOf(futures);
    }

    private List<ProjectSettings> findAllSync() {
        return join(getProjectSettingsDao().findAll());
    }

    private boolean deleteSync(String id) {
        return join(getProjectSettingsDao().delete(id));
    }
}
