package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;


/**
 * @author Sergey Polovko
 */
@YaExternal
public abstract class AbstractProjectsDaoTest {

    public abstract ProjectsDao getProjectsDao();

    @Test
    public void insert() {
        String user = "jamel";
        Project project = newProject("solomon", user, Acl.empty());
        Assert.assertTrue(insertSync(project));

        Optional<Project> fromDb = join(getProjectsDao().findById(project.getId()));
        Assert.assertTrue(fromDb.isPresent());
        Assert.assertEquals(project, fromDb.get());
    }

    @Test
    public void findAll() {
        String user = "jamel";
        Set<String> expectedIds = Set.of("kikimr", "yql", "solomon");

        for (String projectId : expectedIds) {
            Assert.assertTrue(insertSync(newProject(projectId, user, Acl.empty())));
        }

        Set<String> idsFromDb = join(getProjectsDao().findAllNames()).stream()
            .map(Project::getId)
            .collect(Collectors.toSet());

        Assert.assertEquals(expectedIds, idsFromDb);
    }

    @Test
    public void findAllAcls() {
        String user1 = "user1";
        String user2 = "user2";
        String user3 = "user3";

        List<Project> projects = Arrays.asList(
            newProject("p1", user1, Acl.empty()),
            newProject("p2", user1, Acl.empty()),
            newProject("p3", user1, Acl.of(singleton(user2), emptySet(), emptySet(), emptySet())),
            newProject("p4", user1, Acl.of(emptySet(), singleton(user2), emptySet(), emptySet())),
            newProject("p5", user1, Acl.of(emptySet(), emptySet(), singleton(user2), emptySet())),
            newProject("p6", user1, Acl.of(emptySet(), emptySet(), emptySet(), emptySet())),
            newProject("p7", user2, Acl.empty()),
            newProject("p8", user2, Acl.empty()),
            newProject("p9", user2, Acl.of(emptySet(), emptySet(), emptySet(), emptySet())));
            newProject("p10", user2, Acl.of(emptySet(), emptySet(), emptySet(), singleton(user3)));

        for (Project project : projects) {
            Assert.assertTrue(insertSync(project));
        }

        List<Project> projectsFromDb = join(getProjectsDao().findAllNames());

        // owners
        {
            Map<String, String> expectedOwners = projects.stream()
                .collect(Collectors.toMap(Project::getId, Project::getOwner));
            Map<String, String> owners = projectsFromDb.stream()
                .collect(Collectors.toMap(Project::getId, Project::getOwner));
            Assert.assertEquals(expectedOwners, owners);
        }

        // acls
        {
            Map<String, Acl> expectedAcls = projects.stream()
                .collect(Collectors.toMap(Project::getId, Project::getAcl));
            Map<String, Acl> acls = projectsFromDb.stream()
                .collect(Collectors.toMap(Project::getId, Project::getAcl));
            Assert.assertEquals(expectedAcls, acls);
        }
    }

    @Test
    public void pagedFind() {
        insertSync(simpleProject("project1").build());
        insertSync(simpleProject("project2").build());
        insertSync(simpleProject("project3").build());
        insertSync(simpleProject("project4").build());
        insertSync(simpleProject("project5").build());

        PagedResult<Project> result = findSync("", "", new PageOptions(2, 0));

        assertEquals(5, result.getTotalCount());
        assertEquals(0, result.getCurrentPage());
        assertEquals(3, result.getPagesCount());
        assertEquals(2, result.getPageSize());
        assertEquals(2, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());
    }

    @Test
    public void findByAbc() {
        insertSync(simpleProject("project1").setAbcService("abc").build());
        insertSync(simpleProject("project2").setAbcService("abc").build());
        insertSync(simpleProject("project3").setAbcService("abcd").build());

        PagedResult<Project> result = findSync("", "", new PageOptions(10, 0));

        assertEquals(3, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());
        assertEquals("project3", result.getResult().get(2).getId());

        result = findSync("", "abc", new PageOptions(10, 0));

        assertEquals(2, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());

        result = findSync("", "abce", new PageOptions(10, 0));

        assertEquals(0, result.getResult().size());
    }

    @Test
    public void pagedFindV3() {
        insertSync(simpleProject("project1").build());
        insertSync(simpleProject("project2").build());
        insertSync(simpleProject("project3").build());
        insertSync(simpleProject("project4").build());
        insertSync(simpleProject("project5").build());

        TokenBasePage<Project> firstPage = findV3Sync("", 3, "");

        assertEquals(3, firstPage.getItems().size());
        assertEquals("3", firstPage.getNextPageToken());
        assertEquals("project1", firstPage.getItems().get(0).getId());
        assertEquals("project2", firstPage.getItems().get(1).getId());
        assertEquals("project3", firstPage.getItems().get(2).getId());

        TokenBasePage<Project> nextPage = findV3Sync("", 3, firstPage.getNextPageToken());
        assertEquals(2, nextPage.getItems().size());
        assertEquals("", nextPage.getNextPageToken());
        assertEquals("project4", nextPage.getItems().get(0).getId());
        assertEquals("project5", nextPage.getItems().get(1).getId());
    }

    @Test
    public void pagedFindAll() {
        insertSync(simpleProject("project1").build());
        insertSync(simpleProject("project2").build());
        insertSync(simpleProject("project3").build());

        PagedResult<Project> result = findSync("", "", PageOptions.ALL);

        assertEquals(3, result.getTotalCount());
        assertEquals(0, result.getCurrentPage());
        assertEquals(1, result.getPagesCount());
        assertEquals(-1, result.getPageSize());
        assertEquals(3, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());
        assertEquals("project3", result.getResult().get(2).getId());
    }

    @Test
    public void findAllAbc() {
        insertSync(simpleProject("project1").setAbcService("abc").build());
        insertSync(simpleProject("project2").setAbcService("abc").build());
        insertSync(simpleProject("project3").setAbcService("abcd").build());

        PagedResult<Project> result = findSync("", "", PageOptions.ALL);

        assertEquals(3, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());
        assertEquals("project3", result.getResult().get(2).getId());

        result = findSync("", "abc", PageOptions.ALL);

        assertEquals(2, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());

        result = findSync("", "abce", PageOptions.ALL);

        assertEquals(0, result.getResult().size());
    }

    @Test
    public void pagedFindByText() {
        insertSync(simpleProject("project1").build());
        insertSync(simpleProject("project2").build());
        insertSync(simpleProject("project3").build());

        PagedResult<Project> result = findSync("2", "", PageOptions.ALL);

        assertEquals(1, result.getResult().size());
        assertEquals("project2", result.getResult().get(0).getId());
    }

    @Test
    public void pagedFindByPrefixText() {
        insertSync(simpleProject("solomon").build());
        insertSync(simpleProject("project1").setName("solomon1").build());
        insertSync(simpleProject("project2").setName("solomon2").build());
        insertSync(simpleProject("yabs_solo").build());
        insertSync(simpleProject("arcadia").build());
        insertSync(simpleProject("kikimr").build());
        insertSync(simpleProject("yt").build());

        PagedResult<Project> result = findSync("so", "", new PageOptions(10, 0));

        assertEquals(4, result.getResult().size());
        assertEquals("solomon", result.getResult().get(0).getId());
        assertEquals("project1", result.getResult().get(1).getId());
        assertEquals("project2", result.getResult().get(2).getId());
        assertEquals("yabs_solo", result.getResult().get(3).getId());
    }

    @Test
    public void pagedFindByAcl() {
        insertSync(simpleProject("project1")
                .setAcl(Acl.of(Set.of("user"), Set.of("user"), Set.of("user"), Set.of("user")))
                .build());
        insertSync(simpleProject("project2")
                .setAcl(Acl.of(Set.of("user2"), Set.of("user2"), Set.of("user2"), Set.of("user2")))
                .build());
        insertSync(simpleProject("project3").build());

        PagedResult<Project> result = findSync("", "", "user", EnumSet.of(ProjectPermission.CONFIG_UPDATE), PageOptions.ALL);

        assertEquals(1, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
    }

    @Test
    public void pagedFindWithoutAcl() {
        insertSync(simpleProject("project1")
                .setOnlyAuthRead(true)
                .setAcl(Acl.of(Set.of("user"), Set.of("user"), Set.of("user"), Set.of("user")))
                .build());
        insertSync(simpleProject("project2")
                .setOnlyAuthRead(true)
                .setAcl(Acl.of(Set.of("user2"), Set.of("user2"), Set.of("user2"), Set.of("user2")))
                .build());
        insertSync(simpleProject("project3").build());

        {
            // Search without pagination limits
            PagedResult<Project> result = findSync("", "", "user", EnumSet.noneOf(ProjectPermission.class), PageOptions.ALL);
            assertEquals(3, result.getResult().size());
            assertEquals("project1", result.getResult().get(0).getId());
            assertEquals("project2", result.getResult().get(1).getId());
            assertEquals("project3", result.getResult().get(2).getId());
        }

        {
            // Search with pagination limits
            PagedResult<Project> result = findSync("", "", "user", EnumSet.noneOf(ProjectPermission.class), new PageOptions(3, 0));
            assertEquals(3, result.getResult().size());
            assertEquals("project1", result.getResult().get(0).getId());
            assertEquals("project2", result.getResult().get(1).getId());
            assertEquals("project3", result.getResult().get(2).getId());
        }
    }

    @Test
    public void findInProjects() {
        insertSync(simpleProject("project1").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project2").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project3").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project33").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project4").setAcl(Acl.empty()).build());

        {
            // Search without pagination limits
            PagedResult<Project> result = join(getProjectsDao().findInProjects("", "", Set.of("project2", "project3", "project33"), PageOptions.ALL));
            assertEquals(3, result.getResult().size());
            assertEquals("project2", result.getResult().get(0).getId());
            assertEquals("project3", result.getResult().get(1).getId());
            assertEquals("project33", result.getResult().get(2).getId());

            result = join(getProjectsDao().findInProjects("", "", Set.of("project3", "project2", "project33", "project4"), PageOptions.ALL));
            assertEquals(4, result.getResult().size());
            assertEquals("project2", result.getResult().get(0).getId());
            assertEquals("project3", result.getResult().get(1).getId());
            assertEquals("project4", result.getResult().get(2).getId());
            assertEquals("project33", result.getResult().get(3).getId());

            result = join(getProjectsDao().findInProjects("", "", Set.of(), PageOptions.ALL));
            assertEquals(0, result.getResult().size());
        }

        {
            // Search with pagination limits
            PagedResult<Project> result = join(getProjectsDao().findInProjects("", "", Set.of("project33", "project2", "project3"), new PageOptions(2, 0)));
            assertEquals(2, result.getResult().size());
            assertEquals("project2", result.getResult().get(0).getId());
            assertEquals("project3", result.getResult().get(1).getId());

            result = join(getProjectsDao().findInProjects("", "", Set.of("project33", "project2", "project3"), new PageOptions(2, 1)));
            assertEquals(1, result.getResult().size());
            assertEquals("project33", result.getResult().get(0).getId());
        }
    }

    @Test
    public void findInProjects_withText() {
        insertSync(simpleProject("project1").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project2").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project22").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project3").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project33").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project333").setAcl(Acl.empty()).build());
        insertSync(simpleProject("project4").setAcl(Acl.empty()).build());

        {
            // Search without pagination limits
            PagedResult<Project> result = join(getProjectsDao().findInProjects("3", "", Set.of("project2", "project3", "project33"), PageOptions.ALL));
            assertEquals(2, result.getResult().size());
            assertEquals("project3", result.getResult().get(0).getId());
            assertEquals("project33", result.getResult().get(1).getId());

            result = join(getProjectsDao().findInProjects("2", "", Set.of("project3", "project2", "project33", "project22"), PageOptions.ALL));
            assertEquals(2, result.getResult().size());
            assertEquals("project2", result.getResult().get(0).getId());
            assertEquals("project22", result.getResult().get(1).getId());

            result = join(getProjectsDao().findInProjects("2", "", Set.of(), PageOptions.ALL));
            assertEquals(0, result.getResult().size());
        }

        {
            // Search with pagination limits
            PagedResult<Project> result = join(getProjectsDao().findInProjects("3", "", Set.of("project2", "project33", "project333", "project3"), new PageOptions(2, 0)));
            assertEquals(2, result.getResult().size());
            assertEquals("project3", result.getResult().get(0).getId());
            assertEquals("project33", result.getResult().get(1).getId());

            result = join(getProjectsDao().findInProjects("3", "", Set.of("project2", "project33", "project333", "project3"), new PageOptions(2, 1)));
            assertEquals(1, result.getResult().size());
            assertEquals("project333", result.getResult().get(0).getId());
        }
    }

    @Test
    public void findByAbcInProjects() {
        insertSync(simpleProject("project1").setAbcService("abc").build());
        insertSync(simpleProject("project2").setAbcService("abc").build());
        insertSync(simpleProject("project3").setAbcService("abcd").build());
        insertSync(simpleProject("project4").setAbcService("abce").build());

        PagedResult<Project> result = join(getProjectsDao().findInProjects("", "", Set.of("project1", "project2", "project3"), new PageOptions(10, 0)));

        assertEquals(3, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());
        assertEquals("project3", result.getResult().get(2).getId());

        result = join(getProjectsDao().findInProjects("", "abc", Set.of("project1", "project2", "project3"), new PageOptions(10, 0)));

        assertEquals(2, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());

        result = join(getProjectsDao().findInProjects("", "abce", Set.of("project1", "project2", "project3"), new PageOptions(10, 0)));

        assertEquals(0, result.getResult().size());

        result = join(getProjectsDao().findInProjects("", "abce", Set.of("project1", "project2", "project3", "project4"), new PageOptions(10, 0)));

        assertEquals(1, result.getResult().size());
        assertEquals("project4", result.getResult().get(0).getId());
    }

    @Test
    public void findByAbcInProjectsAll() {
        insertSync(simpleProject("project1").setAbcService("abc").build());
        insertSync(simpleProject("project2").setAbcService("abc").build());
        insertSync(simpleProject("project3").setAbcService("abcd").build());
        insertSync(simpleProject("project4").setAbcService("abce").build());

        PagedResult<Project> result = join(getProjectsDao().findInProjects("", "", Set.of("project1", "project2", "project3"), PageOptions.ALL));

        assertEquals(3, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());
        assertEquals("project3", result.getResult().get(2).getId());

        result = join(getProjectsDao().findInProjects("", "abc", Set.of("project1", "project2", "project3"), PageOptions.ALL));

        assertEquals(2, result.getResult().size());
        assertEquals("project1", result.getResult().get(0).getId());
        assertEquals("project2", result.getResult().get(1).getId());

        result = join(getProjectsDao().findInProjects("", "abce", Set.of("project1", "project2", "project3"), PageOptions.ALL));

        assertEquals(0, result.getResult().size());

        result = join(getProjectsDao().findInProjects("", "abce", Set.of("project1", "project2", "project3", "project4"), PageOptions.ALL));

        assertEquals(1, result.getResult().size());
        assertEquals("project4", result.getResult().get(0).getId());
    }

    @Test
    public void partialUpdate() {
        Instant now = Instant.now();
        String user1 = "user1";
        String user2 = "user2";

        Project p1 = newProject("p1", user1, Acl.empty());
        Assert.assertTrue(insertSync(p1));
        Project p2 = newProject("p2", user1, Acl.empty());
        Assert.assertTrue(insertSync(p2));
        Project p3 = newProject("p3", user1, Acl.empty());
        Assert.assertTrue(insertSync(p3));

        Acl acl = Acl.of(singleton("user3"), singleton("user4"), singleton("user5"), singleton("user6"));

        {
            // Cannot change owner and internal options
            Project p1New = new Project(
                "p1", "P1: updated", "description", user2, acl, "p1: updated", false, false, false, false, false, "sensor",
                now, now, user2, user2, 0, Map.of("label1", "value1", "label2", "value2")
            );
            Project p1Updated = partialUpdateSync(p1New, false, false, true)
                .orElseThrow(() -> new AssertionError("project not found"));
            Assert.assertEquals(p1.getId(), p1Updated.getId());
            Assert.assertEquals("P1: updated", p1Updated.getName());
            Assert.assertEquals(p1.getOwner(), p1Updated.getOwner());
            Assert.assertEquals(acl, p1Updated.getAcl());
            Assert.assertEquals("p1: updated", p1Updated.getAbcService());
            Assert.assertEquals(p1New.isOnlyAuthRead(), p1Updated.isOnlyAuthRead());
            Assert.assertEquals(p1New.isOnlyAuthPush(), p1Updated.isOnlyAuthPush());
            Assert.assertEquals(user2, p1Updated.getUpdatedBy());
            Assert.assertEquals(1, p1Updated.getVersion());
        }

        { // Can change owner
            Project p2New = new Project(
                "p2", "P2: updated", "updated description", user2, acl, "p2: updated", true, true, true, true, true, "sensor",
                now, now, user2, user2, 0, Map.of("label1", "value1", "label2", "value2")
            );
            Project p2Updated = partialUpdateSync(p2New, true, false, true)
                .orElseThrow(() -> new AssertionError("project not found"));
            Assert.assertEquals(p2.getId(), p2Updated.getId());
            Assert.assertEquals("P2: updated", p2Updated.getName());
            Assert.assertEquals(user2, p2Updated.getOwner());
            Assert.assertEquals(acl, p2Updated.getAcl());
            Assert.assertEquals("p2: updated", p2Updated.getAbcService());
            Assert.assertEquals(p2New.isOnlyAuthRead(), p2Updated.isOnlyAuthRead());
            Assert.assertEquals(p2New.isOnlyAuthPush(), p2Updated.isOnlyAuthPush());
            Assert.assertEquals(user2, p2Updated.getUpdatedBy());
            Assert.assertEquals(1, p2Updated.getVersion());
        }

        { // Can change owner and deleted
            Project p3New = new Project(
                "p3", "P3: updated", "updated description", user2, acl, "p3: updated", false, false, false, false, false, "sensor",
                now, now, user2, user2, 0, Map.of("label1", "value1", "label2", "value2")
            );
            Project p3Updated = partialUpdateSync(p3New, true, false, true)
                .orElseThrow(() -> new AssertionError("project not found"));
            Assert.assertEquals(p3.getId(), p3Updated.getId());
            Assert.assertEquals("P3: updated", p3Updated.getName());
            Assert.assertEquals(user2, p3Updated.getOwner());
            Assert.assertEquals(acl, p3Updated.getAcl());
            Assert.assertEquals("p3: updated", p3Updated.getAbcService());
            Assert.assertEquals(p3New.isOnlyAuthRead(), p3Updated.isOnlyAuthRead());
            Assert.assertEquals(p3New.isOnlyAuthPush(), p3Updated.isOnlyAuthPush());
            Assert.assertEquals(user2, p3Updated.getUpdatedBy());
            Assert.assertEquals(1, p3Updated.getVersion());
        }
    }

    @Test
    public void partialUpdateIgnoringOldFields() {
        Acl acl = Acl.of(singleton("user3"), singleton("user4"), singleton("user5"), singleton("user6"));

        Project project = Project.newBuilder()
                .setId("project")
                .setName("Project")
                .setDescription("Description")
                .setOwner("user")
                .setAbcService("solomon")
                .setAcl(acl)
                .setOnlyAuthPush(true)
                .setOnlyAuthRead(true)
                .setOnlyMetricNameShards(true)
                .setOnlyNewFormatReads(true)
                .setOnlyNewFormatWrites(true)
                .setMetricNameLabel("sensor")
                .setLabels(Map.of("label1", "value1", "label2", "value2"))
                .build();
        Assert.assertTrue(insertSync(project));

        Project newProject = Project.newBuilder()
                .setId("project")
                .setName("Project: updated")
                .setDescription("Updated description")
                .setOwner("user2")
                .setAbcService("other")
                .setAcl(Acl.empty())
                .setOnlyAuthPush(false)
                .setOnlyAuthRead(false)
                .setOnlyMetricNameShards(false)
                .setOnlyNewFormatReads(false)
                .setOnlyNewFormatWrites(false)
                .setMetricNameLabel("metric")
                .setLabels(Map.of("label1", "value1"))
                .build();

        Project updatedProject = partialUpdateSync(newProject, true, true, false)
                .orElseThrow(() -> new AssertionError("project not found"));

        Assert.assertEquals(newProject.getId(), updatedProject.getId());
        Assert.assertEquals(newProject.getName(), updatedProject.getName());
        Assert.assertEquals(newProject.getOwner(), updatedProject.getOwner());
        Assert.assertEquals(newProject.getAbcService(), updatedProject.getAbcService());

        Assert.assertEquals(project.getAcl(), updatedProject.getAcl());
        Assert.assertEquals(project.isOnlyAuthPush(), updatedProject.isOnlyAuthPush());
        Assert.assertEquals(project.isOnlyAuthRead(), updatedProject.isOnlyAuthRead());
        Assert.assertEquals(project.isOnlyNewFormatReads(), updatedProject.isOnlyNewFormatReads());
        Assert.assertEquals(project.isOnlyNewFormatWrites(), updatedProject.isOnlyNewFormatWrites());

        Assert.assertEquals(newProject.isOnlyMetricNameShards(), updatedProject.isOnlyMetricNameShards());
        Assert.assertEquals(newProject.getMetricNameLabel(), updatedProject.getMetricNameLabel());
        Assert.assertEquals(newProject.getLabels(), updatedProject.getLabels());

        Assert.assertEquals(1, updatedProject.getVersion());
    }

    @Test
    public void partialUpdateWithoutVersion() {
        Project project = Project.newBuilder()
                .setId("project")
                .setName("Project")
                .setDescription("Description")
                .setOwner("user")
                .setAbcService("solomon")
                .build();
        Assert.assertTrue(insertSync(project));

        Project newProject = Project.newBuilder()
                .setId("project")
                .setName("Project: updated")
                .setDescription("Updated description")
                .setOwner("user")
                .setAbcService("solomon")
                .setVersion(-1)
                .build();

        Project updatedProject = partialUpdateSync(newProject, false, false, true)
                .orElseThrow(() -> new AssertionError("project not found"));

        Assert.assertEquals(newProject.getName(), updatedProject.getName());
        Assert.assertEquals(1, updatedProject.getVersion());
    }

    @Test
    public void partialTryToUpdateOnlyAuthPush() {
        Project project = newProject("project", "user", Acl.empty()).toBuilder()
                .setOnlyAuthPush(true)
                .build();
        Assert.assertTrue(insertSync(project));

        Project projectToUpdate = project.toBuilder()
                .setOnlyAuthPush(false)
                .build();

        Project updatedProject = partialUpdateSync(projectToUpdate, false, false, true)
            .orElseThrow(() -> new AssertionError("project not found"));

        Assert.assertEquals(project.getId(), updatedProject.getId());
        assertTrue(updatedProject.isOnlyAuthPush());
    }

    @Test
    public void partialUpdateOnlyAuthPush() {
        Project project = newProject("project", "user", Acl.empty()).toBuilder()
                .setOnlyAuthPush(true)
                .build();
        Assert.assertTrue(insertSync(project));

        Project projectToUpdate = project.toBuilder()
                .setOnlyAuthPush(false)
                .build();

        Project updatedProject = partialUpdateSync(projectToUpdate, false, true, true)
            .orElseThrow(() -> new AssertionError("project not found"));

        Assert.assertEquals(project.getId(), updatedProject.getId());
        assertFalse(updatedProject.isOnlyAuthPush());
    }

    @Test
    public void extraInsertAndFind() {
        Project booProject = simpleProject("Boo").setAcl(simpleAcl()).build();
        assertTrue(insertSync(booProject));

        Project solomonProject = simpleProject("Solomon").setAcl(simpleAcl()).build();
        assertTrue(insertSync(solomonProject));

        Project zooProject = simpleProject("Zoo").setAcl(simpleAcl()).build();
        assertTrue(insertSync(zooProject));

        Project foundProject = findByIdSync(solomonProject.getId()).orElseThrow(AssertionError::new);
        assertEquals(solomonProject, foundProject);

        Project duplicateProject = simpleProject("Solomon").build();
        assertFalse(insertSync(duplicateProject));

        foundProject = findByIdSync(booProject.getId()).orElseThrow(AssertionError::new);
        assertEquals(booProject, foundProject);
        foundProject = findByIdSync(solomonProject.getId()).orElseThrow(AssertionError::new);
        assertEquals(solomonProject, foundProject);
        foundProject = findByIdSync(zooProject.getId()).orElseThrow(AssertionError::new);
        assertEquals(zooProject, foundProject);

        assertFalse(findByIdSync("Woo").isPresent());
    }

    @Test
    public void extraFindAllNames() {
        List<Project> projects = IntStream.range(1000, 1005)
            .mapToObj(String::valueOf)
            .map(AbstractProjectsDaoTest::simpleProject)
            .map(Project.Builder::build)
            .collect(Collectors.toList());
        projects.forEach(this::insertSync);
        assertEquals(projects, join(getProjectsDao().findAllNames()));
    }

    @Test
    public void extraFindAllAcls() {
        List<Project> projects = Arrays.asList(
            simpleProject("first").setAcl(simpleAcl()).build(),
            simpleProject("second").build(),
            simpleProject("third").setAcl(aclForSingleUser("user1", Permission.READ)).build()
        );

        projects.forEach(this::insertSync);
        assertEquals(projects, join(getProjectsDao().findAllNames()));
    }

    @Test
    public void extraExists() {
        insertSync(simpleProject("abc").build());
        insertSync(simpleProject("def").build());

        assertTrue(join(getProjectsDao().exists("abc")));
        assertTrue(join(getProjectsDao().exists("def")));
        assertFalse(join(getProjectsDao().exists("xyz")));

        assertTrue(join(getProjectsDao().deleteOne("abc", true)));
        assertFalse(join(getProjectsDao().exists("abc")));
        assertTrue(join(getProjectsDao().exists("def")));

        assertFalse(join(getProjectsDao().deleteOne("xyz", true)));
    }

    @Test
    public void extraPartialUpdateProjectNotFound() {
        Project.Builder builder = simpleProject("klm");
        Project project = builder.build();
        insertSync(project);
        Project update = builder.setId("vw").build();
        assertNoUpdate(update);
        assertFalse(findByIdSync(update.getId()).isPresent());
    }

    @Test
    public void extraPartialUpdateProjectStaleVersion() {
        Project.Builder builder = simpleProject("klm");
        builder.setVersion(10);
        Project project = builder.build();
        insertSync(project);
        builder.setVersion(9);
        assertNoUpdate(builder.build());
        assertEquals(project, findByIdSync(project.getId()).get());
    }

    @Test
    public void extraPartialUpdate() {
        Project.Builder builder = simpleProject("fcsm");
        Project project = builder.build();
        insertSync(project);
        Project update = builder.setOwner("other").build();
        Project afterUpdate = partialUpdateSync(update, false, false, true).orElseThrow(AssertionError::new);
        Project.Builder expected = update.toBuilder();
        expected.setVersion(project.getVersion() + 1);
        expected.setOwner(project.getOwner());
        assertEquals(expected.build(), afterUpdate);
        Project foundProject = findByIdSync(afterUpdate.getId()).orElseThrow(AssertionError::new);
        assertEquals(afterUpdate, foundProject);
    }

    @Test
    public void upsertNotExistingProjects() {
        Project project = Project.newBuilder()
            .setId("project")
            .setName("name")
            .setOwner("user")
            .build();

        upsertProjectsSync(List.of(project));

        Optional<Project> foundProjectOpt = findByIdSync("project");
        assertTrue(foundProjectOpt.isPresent());
        assertEquals(project, foundProjectOpt.get());
    }

    @Test
    public void updateAfterUpsert() {
        Project project = Project.newBuilder()
            .setId("project")
            .setName("name")
            .setOwner("user")
            .build();

        upsertProjectsSync(List.of(project));

        Project projectToUpdate = Project.newBuilder()
                .setId("project")
                .setName("name")
                .setDescription("description")
                .setOwner("user")
                .setAcl(Acl.of(Set.of("user"), Set.of("user"), Set.of("user"), Set.of("user")))
                .setUpdatedBy("other-user")
                .setVersion(0)
                .build();

        Project expectedProject = projectToUpdate.toBuilder()
                .setVersion(1)
                .build();

        Optional<Project> updatedProjectOpt = partialUpdateSync(projectToUpdate, true, true, true);
        assertTrue(updatedProjectOpt.isPresent());
        assertEquals(expectedProject, updatedProjectOpt.get());
    }

    @Test
    public void upsertTooManyProjects() {
        List<Project> projects = new ArrayList<>();
        for (int i = 0; i < 100; ++i) {
            Project project = Project.newBuilder()
                .setId("project" + i)
                .setName("name" + 1)
                .setOwner("user")
                .setAbcService("abc_service")
                .build();

            projects.add(project);
        }

        upsertProjectsSync(projects);

        for (Project project : projects) {
            Optional<Project> foundProjectOpt = findByIdSync(project.getId());
            assertTrue(foundProjectOpt.isPresent());
        }
    }

    @Test
    public void upsertExistingProjects() {
        Project project = Project.newBuilder()
            .setId("project")
            .setName("name")
            .setDescription("description")
            .setOwner("user")
            .setAbcService("abc_service")
            .setAcl(Acl.of(Set.of("user"), Set.of("user"), Set.of("user"), Set.of("user")))
            .setOnlyAuthRead(true)
            .setOnlyAuthPush(false)
            .setOnlyMetricNameShards(true)
            .setOnlyNewFormatWrites(true)
            .setOnlyNewFormatReads(true)
            .setCreatedBy("robot-solomon")
            .setUpdatedBy("robot-solomon")
            .setVersion(1)
            .build();

        insertSync(project);

        Project projectToUpsert = Project.newBuilder()
            .setId("project")
            .setName("other_name")
            .setOwner("other_user")
            .setOnlyAuthPush(true)
            .setAbcService("other_abc_service")
            .setCreatedBy("other-robot-solomon")
            .setUpdatedBy("other-robot-solomon")
            .build();

        upsertProjectsSync(List.of(projectToUpsert));

        Optional<Project> foundProjectOpt = findByIdSync("project");
        assertTrue(foundProjectOpt.isPresent());

        Project foundProject = foundProjectOpt.get();

        Project expectedProject = project.toBuilder()
            .setName("other_name")
            .setOwner("other_user")
            .setOnlyAuthPush(true)
            .build();

        assertEquals(expectedProject, foundProject);

        Project projectToUpdate = Project.newBuilder()
                .setId("project")
                .setName("other-name2")
                .setOwner("owner")
                .setAbcService("other_abc_service2")
                .setCreatedBy("other-robot-solomon")
                .setUpdatedBy("other-robot-solomon")
                .setVersion(1)
                .build();

        var updatedProjectOpt = partialUpdateSync(projectToUpdate, true, true, true);

        assertTrue(updatedProjectOpt.isPresent());
    }

    private static Project.Builder simpleProject(String id) {
        Project.Builder builder = Project.newBuilder();
        String owner = "snoop";
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());

        builder.setId(id).setName(id + " project").setOwner(owner).setAbcService(id.toLowerCase());
        builder.setCreatedAt(now).setUpdatedAt(now).setCreatedBy(owner).setUpdatedBy(owner).setVersion(1);
        return builder;
    }

    private static Acl simpleAcl() {
        List<String> loginList = Arrays.asList("reader", "updater", "cleaner");

        Set<String> readers = new HashSet<>(loginList);
        Set<String> updaters = new HashSet<>(loginList.subList(1, loginList.size()));
        Set<String> cleaners = new HashSet<>(loginList.subList(2,loginList.size()));
        Set<String> writers = Collections.singleton(loginList.get(loginList.size() - 1));

        return Acl.of(readers, updaters, cleaners, writers);
    }

    private static Acl aclForSingleUser(String login, Permission permission) {
        Set<String> readers = Collections.emptySet();
        Set<String> updaters = Collections.emptySet();
        Set<String> cleaners = Collections.emptySet();
        Set<String> writers = Collections.emptySet();
        Set<String> singleUser = Collections.singleton(login);
        switch (permission) {
            case READ:
                readers = singleUser;
                break;
            case UPDATE:
                updaters = singleUser;
                break;
            case CLEAN:
                cleaners = singleUser;
                break;
            case WRITE:
                writers = singleUser;
                break;
        }
        return Acl.of(readers, updaters, cleaners, writers);
    }

    private void assertNoUpdate(Project update) {
        assertNoUpdate(update, true);
        assertNoUpdate(update, false);
    }

    private void assertNoUpdate(Project update, boolean canChange) {
        assertFalse(partialUpdateSync(update, canChange, canChange, true).isPresent());
    }

    enum Permission {
        READ,
        UPDATE,
        CLEAN,
        WRITE,
    }

    private boolean insertSync(Project project) {
        return join(getProjectsDao().insert(project));
    }

    private Optional<Project> partialUpdateSync(
        Project p,
        boolean canChangeOwner,
        boolean canChangeInternalOptions,
        boolean canChangeOldFields)
    {
        return join(getProjectsDao().partialUpdate(p, canChangeOwner, canChangeInternalOptions, canChangeOldFields));
    }

    private void upsertProjectsSync(List<Project> projects) {
        join(getProjectsDao().upsertProjects(projects));
    }

    private Optional<Project> findByIdSync(String id) {
        return join(getProjectsDao().findById(id));
    }

    private PagedResult<Project> findSync(
            String text,
            String abcFilter,
            String login,
            EnumSet<ProjectPermission> filterByPermissions,
            PageOptions pageOptions)
    {
        return join(getProjectsDao().find(text, abcFilter, login, filterByPermissions, pageOptions));
    }

    private PagedResult<Project> findSync(String text, String abcFilter, PageOptions pageOptions) {
        return join(getProjectsDao().find(text, abcFilter, "", EnumSet.noneOf(ProjectPermission.class), pageOptions));
    }

    private TokenBasePage<Project> findV3Sync(String text, int pageSize, String pageToken) {
        return join(getProjectsDao().findV3(text, pageSize, pageToken));
    }

    private static Project newProject(String id, String owner, Acl acl) {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return new Project(
            id, StringUtils.capitalize(id), "Description for " + id, owner, acl, id, false, false, false, false, false, "sensor",
            now, now, owner, owner, 0, Map.of("label1", "value1", "label2", "value2")
        );
    }
}
