package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;

import javax.annotation.Nullable;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.core.db.model.Quota;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Ivan Tsybulin
 */
@YaExternal
public abstract class AbstractQuotasDaoTest {

    private Instant now;
    protected abstract QuotasDao getQuotasDao();

    @Before
    public void setUp() {
        now = Instant.now().truncatedTo(ChronoUnit.MILLIS);
    }

    @Test
    public void insert() {
        var orig = findAllLimits("alerting", "project", "yc-solomon");
        assertTrue(orig.isEmpty());

        upsert("alerting", "project", "yc-solomon", "expression.alert.count", 100, "uranix", now);
        var next = findAllLimits("alerting", "project", "yc-solomon");
        assertThat(next, iterableWithSize(1));

        assertThat(next.get(0).getNamespace(), equalTo("alerting"));
        assertThat(next.get(0).getScopeType(), equalTo("project"));
        assertThat(next.get(0).getScopeId(), equalTo("yc-solomon"));
        assertThat(next.get(0).getIndicator(), equalTo("expression.alert.count"));
        assertThat(next.get(0).getLimit(), equalTo(100L));
        assertThat(next.get(0).getUpdatedBy(), equalTo("uranix"));
        assertThat(next.get(0).getUpdatedAt(), equalTo(now));
    }

    private void populate() {
        upsert("alerting", "project", "yc-solomon", "expression.alert.count", 100, "uranix", now);
        upsert("alerting", "project", null, "expression.alert.count", 10, "uranix", now);
        upsert("alerting", "project", null, "expression.subAlert.count", 1000, "gordiychuk", now.plusMillis(1000));

        upsert("alerting", "notification", "email", "max.per.day", 10000, "gordiychuk", now.plusMillis(5000));
        upsert("alerting", "notification", "webhook", "max.message.size.bytes", 1000_000, "gordiychuk", now.plusMillis(5000));

        upsert("coremon", "shard", null, "max.fileSensors", 10_000_000, "jamel", now);
        upsert("coremon", "shard", "kikimr_lbk_blobstorage", "max.fileSensors", 1000_000_000, "jamel", now);
    }

    @Test
    public void findWholeService() {
        populate();

        assertThat(findAllByNamespace("alerting"), iterableWithSize(5));
        assertThat(findAllByNamespace("coremon"), iterableWithSize(2));
    }

    @Test
    public void insertMany() {
        var orig = findAllLimits("alerting", "project", "yc-solomon");
        assertTrue(orig.isEmpty());

        populate();

        {
            var next = findAllLimits("alerting", "project", "yc-solomon");

            assertThat(next, iterableWithSize(3));

            assertThat(next, hasItem(Quota.newBuilder()
                .setNamespace("alerting")
                .setScopeType("project")
                .setScopeId("yc-solomon")
                .setIndicator("expression.alert.count")
                .setLimit(100)
                .setUpdatedBy("uranix")
                .setUpdatedAt(now)
                .build()));

            assertThat(next, hasItem(Quota.newBuilder()
                .setNamespace("alerting")
                .setScopeType("project")
                .setIndicator("expression.alert.count")
                .setLimit(10)
                .setUpdatedBy("uranix")
                .setUpdatedAt(now)
                .build()));

            assertThat(next, hasItem(Quota.newBuilder()
                .setNamespace("alerting")
                .setScopeType("project")
                .setIndicator("expression.subAlert.count")
                .setLimit(1000)
                .setUpdatedBy("gordiychuk")
                .setUpdatedAt(now.plusMillis(1000))
                .build()));
        }

        {
            var defaults = findAllLimits("coremon", "shard", null);
            var concreteShard = findAllLimits("coremon", "shard", "solomon_test_example");

            assertThat(defaults, iterableWithSize(1));
            assertThat(concreteShard, equalTo(defaults));

            assertThat(concreteShard.get(0).getLimit(), equalTo(10_000_000L));
        }
    }

    @Test
    public void update() {
        populate();

        {
            upsert("alerting", "project", "yc-solomon", "expression.alert.count", 1000, "guschin", Instant.EPOCH);
            var next = findAllLimits("alerting", "project", "yc-solomon");
            assertThat(next, iterableWithSize(3));

            assertThat(next, hasItem(Quota.newBuilder()
                .setNamespace("alerting")
                .setScopeType("project")
                .setScopeId("yc-solomon")
                .setIndicator("expression.alert.count")
                .setLimit(1000)
                .setUpdatedBy("guschin")
                .setUpdatedAt(Instant.EPOCH)
                .build()));

            assertThat(next, hasItem(Quota.newBuilder()
                .setNamespace("alerting")
                .setScopeType("project")
                .setIndicator("expression.alert.count")
                .setLimit(10)
                .setUpdatedBy("uranix")
                .setUpdatedAt(now)
                .build()));

            assertThat(next, hasItem(Quota.newBuilder()
                .setNamespace("alerting")
                .setScopeType("project")
                .setIndicator("expression.subAlert.count")
                .setLimit(1000)
                .setUpdatedBy("gordiychuk")
                .setUpdatedAt(now.plusMillis(1000))
                .build()));
        }
    }

    @Test
    public void deleteByOne() {
        populate();

        {
            deleteOne("alerting", "project", "yc-solomon", "foo.bar.baz");
            var next = findAllLimits("alerting", "project", "yc-solomon");
            assertThat(next, iterableWithSize(3));
        }

        {
            deleteOne("alerting", "project", "yc-solomon", "expression.alert.count");
            var next = findAllLimits("alerting", "project", "yc-solomon");
            assertThat(next, iterableWithSize(2));
        }

        {
            deleteOne("alerting", "project", null, "expression.alert.count");
            var next = findAllLimits("alerting", "project", "yc-solomon");
            assertThat(next, iterableWithSize(1));
            var coremon = findAllLimits("coremon", "shard", null);
            assertThat(coremon, iterableWithSize(1));
        }
    }

    @Test
    public void deleteBadIndicator() {
        populate();

        {
            delete("alerting", "project", "expression.alert.count");
            var next = findAllLimits("alerting", "project", "yc-solomon");
            assertThat(next, iterableWithSize(1));
        }
    }

    private List<Quota> findAllLimits(String namespace, String scopeType, @Nullable String scopeId) {
        return join(getQuotasDao().findAllIndicators(namespace, scopeType, scopeId));
    }

    private List<Quota> findAllByNamespace(String namespace) {
        return join(getQuotasDao().findAllByNamespace(namespace));
    }

    private void upsert(String namespace, String scopeType, @Nullable String scopeId, String indicator, long newLimit, String updatedBy, Instant updatedAt) {
        join(getQuotasDao().upsert(namespace, scopeType, scopeId, indicator, newLimit, updatedBy, updatedAt));
    }

    private void delete(String namespace, String scopeType, String indicator) {
        join(getQuotasDao().delete(namespace, scopeType, indicator));
    }

    private void deleteOne(String namespace, String scopeType, String scopeId, String indicator) {
        join(getQuotasDao().deleteOne(namespace, scopeType, scopeId, indicator));
    }
}
