package ru.yandex.solomon.core.db.dao;

import java.time.Instant;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import ru.yandex.solomon.core.db.model.ReferenceConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeTrue;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public abstract class AbstractServiceProvidersDaoTest {

    protected abstract ServiceProvidersDao getDao();

    @Test
    public void insert() {
        ServiceProvider serviceProvider = serviceProvider("service");
        assertTrue(insertSync(serviceProvider));

        Optional<ServiceProvider> fromDb = readSync("service");
        assertTrue(fromDb.isPresent());
        assertEquals(serviceProvider, fromDb.get());
    }

    @Test
    public void list() {
        ServiceProvider serviceProvider1 = serviceProvider("service1");
        ServiceProvider serviceProvider2 = serviceProvider("service2");
        ServiceProvider serviceProvider3 = serviceProvider("service3");
        ServiceProvider serviceProvider4 = serviceProvider("service4");

        assumeTrue(insertSync(serviceProvider3));
        assumeTrue(insertSync(serviceProvider1));
        assumeTrue(insertSync(serviceProvider2));
        assumeTrue(insertSync(serviceProvider4));

        {
            TokenBasePage<ServiceProvider> page = listSync("", 2, "");
            assertEquals(List.of(serviceProvider1, serviceProvider2), page.getItems());
            assertEquals("2", page.getNextPageToken());
        }

        {
            TokenBasePage<ServiceProvider> page = listSync("", 2, "2");
            assertEquals(List.of(serviceProvider3, serviceProvider4), page.getItems());
            assertEquals("", page.getNextPageToken());
        }

        {
            TokenBasePage<ServiceProvider> page = listSync("", 3, "2");
            assertEquals(List.of(serviceProvider3, serviceProvider4), page.getItems());
            assertEquals("", page.getNextPageToken());
        }
    }

    @Test
    public void listWithFilter() {
        ServiceProvider serviceProvider1 = serviceProvider("service1");
        ServiceProvider serviceProvider2 = serviceProvider("service2");
        ServiceProvider serviceProvider3 = serviceProvider("service3");
        ServiceProvider serviceProvider10 = serviceProvider("service10");

        assumeTrue(insertSync(serviceProvider1));
        assumeTrue(insertSync(serviceProvider2));
        assumeTrue(insertSync(serviceProvider3));
        assumeTrue(insertSync(serviceProvider10));

        {
            TokenBasePage<ServiceProvider> page = listSync("1", 1, "");
            assertEquals(List.of(serviceProvider1), page.getItems());
            assertEquals("1", page.getNextPageToken());
        }

        {
            TokenBasePage<ServiceProvider> page = listSync("1", 1, "1");
            assertEquals(List.of(serviceProvider10), page.getItems());
            assertEquals("", page.getNextPageToken());
        }
    }

    @Test
    public void partialUpdate() {
        ServiceProvider serviceProvider = serviceProvider("service");
        assumeTrue(insertSync(serviceProvider));

        ServiceProvider update = serviceProvider.toBuilder()
                .setDescription("another description")
                .setShardSettings(ServiceProviderShardSettings.EMPTY)
                .setAbcService("solomon-other")
                .setCloudId("yc-solomon-other")
                .setTvmDestId("789")
                .setIamServiceAccountIds(List.of("012", "013"))
                .setCreatedAt(Instant.parse("2020-01-03T00:00:00Z"))
                .setUpdatedAt(Instant.parse("2020-01-04T00:00:00Z"))
                .setCreatedBy("user2")
                .setCreatedBy("user3")
                .setHasGlobalId(true)
                .setVersion(0)
                .build();

        Optional<ServiceProvider> updatedOpt = updateSync(update);
        assertTrue(updatedOpt.isPresent());
        ServiceProvider updated = updatedOpt.get();

        ServiceProvider expected = update.toBuilder()
                .setCreatedAt(serviceProvider.getCreatedAt())
                .setCreatedBy(serviceProvider.getCreatedBy())
                .setVersion(1)
                .build();

        assertEquals(expected, updated);
    }

    @Test
    public void partialUpdateWithoutVersion() {
        ServiceProvider serviceProvider = serviceProvider("service");
        assumeTrue(insertSync(serviceProvider));

        ServiceProvider update = serviceProvider.toBuilder()
                .setDescription("another description")
                .setShardSettings(ServiceProviderShardSettings.EMPTY)
                .setAbcService("solomon-other")
                .setCloudId("yc-solomon-other")
                .setTvmDestId("789")
                .setIamServiceAccountIds(List.of("012", "013"))
                .setCreatedAt(Instant.parse("2020-01-03T00:00:00Z"))
                .setUpdatedAt(Instant.parse("2020-01-04T00:00:00Z"))
                .setCreatedBy("user2")
                .setCreatedBy("user3")
                .setHasGlobalId(true)
                .setVersion(-1)
                .build();

        Optional<ServiceProvider> updatedOpt = updateSync(update);
        assertTrue(updatedOpt.isPresent());
        ServiceProvider updated = updatedOpt.get();

        ServiceProvider expected = update.toBuilder()
                .setCreatedAt(serviceProvider.getCreatedAt())
                .setCreatedBy(serviceProvider.getCreatedBy())
                .setVersion(1)
                .build();

        assertEquals(expected, updated);
    }

    @Test
    public void delete() {
        ServiceProvider serviceProvider = serviceProvider("service1");
        assumeTrue(insertSync(serviceProvider));
        assertTrue(deleteSync("service1"));
        assertFalse(deleteSync("service2"));
        assertTrue(readSync("service1").isEmpty());
    }

    @Test
    public void exists() {
        ServiceProvider serviceProvider = serviceProvider("service");
        assumeTrue(insertSync(serviceProvider));
        assertTrue(existsSync("service"));
        assertFalse(existsSync("service2"));
    }

    @Test
    public void referenceSettings() {
        ServiceProvider init = serviceProvider("service")
                .toBuilder()
                .setReferences(List.of(new ReferenceConf("resource_id", List.of("compute"), List.of("vm", "disk"), false)))
                .build();
        assumeTrue(insertSync(init));

        var v1 = readSync(init.getId()).orElseThrow();
        assertEquals(init.getReferences(), v1.getReferences());

        ServiceProvider update = v1.toBuilder()
                .setReferences(List.of(
                        new ReferenceConf("resource_id", List.of("compute"), List.of("vm", "disk"), false),
                        new ReferenceConf("certificate", List.of("certificate-manager"), List.of("certificate"), false)))
                .build();

        var v2 = updateSync(update).orElseThrow();
        assertEquals(update.getReferences(), v2.getReferences());

        var v2Read = readSync(init.getId()).orElseThrow();
        assertEquals(v2, v2Read);
    }

    private ServiceProvider serviceProvider(String id) {
        return ServiceProvider.newBuilder()
                .setId(id)
                .setDescription("description")
                .setShardSettings(new ServiceProviderShardSettings(
                        ServiceMetricConf.of(new ServiceMetricConf.AggrRule[]{AggrRule.of("host=*", "host=cluster", null)}, true),
                        30,
                        15,
                        15
                ))
                .setAbcService("solomon")
                .setCloudId("yc-solomon")
                .setTvmDestId("123")
                .setIamServiceAccountIds(List.of("456", "457"))
                .setCreatedAt(Instant.parse("2020-01-01T00:00:00Z"))
                .setUpdatedAt(Instant.parse("2020-01-02T00:00:00Z"))
                .setCreatedBy("user1")
                .setUpdatedBy("user2")
                .setHasGlobalId(true)
                .build();
    }

    private Optional<ServiceProvider> readSync(String id) {
        return getDao().read(id).join();
    }

    private boolean insertSync(ServiceProvider serviceProvider) {
        return getDao().insert(serviceProvider).join();
    }

    private TokenBasePage<ServiceProvider> listSync(String filterById, int pageSize, String pageToken) {
        return getDao().list(filterById, pageSize, pageToken).join();
    }

    private Optional<ServiceProvider> updateSync(ServiceProvider serviceProvider) {
        return getDao().update(serviceProvider).join();
    }

    private boolean deleteSync(String id) {
        return getDao().delete(id).join();
    }

    private boolean existsSync(String id) {
        return getDao().exists(id).join();
    }
}
