package ru.yandex.solomon.core.db.dao;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.byId;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.equalsByString;


/**
 * @author Sergey Polovko
 */
@YaExternal
public abstract class AbstractServicesDaoTest {

    protected abstract ServicesDao getServicesDao();

    @Test
    public void insert() {
        Service service = newService("service1", "project1", "", "user");
        Assert.assertTrue(insertSync(service));

        Optional<Service> fromDb = findOneSync(service.getProjectId(), service.getId());
        Assert.assertTrue(fromDb.isPresent());
        Assert.assertEquals(service, fromDb.get());
    }

    @Test
    public void insertWithFolder() {
        Service service = newService("service1", "project1", "folder1", "user");
        Assert.assertTrue(insertSync(service));

        Optional<Service> fromDb = findOneSync(service.getProjectId(), service.getFolderId(), service.getId());
        Assert.assertTrue(fromDb.isPresent());
        Assert.assertEquals(service, fromDb.get());
    }

    @Test
    public void findByProjectId() {
        List<String> projectIds = List.of("kikimr", "yql", "solomon");
        List<Service> services = new ArrayList<>(projectIds.size() * 4);

        for (String projectId : projectIds) {
            services.add(newService(projectId + "_s1", projectId, "", "user1"));
            services.add(newService(projectId + "_s2", projectId, "", "user1"));
            services.add(newService(projectId + "_s3", projectId, "", "user1"));
            services.add(newService(projectId + "_s4", projectId, "", "user1"));
        }

        for (Service service : services) {
            Assert.assertTrue(insertSync(service));
        }

        {
            List<Service> yqlServices = services.stream()
                .filter(s -> "yql".equals(s.getProjectId()))
                .collect(Collectors.toList());
            PagedResult<Service> yqlServicesFromDb = findByProjectIdSync("yql", new PageOptions(10, 0), "", ShardSettings.Type.UNSPECIFIED);

            Assert.assertEquals(yqlServices.size(), yqlServicesFromDb.getResult().size());

            for (int i = 0; i < yqlServices.size(); i++) {
                Service service = yqlServices.get(i);
                Service fromDb = yqlServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, yqlServicesFromDb.getCurrentPage());
            Assert.assertEquals(1, yqlServicesFromDb.getPagesCount());
        }

        {
            List<Service> solomonServices = services.stream()
                .filter(s -> "solomon".equals(s.getProjectId()))
                .limit(2)
                .collect(Collectors.toList());
            PagedResult<Service> solomonServicesFromDb = findByProjectIdSync("solomon", new PageOptions(2, 0), "", ShardSettings.Type.UNSPECIFIED);
            Assert.assertEquals(solomonServices.size(), solomonServicesFromDb.getResult().size());

            for (int i = 0; i < solomonServices.size(); i++) {
                Service service = solomonServices.get(i);
                Service fromDb = solomonServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, solomonServicesFromDb.getCurrentPage());
            Assert.assertEquals(2, solomonServicesFromDb.getPagesCount());
        }
    }

    @Test
    public void findByProjectId_byModel() {
        List<String> projectIds = List.of("yql");
        List<Service> services = new ArrayList<>(projectIds.size() * 4);

        for (String projectId : projectIds) {
            services.add(newService(projectId + "_s1", projectId, "", "user1"));
            services.add(newService(projectId + "_s2", projectId, "", "user1"));
            services.add(newService(projectId + "_s3", projectId, "", "user1").toBuilder()
                    .setShardSettings(ShardSettings.of(ShardSettings.Type.PULL,
                            ShardSettings.PullSettings.of(11, "abc", false, ShardSettings.PullProtocol.HTTP, "", ShardSettings.HostLabelPolicy.SHORT_HOSTNAME),
                            10,
                            37,
                            DecimPolicy.UNDEFINED,
                            ShardSettings.AggregationSettings.of(true, new AggrRule[0], false),
                            10))
                    .build());
        }

        for (Service service : services) {
            Assert.assertTrue(insertSync(service));
        }

        {
            PagedResult<Service> yqlServicesFromDb = findByProjectIdSync("yql", new PageOptions(10, 0), "", ShardSettings.Type.UNSPECIFIED);

            Assert.assertEquals(services.size(), yqlServicesFromDb.getResult().size());

            for (int i = 0; i < services.size(); i++) {
                Service service = services.get(i);
                Service fromDb = yqlServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, yqlServicesFromDb.getCurrentPage());
            Assert.assertEquals(1, yqlServicesFromDb.getPagesCount());
        }
        {
            PagedResult<Service> yqlServicesFromDb = findByProjectIdSync("yql", new PageOptions(10, 0), "", ShardSettings.Type.PULL);
            List<Service> expectedFirstPageItems = List.of(services.get(2));
            Assert.assertEquals(expectedFirstPageItems.size(), yqlServicesFromDb.getResult().size());

            for (int i = 0; i < expectedFirstPageItems.size(); i++) {
                Service service = expectedFirstPageItems.get(i);
                Service fromDb = yqlServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, yqlServicesFromDb.getCurrentPage());
            Assert.assertEquals(1, yqlServicesFromDb.getPagesCount());
        }
        {
            PagedResult<Service> yqlServicesFromDb = findByProjectIdSync("yql", new PageOptions(10, 0), "", ShardSettings.Type.PUSH);
            List<Service> expectedFirstPageItems = List.of(services.get(0), services.get(1));
            Assert.assertEquals(expectedFirstPageItems.size(), yqlServicesFromDb.getResult().size());

            for (int i = 0; i < expectedFirstPageItems.size(); i++) {
                Service service = expectedFirstPageItems.get(i);
                Service fromDb = yqlServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, yqlServicesFromDb.getCurrentPage());
            Assert.assertEquals(1, yqlServicesFromDb.getPagesCount());
        }
    }

    @Test
    public void findByProjectIdPaged() {
        List<String> projectIds = List.of("kikimr", "yql", "solomon");
        List<Service> services = new ArrayList<>(projectIds.size() * 4);

        for (String projectId : projectIds) {
            services.add(newService(projectId + "_s1", projectId, "", "user1"));
            services.add(newService(projectId + "_s2", projectId, "", "user1"));
            services.add(newService(projectId + "_s3", projectId, "", "user1"));
            services.add(newService(projectId + "_s4", projectId, "", "user1"));
        }

        for (Service service : services) {
            Assert.assertTrue(insertSync(service));
        }

        TokenBasePage<Service> firstPage = findByProjectIdPagedSync("yql", 3, "", "");

        assertEquals(3, firstPage.getItems().size());
        assertEquals("3", firstPage.getNextPageToken());

        List<Service> expectedFirstPageItems = services.stream()
                .filter(s -> "yql".equals(s.getProjectId()))
                .sorted(Comparator.comparing(Service::getId))
                .limit(3)
                .collect(Collectors.toList());

        assertEquals(expectedFirstPageItems, firstPage.getItems());

        TokenBasePage<Service> nextPage = findByProjectIdPagedSync("yql", 3, "3", "");

        assertEquals(1, nextPage.getItems().size());
        assertEquals("", nextPage.getNextPageToken());

        List<Service> expectedNextPageItems = services.stream()
                .filter(s -> "yql".equals(s.getProjectId()))
                .sorted(Comparator.comparing(Service::getId))
                .skip(3)
                .limit(3)
                .collect(Collectors.toList());

        assertEquals(expectedNextPageItems, nextPage.getItems());
    }

    @Test
    public void findByFolderId() {
        List<String> folderIds = List.of("kikimr", "yql", "solomon");
        List<Service> services = new ArrayList<>(folderIds.size() * 4);

        for (String folderId : folderIds) {
            services.add(newService(folderId + "_s1", "solomon", folderId, "user1"));
            services.add(newService(folderId + "_s2", "solomon", folderId, "user1"));
            services.add(newService(folderId + "_s3", "solomon", folderId, "user1"));
            services.add(newService(folderId + "_s4", "solomon", folderId, "user1"));
        }

        for (Service service : services) {
            Assert.assertTrue(insertSync(service));
        }

        {
            List<Service> yqlServices = services.stream()
                .filter(s -> "yql".equals(s.getFolderId()))
                .collect(Collectors.toList());
            PagedResult<Service> yqlServicesFromDb = findByFolderIdSync("solomon", "yql", new PageOptions(10, 0), "");

            Assert.assertEquals(yqlServices.size(), yqlServicesFromDb.getResult().size());

            for (int i = 0; i < yqlServices.size(); i++) {
                Service service = yqlServices.get(i);
                Service fromDb = yqlServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, yqlServicesFromDb.getCurrentPage());
            Assert.assertEquals(1, yqlServicesFromDb.getPagesCount());
        }

        {
            List<Service> solomonServices = services.stream()
                .filter(s -> "solomon".equals(s.getFolderId()))
                .limit(2)
                .collect(Collectors.toList());
            PagedResult<Service> solomonServicesFromDb = findByFolderIdSync("solomon", "solomon", new PageOptions(2, 0), "");
            Assert.assertEquals(solomonServices.size(), solomonServicesFromDb.getResult().size());

            for (int i = 0; i < solomonServices.size(); i++) {
                Service service = solomonServices.get(i);
                Service fromDb = solomonServicesFromDb.getResult().get(i);
                assertPartialEquality(service, fromDb);
            }

            Assert.assertEquals(0, solomonServicesFromDb.getCurrentPage());
            Assert.assertEquals(2, solomonServicesFromDb.getPagesCount());
        }
    }

    @Test
    public void partialUpdate() {
        Service s1 = newService("s1", "p1", "", "user1");
        Assert.assertTrue(insertSync(s1));

        Service s1New = newService("s1", "p1", "", "user2")
            .toBuilder()
            .setMetricNameLabel("metric")
            .build();

        Optional<Service> s1UpdatedO = join(getServicesDao().partialUpdate(s1New));
        Assert.assertTrue(s1UpdatedO.isPresent());

        Service s1Updated = s1UpdatedO.get();
        Assert.assertEquals("s1", s1Updated.getId());
        Assert.assertEquals("p1", s1Updated.getProjectId());
        Assert.assertEquals("metric", s1Updated.getMetricNameLabel());
        Assert.assertEquals(1, s1Updated.getVersion());
        Assert.assertEquals(s1.getCreatedAt(), s1Updated.getCreatedAt());
        Assert.assertEquals("user2", s1Updated.getUpdatedBy());
    }

    @Test
    public void partialUpdateWithVersion() {
        Service s1 = newService("s1", "p1", "", "user1");
        Assert.assertTrue(insertSync(s1));

        Service s1New = newService("s1", "p1", "", "user2")
            .toBuilder()
            .setMetricNameLabel("metric")
            .setVersion(-1)
            .build();

        Optional<Service> s1UpdatedO = join(getServicesDao().partialUpdate(s1New));
        Assert.assertTrue(s1UpdatedO.isPresent());

        Service s1Updated = s1UpdatedO.get();
        Assert.assertEquals("s1", s1Updated.getId());
        Assert.assertEquals("p1", s1Updated.getProjectId());
        Assert.assertEquals("metric", s1Updated.getMetricNameLabel());
        Assert.assertEquals(1, s1Updated.getVersion());
        Assert.assertEquals(s1.getCreatedAt(), s1Updated.getCreatedAt());
        Assert.assertEquals("user2", s1Updated.getUpdatedBy());
    }

    @Test
    public void partialUpdateWithFolder() {
        Service s1 = newService("s1", "p1", "f1", "user1");
        Assert.assertTrue(insertSync(s1));

        Service s1New = newService("s1", "p1", "f1", "user2")
            .toBuilder()
            .setMetricNameLabel("metric")
            .build();

        Optional<Service> s1UpdatedO = join(getServicesDao().partialUpdate(s1New));
        Assert.assertTrue(s1UpdatedO.isPresent());

        Service s1Updated = s1UpdatedO.get();
        Assert.assertEquals("s1", s1Updated.getId());
        Assert.assertEquals("p1", s1Updated.getProjectId());
        Assert.assertEquals("metric", s1Updated.getMetricNameLabel());
        Assert.assertEquals(1, s1Updated.getVersion());
        Assert.assertEquals(s1.getCreatedAt(), s1Updated.getCreatedAt());
        Assert.assertEquals("user2", s1Updated.getUpdatedBy());
    }

    @Test
    public void deleteOne() {
        String user1 = "user1";

        Service s1 = newService("s1", "p1", "", user1);
        Assert.assertTrue(insertSync(s1));

        Assert.assertTrue(deleteOneSync(s1.getProjectId(), s1.getId()));
        Assert.assertFalse(findOneSync(s1.getProjectId(), s1.getId()).isPresent());

        Assert.assertFalse(deleteOneSync("p1", "someOtherId"));
        Assert.assertFalse(deleteOneSync("someOtherId", "s1"));
    }

    @Test
    public void deleteOneWithFolder() {
        String user1 = "user1";

        Service s1 = newService("s1", "p1", "f1", user1);
        Assert.assertTrue(insertSync(s1));

        Assert.assertTrue(deleteOneSync(s1.getProjectId(), s1.getFolderId(), s1.getId()));
        Assert.assertFalse(findOneSync(s1.getProjectId(), s1.getFolderId(), s1.getId()).isPresent());

        Assert.assertFalse(deleteOneSync("p1", "f1", "someOtherId"));
        Assert.assertFalse(deleteOneSync("p1", "other", "s1"));
        Assert.assertFalse(deleteOneSync("other", "other", "s1"));
    }

    @Test
    public void insertFindAndDelete() {
        Service.Builder builder = simpleService();
        Service solomonA = builder.setId("A").setProjectId("Solomon").build();
        Service solomonB = builder.setId("B").setProjectId("Solomon").build();
        Service solomonC = builder.setId("C").setProjectId("Solomon").build();
        Service otherX = builder.setId("X").setProjectId("Other").build();

        assertTrue(insertSync(solomonA));
        assertTrue(insertSync(solomonB));
        assertTrue(insertSync(solomonC));
        assertTrue(insertSync(otherX));

        Map<String, Service> byId = byId(findAllSync(), Service::getId);
        assertEquals(4, byId.size());
        equalsByString(solomonA, byId.get(solomonA.getId()));
        equalsByString(solomonB, byId.get(solomonB.getId()));
        equalsByString(solomonC, byId.get(solomonC.getId()));
        equalsByString(otherX, byId.get(otherX.getId()));

        assertTrue(existsSync("Solomon", "A"));
        assertFalse(existsSync("Solomon", "D"));
        assertFalse(existsSync("Other", "A"));

        join(getServicesDao().deleteByProjectId("Solomon", ""));

        List<Service> serviceList = findAllSync();
        assertEquals(1, serviceList.size());
        equalsByString(otherX, serviceList.get(0));
    }

    @Test
    public void insertFindAndDeleteWithFolder() {
        Service.Builder builder = simpleService();
        Service folderA = builder.setId("A").setProjectId("Solomon").setFolderId("folder").build();
        Service folderB = builder.setId("B").setProjectId("Solomon").setFolderId("folder").build();
        Service folderC = builder.setId("C").setProjectId("Solomon").setFolderId("folder").build();
        Service otherX = builder.setId("X").setProjectId("Solomon").setFolderId("other").build();

        assertTrue(insertSync(folderA));
        assertTrue(insertSync(folderB));
        assertTrue(insertSync(folderC));
        assertTrue(insertSync(otherX));

        Map<String, Service> byId = byId(findAllSync(), Service::getId);
        assertEquals(4, byId.size());
        equalsByString(folderA, byId.get(folderA.getId()));
        equalsByString(folderB, byId.get(folderB.getId()));
        equalsByString(folderC, byId.get(folderC.getId()));
        equalsByString(otherX, byId.get(otherX.getId()));

        assertTrue(existsSync("Solomon", "folder", "A"));
        assertFalse(existsSync("Solomon", "folder", "D"));
        assertFalse(existsSync("Solomon", "other", "A"));

        join(getServicesDao().deleteByProjectId("Solomon", "folder"));

        List<Service> serviceList = findAllSync();
        assertEquals(1, serviceList.size());
        equalsByString(otherX, serviceList.get(0));
    }

    @Test
    public void findAllPaged() {
        Service.Builder builder = simpleService();
        insertSync(builder.setId("service1").setName("Service1").setProjectId("air").build());
        insertSync(builder.setId("service2").setName("Service2").setProjectId("air").build());
        insertSync(builder.setId("service3").setName("Service3").setProjectId("solomon").setFolderId("folder1").build());
        insertSync(builder.setId("service4").setName("Service4").setProjectId("solomon").setFolderId("folder2").build());

        List<Service> services = join(getServicesDao().findAll(new PageOptions(2, 0), "")).getResult();
        List<String> clusterIds = services.stream()
            .map(Service::getId)
            .collect(Collectors.toList());

        Assert.assertEquals(clusterIds, List.of("service1", "service2"));
    }

    @Test
    public void findAllPagedWithText() {
        Service.Builder builder = simpleService();
        insertSync(builder.setId("service1").setName("Service1").setProjectId("solomon").setFolderId("folder1").build());
        insertSync(builder.setId("service2").setName("Service2").setProjectId("solomon").setFolderId("folder2").build());
        insertSync(builder.setId("service10").setName("Service10").setProjectId("air").build());
        insertSync(builder.setId("service12").setName("Service12").setProjectId("air").build());

        List<Service> services = join(getServicesDao().findAll(new PageOptions(2, 0), "1")).getResult();
        List<String> serviceIds = services.stream()
            .map(Service::getId)
            .collect(Collectors.toList());

        Assert.assertEquals(serviceIds, List.of("service1", "service10"));
    }

    @Test
    public void changeGrid() {
        var init = simpleService().setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                null,
                1,
                37,
                DecimPolicy.UNDEFINED,
                ShardSettings.AggregationSettings.of(true, new AggrRule[0], false),
                0)).build();
        insertSync(init);

        var v1 = findOneSync(init.getProjectId(), init.getId()).get();
        assertEquals(1, v1.getShardSettings().getGrid());

        getServicesDao().partialUpdate(v1.toBuilder().setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                null,
                30,
                37,
                DecimPolicy.UNDEFINED,
                ShardSettings.AggregationSettings.of(true, new AggrRule[0], false),
                20)).build()).join();

        var v2 = findOneSync(init.getProjectId(), init.getId()).get();
        assertEquals(30, v2.getShardSettings().getGrid());
        assertEquals(20, v2.getShardSettings().getInterval());

        getServicesDao().partialUpdate(v2.toBuilder().setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                null,
                60,
                37,
                DecimPolicy.UNDEFINED,
                ShardSettings.AggregationSettings.of(true, new AggrRule[0], false),
                0)).build()).join();

        var v3 = findOneSync(init.getProjectId(), init.getId()).get();
        assertEquals(60, v3.getShardSettings().getGrid());
        assertEquals(0, v3.getShardSettings().getInterval());
    }

    @Test
    public void gridDefault() {
        var service = Service.newBuilder()
                .setId("myServiceId")
                .setName("myServiceName")
                .setProjectId("projectId")
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        0))
                .build();
        insertSync(service);

        var result = findOneSync(service.getProjectId(), service.getId()).get();
        assertEquals(Service.GRID_UNKNOWN, result.getShardSettings().getGrid());
    }

    @Test
    public void serviceProvider() {
        var service = Service.newBuilder()
                .setId("serviceWithSp")
                .setName("myServiceName")
                .setProjectId("myProjectId")
                .setServiceProvider("test-service-provider")
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                        null,
                        15,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new AggrRule[0], false),
                        15))
                .build();
        insertSync(service);


        var result = findOneSync(service.getProjectId(), service.getId());
        assertTrue(result.isPresent());
        assertEquals(service, result.get());
        assertEquals("test-service-provider", result.get().getServiceProvider());
    }

    @Test
    public void aggregationFunction() {
        ServiceMetricConf conf = ServiceMetricConf.of(new AggrRule[]{
                AggrRule.of("host=*, name=*.size", "host=cluster", MetricAggregation.LAST)
        }, true);
        var service = simpleService()
                .setMetricConf(conf)
                .setShardSettings(simpleShardSettings(conf))
                .build();
        insertSync(service);

        var result = findOneSync(service.getProjectId(), service.getId());
        assertEquals(Optional.of(service), result);
    }

    @Test
    public void unableUpdateServiceProvider() {
        var v1 = Service.newBuilder()
                .setId("serviceWithSp")
                .setName("myServiceName")
                .setProjectId("myProjectId")
                .setServiceProvider("test-service-provider")
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.EMPTY,
                        0))
                .build();
        insertSync(v1);

        getServicesDao().partialUpdate(v1.toBuilder().setServiceProvider("changed").build()).join();

        var result = findOneSync(v1.getProjectId(), v1.getId()).get();
        assertEquals("test-service-provider", result.getServiceProvider());
    }

    private boolean insertSync(Service service) {
        return join(getServicesDao().insert(service));
    }

    private boolean existsSync(String projectId, String serviceId) {
        return join(getServicesDao().exists(projectId, "", serviceId));
    }

    private boolean existsSync(String projectId, String folderId, String serviceId) {
        return join(getServicesDao().exists(projectId, folderId, serviceId));
    }

    private PagedResult<Service> findByProjectIdSync(String projectId, PageOptions pageOpts, String text, ShardSettings.Type model) {
        return join(getServicesDao().findByProjectId(projectId, "", pageOpts, text, model));
    }

    private TokenBasePage<Service> findByProjectIdPagedSync(String projectId, int pageSize, String token, String text) {
        return join(getServicesDao().findByProjectIdPaged(projectId, "", pageSize, token, text));
    }

    private PagedResult<Service> findByFolderIdSync(String projectId, String folderId, PageOptions pageOpts, String text) {
        return join(getServicesDao().findByProjectId(projectId, folderId, pageOpts, text, ShardSettings.Type.UNSPECIFIED));
    }

    private Optional<Service> findOneSync(String projectId, String serviceId) {
        return join(getServicesDao().findOne(projectId, "", serviceId));
    }

    private Optional<Service> findOneSync(String projectId, String folderId, String serviceId) {
        return join(getServicesDao().findOne(projectId, folderId, serviceId));
    }

    private List<Service> findAllSync() {
        return join(getServicesDao().findAll());
    }

    private boolean deleteOneSync(String projectId, String serviceId) {
        return join(getServicesDao().deleteOne(projectId, "", serviceId));
    }

    private boolean deleteOneSync(String projectId, String folderId, String serviceId) {
        return join(getServicesDao().deleteOne(projectId, folderId, serviceId));
    }

    private static Service.Builder simpleService() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return Service
            .newBuilder()
            .setId("Service1")
            .setProjectId("Project1")
            .setName("Name1")
            .setDescription("Description")
            .setAddTsArgs(true)
            .setMetricConf(simpleMetricConf())
            .setShardSettings(simpleShardSettings(null))
            .setLabels(Map.of("label1", "value1", "label2", "value2"))
            .setMetricNameLabel("sensor")
            .setCreatedAt(now)
            .setCreatedBy("user1")
            .setUpdatedAt(now.plus(Duration.ofHours(2)))
            .setUpdatedBy("user2")
            .setVersion(1);
    }

    private static ServiceMetricConf simpleMetricConf() {
        return ServiceMetricConf.of(
            new ServiceMetricConf.AggrRule[]{ simpleAggrRule() },
            true);
    }

    private static ShardSettings simpleShardSettings(ServiceMetricConf conf) {
        var pullSettings = ShardSettings.PullSettings.newBuilder()
                .setPort(12345)
                .setPath("Path1")
                .setAddTsArgs(true)
                .setProtocol(ShardSettings.PullProtocol.HTTP)
                .setTvmDestinationId("tvm")
                .build();

        var aggregationSettings = ShardSettings.AggregationSettings.of(true,
                new ServiceMetricConf.AggrRule[]{simpleAggrRule()}, true);
        if (conf != null) {
            aggregationSettings = ShardSettings.AggregationSettings.of(true,
                    conf.getAggrRules(), conf.isRawDataMemOnly());
        }

        return ShardSettings.of(
                ShardSettings.Type.PULL,
                pullSettings,
                15,
                23,
                DecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS,
                aggregationSettings,
                123
        );
    }

    private static ServiceMetricConf.AggrRule simpleAggrRule() {
        return AggrRule.of("host=*", "host=cluster", null);
    }

    private static void assertPartialEquality(Service service, Service fromDb) {
        Assert.assertEquals(service.getId(), fromDb.getId());
        Assert.assertEquals(service.getName(), fromDb.getName());
        Assert.assertEquals(service.getProjectId(), fromDb.getProjectId());
    }

    private static Service newService(String id, String projectId, String folderId, String user) {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return Service.newBuilder()
                .setId(id)
                .setName(StringUtils.capitalize(id))
                .setDescription("description for " + id)
                .setProjectId(projectId)
                .setFolderId(folderId)
                .setCreatedBy(user)
                .setUpdatedBy(user)
                .setMetricNameLabel("sensor")
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.PUSH,
                        null,
                        10,
                        37,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new AggrRule[0], false),
                        10))
                .build();
    }
}
