package ru.yandex.solomon.core.db.dao;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static java.lang.Boolean.FALSE;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.byId;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.equalsByString;
import static ru.yandex.solomon.core.db.model.ShardState.ACTIVE;
import static ru.yandex.solomon.core.db.model.ShardState.INACTIVE;
import static ru.yandex.solomon.core.db.model.ShardState.READ_ONLY;
import static ru.yandex.solomon.core.db.model.ShardState.RW;
import static ru.yandex.solomon.core.db.model.ShardState.WRITE_ONLY;


/**
 * @author Sergey Polovko
 */
@YaExternal
public abstract class AbstractShardsDaoTest {

    private static final AtomicInteger counter = new AtomicInteger();

    protected abstract ShardsDao getShardsDao();

    @Test
    public void insert() {
        Shard shard = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW)
                        .toBuilder()
                            .setNumPartitions(75)
                            .build();
        assertTrue(insertSync(shard));

        Optional<Shard> fromDb = findOneSync(shard.getProjectId(), shard.getId());
        assertTrue(fromDb.isPresent());
        Assert.assertEquals(shard, fromDb.get());
    }

    private Optional<Shard> findOneSync(String projectId, String shardId) {
        return join(getShardsDao().findOne(projectId, "", shardId));
    }

    private Optional<Shard> findOneSync(String projectId, String folderId, String shardId) {
        return join(getShardsDao().findOne(projectId, folderId, shardId));
    }

    private boolean deleteOneSync(String projectId, String shardId) {
        return join(getShardsDao().deleteOne(projectId, "", shardId));
    }

    private boolean deleteOneSync(String projectId, String folderId, String shardId) {
        return join(getShardsDao().deleteOne(projectId, "", shardId));
    }

    private boolean insertSync(Shard shard) {
        return join(getShardsDao().insert(shard));
    }

    private List<Shard> findByClusterIdSync(String projectId, String clusterId) {
        return join(getShardsDao().findByClusterId(projectId, "", clusterId));
    }

    private List<Shard> findByClusterIdSync(String projectId, String folderId, String clusterId) {
        return join(getShardsDao().findByClusterId(projectId, folderId, clusterId));
    }

    private TokenBasePage<Shard> findByClusterIdSync(
            String projectId, String clusterId,
            Set<ShardState> states, String filter,
            int pageSize, String pageToken) {
        return join(getShardsDao().findByClusterIdV3(projectId, "", clusterId, states, filter, pageSize, pageToken));
    }

    private TokenBasePage<Shard> findByServiceIdSync(
            String projectId, String serviceId,
            Set<ShardState> states, String filter,
            int pageSize, String pageToken) {
        return join(getShardsDao().findByServiceIdV3(projectId, "", serviceId, states, filter, pageSize, pageToken));
    }


    private List<Shard> findByServiceIdSync(String projectId, String folderId, String clusterId) {
        return join(getShardsDao().findByServiceId(projectId, folderId, clusterId));
    }

    private PagedResult<Shard> findByProjectIdSync(String projectId, PageOptions pageOpts, ShardState state, String text) {
        return join(getShardsDao().findByProjectId(projectId, "", pageOpts, EnumSet.of(state), text));
    }

    private TokenBasePage<Shard> findByProjectIdV3Sync(String projectId, int pageSize, String pageToken, String text) {
        return join(getShardsDao().findByProjectIdV3(projectId, "", pageSize, pageToken, text));
    }

    private PagedResult<Shard> findByFolderIdSync(String projectId, String folderId, PageOptions pageOpts, ShardState state, String text) {
        return join(getShardsDao().findByProjectId(projectId, "", pageOpts, EnumSet.of(state), text));
    }

    private List<Shard> findByProjectIdSync(String projectId) {
        return join(getShardsDao().findByProjectId(projectId, ""));
    }

    private List<Shard> findByFolderIdSync(String projectId) {
        return join(getShardsDao().findByProjectId(projectId, ""));
    }

    private boolean existsSync(String projectId, String shardId) {
        return join(getShardsDao().exists(projectId, "", shardId));
    }

    private boolean existsSync(String projectId, String folderId, String shardId) {
        return join(getShardsDao().exists(projectId, "", shardId));
    }

    private Optional<Shard> partialUpdateSync(Shard shard, boolean canUpdateInternals) {
        return join(getShardsDao().partialUpdate(shard, canUpdateInternals));
    }

    private void patchWithClusterNameSync(String projectId, String clusterId, String clusterName) {
        join(getShardsDao().patchWithClusterName(projectId, clusterId, clusterName));
    }

    private void patchWithServiceNameSync(String projectId, String serviceId, String serviceName) {
        join(getShardsDao().patchWithServiceName(projectId, serviceId, serviceName));
    }

    @Test
    public void findByClusterId() {
        Shard shard1 = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "", "cluster1", "service2", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "", "cluster2", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "", "cluster1", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        {
            List<Shard> shards = findByClusterIdSync("project1", "cluster1");
            Assert.assertEquals(2, shards.size());
            assertPartiallyEqual(shard1, shards.get(0));
            assertPartiallyEqual(shard2, shards.get(1));
        }
        {
            List<Shard> shards = findByClusterIdSync("project1", "cluster2");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard3, shards.get(0));
        }
        {
            List<Shard> shards = findByClusterIdSync("project1", "cluster3");
            assertTrue(shards.isEmpty());
        }
        {
            List<Shard> shards = findByClusterIdSync("project2", "cluster1");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard4, shards.get(0));
        }

        assertTrue(deleteOneSync("project1", "shard2"));

        {
            List<Shard> shards = findByClusterIdSync("project1", "cluster1");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard1, shards.get(0));
        }
    }

    @Test
    public void findByClusterIdWithFolder() {
        Shard shard1 = newShard("shard1", "project1", "folder1", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "folder1", "cluster1", "service2", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "folder1", "cluster2", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "folder2", "cluster1", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        {
            List<Shard> shards = findByClusterIdSync("project1", "folder1", "cluster1");
            Assert.assertEquals(2, shards.size());
            assertPartiallyEqual(shard1, shards.get(0));
            assertPartiallyEqual(shard2, shards.get(1));
        }
        {
            List<Shard> shards = findByClusterIdSync("project1", "folder1", "cluster2");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard3, shards.get(0));
        }
        {
            List<Shard> shards = findByClusterIdSync("project1", "folder1", "cluster3");
            assertTrue(shards.isEmpty());
        }
        {
            List<Shard> shards = findByClusterIdSync("project2", "folder2", "cluster1");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard4, shards.get(0));
        }

        assertTrue(deleteOneSync("project1", "folder1", "shard2"));

        {
            List<Shard> shards = findByClusterIdSync("project1", "folder1", "cluster1");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard1, shards.get(0));
        }
    }

    private static void assertPartiallyEqual(Shard shard1, Shard shard2) {
        Assert.assertEquals(shard1.getId(), shard2.getId());
        Assert.assertEquals(shard1.getProjectId(), shard2.getProjectId());
        Assert.assertEquals(shard1.getFolderId(), shard2.getFolderId());
        Assert.assertEquals(shard1.getNumId(), shard2.getNumId());
        Assert.assertEquals(shard1.getClusterId(), shard2.getClusterId());
        Assert.assertEquals(shard1.getClusterName(), shard2.getClusterName());
        Assert.assertEquals(shard1.getServiceId(), shard2.getServiceId());
        Assert.assertEquals(shard1.getServiceName(), shard2.getServiceName());
    }

    @Test
    public void findByServiceId() {
        Shard shard1 = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "", "cluster2", "service1", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "", "cluster1", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "", "cluster2", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        {
            List<Shard> shards = findByServiceIdSync("project1", "", "service1");
            Assert.assertEquals(2, shards.size());
            assertPartiallyEqual(shard1, shards.get(0));
            assertPartiallyEqual(shard2, shards.get(1));
        }
        {
            List<Shard> shards = findByServiceIdSync("project1", "", "service2");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard3, shards.get(0));
        }

        assertTrue(findByServiceIdSync("project1", "", "service3").isEmpty());
        assertTrue(deleteOneSync("project1", "shard2"));

        {
            List<Shard> shards = findByServiceIdSync("project1", "", "service2");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard3, shards.get(0));
        }
    }

    @Test
    public void findByServiceIdV3() {
        Shard shard1 = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "", "cluster2", "service1", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "", "cluster1", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "", "cluster2", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        {
            // Find associated clusters for service1.
            TokenBasePage<Shard> shards = findByServiceIdSync("project1", "service1", Set.of(), "", 2, "");
            Assert.assertEquals(2, shards.getItems().size());
            assertPartiallyEqual(shard1, shards.getItems().get(0));
            assertPartiallyEqual(shard2, shards.getItems().get(1));
            assertEquals("", shards.getNextPageToken());
        }

        {
            // Find first page of associated clusters for service1 .
            TokenBasePage<Shard> shards = findByServiceIdSync("project1", "service1", Set.of(), "", 1, "");
            Assert.assertEquals(1, shards.getItems().size());
            assertPartiallyEqual(shard1, shards.getItems().get(0));
            assertEquals("1", shards.getNextPageToken());
        }

        {
            // Find second page of associated clusters for project1/service1.
            TokenBasePage<Shard> shards = findByServiceIdSync("project1", "service1", Set.of(), "1", 1, "");
            Assert.assertEquals(1, shards.getItems().size());
            assertEquals("", shards.getNextPageToken());
        }

        {
            // Find associated clusters for project1/service2 (project2/service2 is ignored).
            TokenBasePage<Shard> shards = findByServiceIdSync("project1", "service2", Set.of(), "", 2, "");
            Assert.assertEquals(1, shards.getItems().size());
            assertPartiallyEqual(shard3, shards.getItems().get(0));
            assertEquals("", shards.getNextPageToken());
        }

        {
            // Find associated clusters with mode=READ_ONLY for service1.
            TokenBasePage<Shard> shards = findByServiceIdSync("project1", "service1", Set.of(READ_ONLY), "", 1, "");
            Assert.assertEquals(1, shards.getItems().size());
            assertPartiallyEqual(shard2, shards.getItems().get(0));
            assertEquals("", shards.getNextPageToken());
        }

        {
            // Find second page of associated clusters with name containing "2" for service1.
            TokenBasePage<Shard> shards = findByServiceIdSync("project1", "service1", Set.of(), "2", 1, "");
            Assert.assertEquals(1, shards.getItems().size());
            assertPartiallyEqual(shard2, shards.getItems().get(0));
            assertEquals("", shards.getNextPageToken());
        }

        assertTrue(findByServiceIdSync("project1", "", "service3").isEmpty());
    }

    @Test
    public void findByServiceIdWithFolder() {
        Shard shard1 = newShard("shard1", "project1", "folder1", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "folder1", "cluster2", "service1", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "folder1", "cluster1", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "folder2", "cluster2", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        {
            List<Shard> shards = findByServiceIdSync("project1", "folder1", "service1");
            Assert.assertEquals(2, shards.size());
            assertPartiallyEqual(shard1, shards.get(0));
            assertPartiallyEqual(shard2, shards.get(1));
        }
        {
            List<Shard> shards = findByServiceIdSync("project1", "folder1", "service2");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard3, shards.get(0));
        }

        assertTrue(findByServiceIdSync("project1", "folder1", "service3").isEmpty());
        assertTrue(deleteOneSync("project1", "shard2"));

        {
            List<Shard> shards = findByServiceIdSync("project1", "folder1", "service2");
            Assert.assertEquals(1, shards.size());
            assertPartiallyEqual(shard3, shards.get(0));
        }
    }

    @Test
    public void partialUpdateWithInvalidVersion() {
        int version = 10;
        Shard.Builder builder = shard().setId("A").setProjectId("Solomon").setVersion(version);
        Shard initial = builder.build();
        assertTrue(insertSync(builder.build()));

        //current version > update version
        builder.setVersion(version - 1);
        assertFalse(partialUpdateSync(builder.build(), false).isPresent());
        Shard foundShard = findOneSync(initial.getProjectId(), initial.getId()).orElseThrow(AssertionError::new);
        equalsByString(initial, foundShard);
    }

    @Test
    public void partialUpdateWithoutVersion() {
        int version = 10;
        Shard.Builder builder = shard().setId("A").setProjectId("Solomon").setVersion(version);
        assertTrue(insertSync(builder.build()));
        builder.setVersion(-1);
        Optional<Shard> updatedShard = partialUpdateSync(builder.build(), false);
        assertTrue(updatedShard.isPresent());
        assertEquals(version + 1, updatedShard.get().getVersion());
    }

    @Test
    public void partialUpdateWithoutPermissions() {
        Shard.Builder builder = shard();
        Shard initial = builder.build();
        assertTrue(insertSync(initial));

        Shard shardToUpdate = builder
            .setMaxFileMetrics(builder.getMaxFileMetrics() + 1000)
            .setMaxMemMetrics(builder.getMaxMemMetrics() + 1000)
            .setMaxMetricsPerUrl(builder.getMaxMetricsPerUrl() + 1000)
            .setMaxResponseSizeBytes(builder.getMaxResponseSizeBytes() + 1000)
            .setNumPartitions(builder.getNumPartitions() + 100)
            // Not internal field
            .setDescription("descr")
            .setShardSettings(shardSettings(DecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS))
            .setValidationMode(ValidationMode.STRICT_FAIL)
            .build();

        assertTrue(partialUpdateSync(shardToUpdate, false).isPresent());
        Shard foundShard = findOneSync(initial.getProjectId(), initial.getId()).orElseThrow(AssertionError::new);

        // Not internal field
        assertEquals(shardToUpdate.getDescription(), foundShard.getDescription());

        assertEquals(initial.getMaxMetricsPerUrl(), foundShard.getMaxMetricsPerUrl());
        assertEquals(initial.getMaxResponseSizeBytes(), foundShard.getMaxResponseSizeBytes());
        assertEquals(initial.getMaxFileMetrics(), foundShard.getMaxFileMetrics());
        assertEquals(initial.getMaxMemMetrics(), foundShard.getMaxMemMetrics());
        assertEquals(initial.getValidationMode(), foundShard.getValidationMode());
        assertEquals(initial.getNumPartitions(), foundShard.getNumPartitions());
        assertEquals(initial.getShardSettings().getRetentionPolicy(), DecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS);
    }

    @Test
    public void partialUpdateHosts() {
        Shard.Builder builder = shard();
        Shard initial = builder.build();
        assertTrue(insertSync(initial));

        Shard shardToUpdate = builder.build();
        assertTrue(partialUpdateSync(shardToUpdate, false).isPresent());
        Shard foundShard = findOneSync(initial.getProjectId(), initial.getId()).orElseThrow(AssertionError::new);
    }

    @Test
    public void partialUpdateInternals() {
        Shard.Builder builder = shard();
        Shard initial = builder.build();
        assertTrue(insertSync(initial));

        Shard shardToUpdate = builder
            .setMaxFileMetrics(builder.getMaxFileMetrics() + 1000)
            .setMaxMemMetrics(builder.getMaxMemMetrics() + 1000)
            .setMaxMetricsPerUrl(builder.getMaxMetricsPerUrl() + 1000)
            .setMaxResponseSizeBytes(builder.getMaxResponseSizeBytes() + 1000)
            .setNumPartitions(builder.getNumPartitions() + 10)
            // Not internal field
            .setDescription("descr")
            .setValidationMode(ValidationMode.STRICT_FAIL)
            .setShardSettings(shardSettings(DecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS))
            .build();

        assertTrue(partialUpdateSync(shardToUpdate, true).isPresent());
        Shard foundShard = findOneSync(initial.getProjectId(), initial.getId()).orElseThrow(AssertionError::new);

        // Not internal field
        assertEquals(shardToUpdate.getDescription(), foundShard.getDescription());

        assertEquals(shardToUpdate.getMaxMetricsPerUrl(), foundShard.getMaxMetricsPerUrl());
        assertEquals(shardToUpdate.getMaxResponseSizeBytes(), foundShard.getMaxResponseSizeBytes());
        assertEquals(shardToUpdate.getMaxFileMetrics(), foundShard.getMaxFileMetrics());
        assertEquals(shardToUpdate.getMaxMemMetrics(), foundShard.getMaxMemMetrics());
        assertEquals(shardToUpdate.getValidationMode(), foundShard.getValidationMode());
        assertEquals(shardToUpdate.getShardSettings().getRetentionPolicy(), DecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS);
        assertEquals(shardToUpdate.getNumPartitions(), foundShard.getNumPartitions());
    }

    @Test
    public void patchWithClusterName() {
        assertTrue(insertSync(newShard("shard1", "project", "", "cluster", "service", "jamel", RW)));
        patchWithClusterNameSync("project", "cluster", "new_cluster_name");
        Optional<Shard> shardOpt = findOneSync("project", "shard1");
        if (!shardOpt.isPresent()) {
            throw new AssertionError("shard doesn't find");
        }
        Assert.assertEquals("new_cluster_name", shardOpt.get().getClusterName());
    }

    @Test
    public void patchWithServiceName() {
        assertTrue(insertSync(newShard("shard1", "project", "", "cluster", "service", "jamel", RW)));
        patchWithServiceNameSync("project", "service", "new_service_name");
        Optional<Shard> shardOpt = findOneSync("project", "shard1");
        if (!shardOpt.isPresent()) {
            throw new AssertionError("shard doesn't find");
        }
        Assert.assertEquals("new_service_name", shardOpt.get().getServiceName());
    }

    @Test
    public void insertDuplicate() {
        Shard shardA = shard().setId("A").build();
        Shard otherA = shard().setId("A").build();
        assertTrue(insertSync(shardA));
        assertFalse(insertSync(otherA));

        Shard shardB = shard()
            .setId("B")
            .setProjectId(shardA.getProjectId())
            .setClusterName(shardA.getClusterName())
            .setServiceName(shardA.getServiceName())
            .build();
        assertFalse(insertSync(shardB));
    }

    @Test
    public void insertAndFind() {
        String solomon = "Solomon";
        Shard shardA = shard().setProjectId(solomon).setId("AЯ").build();
        Shard shardB1 = shard().setProjectId(solomon).setId("B1").setClusterId("ClusterЯ").build();
        Shard shardB2 = shard().setProjectId(solomon).setId("B2").setState(RW).setClusterId("ClusterЯ").build();
        Shard shardC = shard().setProjectId(solomon).setId("C").setState(INACTIVE).setServiceId("ServiceЯ").build();
        Shard shardD = shard().setProjectId(solomon).setId("D").setClusterName("NameЯ").build();
        Shard shardE = shard().setProjectId(solomon).setId("E").setServiceName("NameЯ").build();
        Shard shardF = shard().setProjectId(solomon).setId("F").build();
        Shard shardG = shard().setProjectId(solomon).setId("G").build();
        Shard otherH = shard().setProjectId("Other").setId("HЯ").setState(RW).build();

        List<Shard> solomonShards = Arrays.asList(shardA, shardB1, shardB2, shardC, shardD, shardE, shardF, shardG);
        List<Shard> solomonYaShards = Arrays.asList(shardA, shardB1, shardB2, shardC, shardD, shardE);

        assertTrue(solomonShards.stream().map(this::insertSync).filter(FALSE::equals).findFirst().orElse(true));
        assertTrue(insertSync(otherH));

        Shard found = findOneSync(shardA.getProjectId(), shardA.getId()).orElseThrow(AssertionError::new);
        equalsByString(shardA, found);

        List<Shard> shards = findByProjectIdSync("Other");
        assertEquals(1, shards.size());
        assertEquals(otherH.getId(), shards.get(0).getId());

        shards = findByProjectIdSync(solomon);
        assertEquals(solomonShards.size(), shards.size());
        assertEquals(
            solomonShards.stream().map(Shard::getId).collect(Collectors.toSet()),
            shards.stream().map(Shard::getId).collect(Collectors.toSet()));

        assertTrue(existsSync(shardA.getProjectId(), shardA.getId()));
        assertFalse(existsSync("Ж", shardA.getId()));
        assertFalse(existsSync(shardA.getProjectId(), "Ж"));

        Map<String, Shard> byId = byId(findByClusterIdSync(solomon, "ClusterЯ"), Shard::getId);
        assertEquals(2, byId.size());
        assertPartiallyEqual(shardB1, byId.get(shardB1.getId()));
        assertPartiallyEqual(shardB2, byId.get(shardB2.getId()));

        List<Shard> byService = findByServiceIdSync(solomon, "", "ServiceЯ");
        assertEquals(1, byService.size());
        assertPartiallyEqual(shardC, byService.get(0));

        PagedResult<Shard> pagedResult = findByProjectIdSync(solomon, PageOptions.ALL, INACTIVE, "");
        assertEquals(solomonShards.size() - 1, pagedResult.getTotalCount());
        List<Shard> result = pagedResult.getResult();
        assertEquals(solomonShards.size() - 1, result.size());
        equalsByString(partial(shardA).build(), result.get(0));
        equalsByString(partial(shardB1).build(), result.get(1));
        equalsByString(partial(shardC).build(), result.get(2));

        pagedResult = findByProjectIdSync(solomon, new PageOptions(3, 0), INACTIVE, "Я");
        assertEquals(solomonYaShards.size() - 1, pagedResult.getTotalCount());
        result = pagedResult.getResult();
        assertEquals(3, result.size());
        Iterator<Shard> si = result.iterator();
        equalsByString(partial(shardA).build(), si.next());
        equalsByString(partial(shardB1).build(), si.next());
        equalsByString(partial(shardC).build(), si.next());

        pagedResult = findByProjectIdSync(solomon, new PageOptions(3, 1), INACTIVE, "Я");
        assertEquals(solomonYaShards.size() - 1, pagedResult.getTotalCount());
        result = pagedResult.getResult();
        assertEquals(2, result.size());
        si = result.iterator();
        equalsByString(partial(shardD).build(), si.next());
        equalsByString(partial(shardE).build(), si.next());

        pagedResult = findByProjectIdSync(solomon, new PageOptions(3, 2), INACTIVE, "Я");
        assertEquals(solomonYaShards.size() - 1, pagedResult.getTotalCount());
        result = pagedResult.getResult();
        assertTrue(result.isEmpty());

        pagedResult = findByProjectIdSync(solomon, PageOptions.ALL, RW, "Я");
        assertEquals(1, pagedResult.getTotalCount());
        result = pagedResult.getResult();
        assertEquals(1, result.size());
        equalsByString(partial(shardB2).build(), result.get(0));
        pagedResult = join(getShardsDao().findAll(PageOptions.ALL, RW, "Я"));
        assertEquals(2, pagedResult.getTotalCount());
        result = pagedResult.getResult();
        assertEquals(2, result.size());
        equalsByString(partial(shardB2).build(), result.get(0));
        equalsByString(partial(otherH).build(), result.get(1));
        Map<String, Shard> activeById = byId(join(getShardsDao().findAllNotInactive()), Shard::getId);
        assertEquals(2, activeById.size());
        equalsByString(shardB2, activeById.get(shardB2.getId()));
        equalsByString(otherH, activeById.get(otherH.getId()));

        pagedResult = join(getShardsDao().findAll(PageOptions.MAX, RW, ""));
        assertEquals(2, pagedResult.getTotalCount());
    }

    @Test
    public void extraPatchClusterName() {
        Shard shard111 = name("111", "P1", "C1", "S1", "CA", "SA").build();
        Shard shard112 = name("112", "P1", "C1", "S2", "CA", "SB").build();
        Shard shard121 = name("121", "P1", "C2", "S1", "CB", "SA").build();
        Shard shard122 = name("122", "P1", "C2", "S2", "CB", "SB").build();
        Shard shard211 = name("211", "P2", "C1", "S1", "CA", "SA").build();
        Shard shard212 = name("212", "P2", "C1", "S2", "CA", "SB").build();
        Shard shard221 = name("221", "P2", "C2", "S1", "CB", "SA").build();
        Shard shard222 = name("222", "P2", "C2", "S2", "CB", "SB").build();

        List<Shard> shards =
            Arrays.asList(shard111, shard112, shard121, shard122, shard211, shard212, shard221, shard222);
        for (Shard shard : shards) {
            assertTrue(insertSync(shard));
        }

        patchWithClusterNameSync(shard111.getProjectId(), shard111.getClusterId(), "CC");

        Shard shard111New = shard111.toBuilder().setClusterName("CC").build();
        equalsByString(shard111New, findOneSync(shard111.getProjectId(), shard111.getId()).orElseThrow(AssertionError::new));

        Shard shard112New = shard112.toBuilder().setClusterName("CC").build();
        equalsByString(shard112New,
            findOneSync(shard112.getProjectId(), shard112.getId()).orElseThrow(AssertionError::new));

        for (Shard shard : shards.subList(2, shards.size())) {
            equalsByString(shard,
                findOneSync(shard.getProjectId(), shard.getId()).orElseThrow(AssertionError::new));
        }

        try {
            patchWithClusterNameSync(shard121.getProjectId(), shard121.getClusterId(), "CC");
            Assert.fail("expected exception was not thrown");
        } catch (Exception e) {
            Throwable cause = CompletableFutures.unwrapCompletionException(e);
            if (!(cause instanceof ConflictException)) {
                throw e;
            }
        }
    }

    @Test
    public void extraPatchServiceName() {
        Shard shard111 = name("111", "P1", "C1", "S1", "CA", "SA").build();
        Shard shard112 = name("112", "P1", "C1", "S2", "CA", "SB").build();
        Shard shard121 = name("121", "P1", "C2", "S1", "CB", "SA").build();
        Shard shard122 = name("122", "P1", "C2", "S2", "CB", "SB").build();
        Shard shard211 = name("211", "P2", "C1", "S1", "CA", "SA").build();
        Shard shard212 = name("212", "P2", "C1", "S2", "CA", "SB").build();
        Shard shard221 = name("221", "P2", "C2", "S1", "CB", "SA").build();
        Shard shard222 = name("222", "P2", "C2", "S2", "CB", "SB").build();

        List<Shard> shards =
            Arrays.asList(shard111, shard112, shard121, shard122, shard211, shard212, shard221, shard222);
        for (Shard shard : shards) {
            assertTrue(insertSync(shard));
        }

        patchWithServiceNameSync(shard111.getProjectId(), shard111.getServiceId(), "SS");

        Shard shard111New = shard111.toBuilder().setServiceName("SS").build();
        equalsByString(shard111New,
            findOneSync(shard111.getProjectId(), shard111.getId()).orElseThrow(AssertionError::new));

        Shard shard121New = shard121.toBuilder().setServiceName("SS").build();
        equalsByString(shard121New,
            findOneSync(shard121.getProjectId(), shard121.getId()).orElseThrow(AssertionError::new));
        equalsByString(shard112,
            findOneSync(shard112.getProjectId(), shard112.getId()).orElseThrow(AssertionError::new));

        for (Shard shard : shards.subList(3, shards.size())) {
            equalsByString(shard,
                findOneSync(shard.getProjectId(), shard.getId()).orElseThrow(AssertionError::new));
        }

        try {
            patchWithServiceNameSync(shard112.getProjectId(), shard112.getServiceId(), "SS");
            Assert.fail("expected exception was not thrown");
        } catch (Exception e) {
            Throwable cause = CompletableFutures.unwrapCompletionException(e);
            if (!(cause instanceof ConflictException)) {
                throw e;
            }
        }

        equalsByString(shard112, findOneSync(shard112.getProjectId(), shard112.getId()).get());
        equalsByString(shard122, findOneSync(shard122.getProjectId(), shard122.getId()).get());
        equalsByString(shard111New, findOneSync(shard111.getProjectId(), shard111.getId()).get());
        equalsByString(shard121New, findOneSync(shard121.getProjectId(), shard121.getId()).get());
    }

    @Test
    public void deleteOne() {
        Shard solomonA = shard().setProjectId("Solomon").setId("A").build();
        Shard solomonB = shard().setProjectId("Solomon").setId("B").build();
        Shard otherX = shard().setProjectId("Other").setId("X").build();

        assertFalse(deleteOneSync(solomonA.getProjectId(), solomonA.getId()));

        assertTrue(insertSync(solomonA));
        assertTrue(insertSync(solomonB));
        assertTrue(insertSync(otherX));

        assertTrue(deleteOneSync(solomonA.getProjectId(), solomonA.getId()));
        assertFalse(findOneSync(solomonA.getProjectId(), solomonA.getId()).isPresent());
        assertFalse(deleteOneSync(solomonA.getProjectId(), solomonA.getId()));
        findOneSync(solomonB.getProjectId(), solomonB.getId()).orElseThrow(AssertionError::new);
        findOneSync(otherX.getProjectId(), otherX.getId()).orElseThrow(AssertionError::new);
    }

    @Test
    public void migrationToNewItemState() {
        String solomon = "Solomon";
        Shard shardA = shard().setProjectId(solomon).setId("AЯ").setState(RW).build();
        Shard shardB = shard().setProjectId(solomon).setId("B").setState(ACTIVE).build();
        Shard shardC = shard().setProjectId(solomon).setId("C").setState(ACTIVE).build();
        Shard shardD = shard().setProjectId(solomon).setId("D").setState(INACTIVE).build();
        Shard shardE = shard().setProjectId(solomon).setId("E").setState(INACTIVE).build();
        Shard shardF = shard().setProjectId(solomon).setId("F").setState(INACTIVE).build();
        Shard shardG = shard().setProjectId(solomon).setId("G").setState(READ_ONLY).build();
        Shard otherH = shard().setProjectId(solomon).setId("H").setState(READ_ONLY).build();

        List<Shard> solomonShards = Arrays.asList(shardA, shardB, shardC, shardD, shardE, shardF, shardG);

        assertTrue(solomonShards.stream().map(this::insertSync).filter(FALSE::equals).findFirst().orElse(true));
        assertTrue(insertSync(otherH));

        Shard found = findOneSync(shardA.getProjectId(), shardA.getId()).orElseThrow(AssertionError::new);
        equalsByString(shardA, found);

        {
            PagedResult<Shard> pagedResult = findByProjectIdSync(solomon, PageOptions.ALL, RW, "");
            // RW+ACTIVE
            assertEquals(3, pagedResult.getTotalCount());
        }
        {
            PagedResult<Shard> pagedResult = findByProjectIdSync(solomon, PageOptions.ALL, ACTIVE, "");
            // RW+ACTIVE
            assertEquals(3, pagedResult.getTotalCount());
        }
        {
            PagedResult<Shard> pagedResult = findByProjectIdSync(solomon, PageOptions.ALL, INACTIVE, "");
            assertEquals(3, pagedResult.getTotalCount());
        }
        {
            PagedResult<Shard> pagedResult = findByProjectIdSync(solomon, PageOptions.ALL, READ_ONLY, "");
            assertEquals(2, pagedResult.getTotalCount());
        }
    }

    @Test
    public void showAllNotInactive() {
        String projectId = "solomon";
        Shard shardA = shard().setProjectId(projectId).setId("A").setState(RW).build();
        Shard shardB = shard().setProjectId(projectId).setId("B").setState(ACTIVE).build();
        Shard shardC = shard().setProjectId(projectId).setId("C").setState(READ_ONLY).build();
        Shard shardD = shard().setProjectId(projectId).setId("D").setState(WRITE_ONLY).build();
        Shard shardE = shard().setProjectId(projectId).setId("E").setState(INACTIVE).build();

        List<Shard> solomonShards = Arrays.asList(shardA, shardB, shardC, shardD, shardE);

        assertTrue(solomonShards.stream().map(this::insertSync).filter(FALSE::equals).findFirst().orElse(true));

        List<Shard> found = getShardsDao().findAllNotInactive().join();

        List<String> foundShardIds = found.stream().map(Shard::getId).collect(Collectors.toList());

        assertThat(foundShardIds, allOf(
            iterableWithSize(4),
            hasItems("A", "B", "C", "D")
        ));
    }

    @Test
    public void findByKey() {
        List<String> hosts = Arrays.asList("host1", "host2");
        Shard shard1 = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "", "cluster1", "service2", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "", "cluster2", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "", "cluster1", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        {
            Shard shard = join(getShardsDao().findByShardKey("project1", "cluster1_name", "service1_name")).get();
            Assert.assertEquals(shard1, shard);
        }
        {
            Shard shard = join(getShardsDao().findByShardKey("project1", "cluster1_name", "service2_name")).get();
            Assert.assertEquals(shard2, shard);
        }
    }

    @Test
    public void numIdIsUnique() {
        Shard alice = Shard.newBuilder()
            .setNumId(42)
            .setId("alice")
            .setProjectId("solomon")
            .setClusterId("prod")
            .setClusterName("prod_name")
            .setServiceId("alice")
            .setServiceName("alice")
            .setShardSettings(shardSettings(DecimPolicy.DEFAULT))
            .build();

        Shard bob = Shard.newBuilder()
            .setNumId(43)
            .setId("bob")
            .setProjectId("solomon")
            .setClusterId("prod")
            .setClusterName("prod_name")
            .setServiceId("bob")
            .setServiceName("bob")
            .setShardSettings(shardSettings(DecimPolicy.DEFAULT))
            .build();

        Shard eva = Shard.newBuilder()
            .setNumId(42)
            .setId("bob")
            .setProjectId("solomon")
            .setClusterId("prod")
            .setClusterName("prod_name")
            .setServiceId("bob")
            .setServiceName("bob")
            .setShardSettings(shardSettings(DecimPolicy.DEFAULT))
            .build();

        assertTrue(insertSync(alice));
        assertTrue(insertSync(bob));
        assertFalse(insertSync(eva));

        var numIdToShardMapping = new HashMap<>(getShardsDao().findAllIdToShardId().join());
        assertEquals(ImmutableMap.of(42, "alice", 43, "bob"), numIdToShardMapping);
    }

    @Test
    public void numIdNotAbleChange() {
        Shard alice = Shard.newBuilder()
            .setNumId(42)
            .setId("alice")
            .setProjectId("solomon")
            .setClusterId("prod")
            .setClusterName("prod_name")
            .setServiceId("alice")
            .setServiceName("alice")
            .setShardSettings(shardSettings(DecimPolicy.DEFAULT))
            .build();

        assertTrue(insertSync(alice));

        getShardsDao().partialUpdate(alice.toBuilder().setNumId(43).build(), true).join();

        var v2 = findOneSync("solomon", "alice").get();
        assertEquals(42, v2.getNumId());
    }

    @Test
    public void findByProjectIdV3() {
        Shard shard1 = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        Shard shard2 = newShard("shard2", "project1", "", "cluster1", "service2", "user1", READ_ONLY);
        Shard shard3 = newShard("shard3", "project1", "", "cluster2", "service2", "user1", INACTIVE);
        Shard shard4 = newShard("shard4", "project2", "", "cluster1", "service2", "user1", ACTIVE);

        assertTrue(insertSync(shard1));
        assertTrue(insertSync(shard2));
        assertTrue(insertSync(shard3));
        assertTrue(insertSync(shard4));

        TokenBasePage<Shard> firstPage = findByProjectIdV3Sync("project1", 2, "", "");
        List<Shard> result = firstPage.getItems();
        assertEquals(2, result.size());
        assertEquals(shard1, result.get(0));
        assertEquals(shard2, result.get(1));
        assertEquals("2", firstPage.getNextPageToken());

        TokenBasePage<Shard> nextPage = findByProjectIdV3Sync("project1", 2, firstPage.getNextPageToken(), "");
        result = nextPage.getItems();
        assertEquals(1, result.size());
        assertEquals(shard3, result.get(0));
        assertEquals("", nextPage.getNextPageToken());
    }

    @Test
    public void releaseNumIdShardNotRemoved() {
        Shard shard = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        assertTrue(insertSync(shard));

        var dao = getShardsDao();
        var numIds = dao.findAllIdToShardId().join();
        assertFalse(dao.releaseNumId(shard.getProjectId(), shard.getId(), shard.getNumId()).join());
        assertEquals(numIds, dao.findAllIdToShardId().join());
    }

    @Test
    public void releaseNumIdUsedByOtherShard() {
        Shard shard = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        assertTrue(insertSync(shard));

        var dao = getShardsDao();
        var numIds = dao.findAllIdToShardId().join();
        assertFalse(dao.releaseNumId("project", "no_exist_id", shard.getNumId()).join());
        assertEquals(numIds, dao.findAllIdToShardId().join());
    }

    @Test
    public void releaseNumIdAlreadyReleased() {
        var dao = getShardsDao();
        assertTrue(dao.releaseNumId("projectId", "shardId", 42).join());
        assertEquals(new Int2ObjectOpenHashMap<>(), dao.findAllIdToShardId().join());
    }

    @Test
    public void releaseNumId() {
        Shard shard = newShard("shard1", "project1", "", "cluster1", "service1", "user1", RW);
        assertTrue(insertSync(shard));

        var dao = getShardsDao();
        var notEmptyNumIds = dao.findAllIdToShardId().join();
        assertNotEquals(new Int2ObjectOpenHashMap<>(), notEmptyNumIds);
        assertTrue(deleteOneSync(shard.getProjectId(), shard.getFolderId(), shard.getId()));
        assertEquals(notEmptyNumIds, dao.findAllIdToShardId().join());

        assertTrue(dao.releaseNumId(shard.getProjectId(), shard.getId(), shard.getNumId()).join());
        assertEquals(new Int2ObjectOpenHashMap<>(), dao.findAllIdToShardId().join());
    }

    private static Shard.Builder partial(Shard shard) {
        return Shard.newBuilder()
            .setId(shard.getId())
            .setNumId(shard.getNumId())
            .setProjectId(shard.getProjectId())
            .setClusterId(shard.getClusterId())
            .setServiceId(shard.getServiceId())
            .setClusterName(shard.getClusterName())
            .setServiceName(shard.getServiceName())
            .setCreatedAt(shard.getCreatedAt())
            .setUpdatedAt(shard.getUpdatedAt())
            .setState(shard.getState());
    }

    private static Shard.Builder name(
        String id,
        String projectId,
        String clusterId,
        String serviceId,
        String clusterName,
        String serviceName)
    {
        return shard()
            .setId(id)
            .setProjectId(projectId)
            .setClusterId(clusterId)
            .setServiceId(serviceId)
            .setClusterName(clusterName)
            .setServiceName(serviceName);
    }

    protected Shard newShard(
        String id,
        String projectId,
        String folderId,
        String clusterId,
        String serviceId,
        String updatedBy,
        ShardState state) {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return Shard.newBuilder()
                .setId(id)
                .setNumId(id.hashCode())
                .setProjectId(projectId)
                .setFolderId(folderId)
                .setClusterId(clusterId)
                .setServiceId(serviceId)
                .setClusterName(clusterId + "_name")
                .setServiceName(serviceId + "_name")
                .setDescription("Description for " + projectId + "/" + id + " shard")
                .setValidationMode(ValidationMode.STRICT_FAIL)
                .setMetricNameLabel("sensor")
                .setShardSettings(shardSettings(DecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS))
                .setLabels(Map.of("label1", "value1", "label2", "value2"))
                .setState(state)
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(updatedBy)
                .setUpdatedBy(updatedBy)
                .build();
    }

    protected static Shard.Builder shard() {
        int num = counter.incrementAndGet();
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return Shard.newBuilder()
            .setId("Shard1")
            .setNumId(num)
            .setProjectId("Project1")
            .setClusterId("Cluster1")
            .setServiceId("Service1")
            .setClusterName("ClusterName" + num)
            .setServiceName("ServiceName" + num)
            .setDescription("Description")
            .setMaxMetricsPerUrl(5)
            .setMaxFileMetrics(7)
            .setMaxMemMetrics(9)
            .setMaxResponseSizeBytes(11)
            .setNumPartitions(5)
            .setState(INACTIVE)
            .setValidationMode(ValidationMode.STRICT_FAIL)
            .setMetricNameLabel("sensor")
            .setShardSettings(shardSettings(DecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS))
            .setLabels(Map.of("label1", "value1", "label2", "value2"))
            .setVersion(1)
            .setCreatedAt(now)
            .setUpdatedAt(now.plus(Duration.ofMinutes(10)))
            .setCreatedBy("user1")
            .setUpdatedBy("user2");
    }

    private static ShardSettings shardSettings(DecimPolicy policy) {
        var pullSettings = ShardSettings.PullSettings.newBuilder()
                .setPort(5450)
                .setPath("/metrics")
                .build();

        var aggregationSettings = ShardSettings.AggregationSettings.of(false,
                new ServiceMetricConf.AggrRule[0], false);

        return ShardSettings.of(
                ShardSettings.Type.PULL,
                pullSettings,
                5,
                57,
                policy,
                aggregationSettings,
                15
        );
    }
}
