package ru.yandex.solomon.core.db.dao;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.collect.Lists;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.UserSettings;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Oleg Baryshnikov
 */
public abstract class AbstractUserSettingsDaoTest {

    @Test
    public void findAll() {
        List<UserSettings> userSettingsList = new ArrayList<>();

        for (int i = 0; i < 2000; ++i) {
            UserSettings userSettings = new UserSettings("user_" + i, Map.of());
            userSettingsList.add(userSettings);
        }

        upsertAllSync(userSettingsList);

        List<UserSettings> foundUserSettingsList = findAllSync();

        foundUserSettingsList.sort(Comparator.comparing(UserSettings::getLogin, TextWithNumbersComparator.instance));

        assertEquals(userSettingsList, foundUserSettingsList);
    }

    @Test
    public void findKnownLogin() {
        UserSettings userSettings = new UserSettings("user", Map.of("option1", "value1"));

        upsertSync(userSettings);

        UserSettings result = findByIdSync("user");
        assertEquals(userSettings, result);
    }

    @Test
    public void findUnknownLogin() {
        UserSettings userSettings = new UserSettings("user", Map.of("option1", "value1"));

        upsertSync(userSettings);

        UserSettings result = findByIdSync("other_user");
        assertEquals(new UserSettings("other_user", Map.of()), result);
    }

    @Test
    public void upsertNewUserSettings() {
        UserSettings userSettings = new UserSettings("user", Map.of("option1", "value1"));
        upsertSync(userSettings);
        assertEquals(userSettings, findByIdSync("user"));
    }

    @Test
    public void upsertExistingUserSettings() {
        upsertSync(new UserSettings("user", Map.of()));

        UserSettings userSettings = new UserSettings("user", Map.of("option1", "value1"));

        upsertSync(userSettings);

        assertEquals(userSettings, findByIdSync("user"));
    }

    @Test
    public void deleteKnownUser() {
        upsertSync(new UserSettings("user", Map.of()));
        assertTrue(deleteUserSync("user"));
        assertTrue(findAllSync().isEmpty());
    }

    @Test
    public void deleteUnknownUser() {
        upsertSync(new UserSettings("user", Map.of()));
        assertFalse(deleteUserSync("other_user"));
    }

    protected abstract UserSettingsDao getUserSettingsDao();

    private UserSettings findByIdSync(String login) {
        return join(getUserSettingsDao().findByLogin(login));
    }

    private UserSettings upsertSync(UserSettings userSettings) {
        return join(getUserSettingsDao().upsert(userSettings));
    }

    private void upsertAllSync(List<UserSettings> userSettingsList) {
        for (List<UserSettings> batch : Lists.partition(userSettingsList, 50)) {
            join(upsertBatch(batch));
        }
    }

    private CompletableFuture<Void> upsertBatch(List<UserSettings> batch) {
        UserSettingsDao dao = getUserSettingsDao();
        final int size = batch.size();
        final CompletableFuture[] futures = new CompletableFuture[size];
        for (int i = 0; i < size; i++) {
            futures[i] = dao.upsert(batch.get(i));
        }
        return CompletableFuture.allOf(futures);
    }

    private List<UserSettings> findAllSync() {
        return join(getUserSettingsDao().findAll());
    }

    private boolean deleteUserSync(String login) {
        return join(getUserSettingsDao().deleteUser(login));
    }
}
