package ru.yandex.solomon.core.db.dao;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.ViewHistory;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;


/**
 * @author albazh
 */
public abstract class AbstractViewHistoryDaoTest {

    @Test
    public void upsert() {
        ViewHistory record = newSimple(true);
        assertNull(record.getId());

        ViewHistory result = upsertSync(record).orElseThrow(AssertionError::new);
        assertNotNull(result.getId());
    }

    @Test
    public void saveLoad() {
        ViewHistory viewHistory = newSimple(false);
        ViewHistory result = upsertSync(viewHistory).orElseThrow(AssertionError::new);
        assertTrue(StringUtils.isNoneBlank(result.getId()));

        List<ViewHistory> found = findAllSync();
        assertEquals(1, found.size());

        ViewHistory record = found.get(0);

        assertEquals("login", record.getLogin());
        assertEquals("/?project=x&cluster=y&service=z", record.getUrl());
        assertEquals("title", record.getPageTitle());
        assertFalse(record.isPinned());
    }

    @Test
    public void delete() {
        ViewHistory viewHistory = newSimple(false).withUrl("/?project=x&cluster=y&service=z1");
        String login = viewHistory.getLogin();
        ViewHistory firstInsert = upsertSync(viewHistory).orElseThrow(AssertionError::new);
        assertNotNull(firstInsert.getId());
        ViewHistory nextUpdate = viewHistory.withUrl("/?project=x&cluster=y&service=z2");
        ViewHistory secondInsert = upsertSync(nextUpdate).orElseThrow(AssertionError::new);
        assertNotNull(secondInsert.getId());
        assertNotEquals(firstInsert.getId(), secondInsert.getId());

        List<ViewHistory> found = findAllSync();
        assertEquals(2, found.size());

        String idToDelete = found.get(0).getId();
        String idToSave = found.get(1).getId();

        assertTrue(join(getViewHistoryDao().deleteById(login, idToDelete)));
        found = findAllSync();
        assertEquals(1, found.size());
        assertEquals(idToSave, found.get(0).getId());
    }

    @Test
    public void findByLogin() {
        ViewHistory record = newSimple(false);
        ViewHistory result = upsertSync(record).orElseThrow(AssertionError::new);

        assertNotNull(result.getId());
        String realId = result.getId();

        Set<String> ids = join(getViewHistoryDao().findHistoryByLogin(record.getLogin()))
            .stream()
            .map(ViewHistory::getId).collect(Collectors.toSet());
        assertTrue(ids.contains(realId));
        assertFalse(ids.contains("fakeId"));

        assertTrue(join(getViewHistoryDao().findHistoryByLogin("fakeLogin")).isEmpty());
    }

    @Test
    public void pinNonexistent() {
        ViewHistory viewHistory = newSimple(true);
        upsertSync(viewHistory).orElseThrow(AssertionError::new);
        List<ViewHistory> records = findAllSync();
        assertEquals(1, records.size());
        ViewHistory record = records.iterator().next();
        assertEquals("title", record.getPageTitle());
        assertTrue(record.isPinned());
    }

    @Test
    public void pinUnpin() {
        ViewHistory history = newSimple(false).withPageTitle("title1");

        ViewHistory updateResult = updateAndCheck(history);
        String historyId = updateResult.getId();
        assertNotNull(historyId);

        history = history.withId(historyId);
        updateResult = updateAndCheck(history.withPin(true).withPageTitle("title2"));
        assertEquals(historyId, updateResult.getId());

        updateResult = updateAndCheck(history.withPin(false).withPageTitle("title3"));
        assertEquals(historyId, updateResult.getId());
    }

    protected abstract ViewHistoryDao getViewHistoryDao();

    private Optional<ViewHistory> upsertSync(ViewHistory viewHistory) {
        return join(getViewHistoryDao().upsert(viewHistory));
    }

    private List<ViewHistory> findAllSync() {
        return join(getViewHistoryDao().findAll());
    }

    private ViewHistory updateAndCheck(ViewHistory update) {
        ViewHistory result = upsertSync(update).orElseThrow(AssertionError::new);
        checkFieldEquals(update, result);
        List<ViewHistory> records = findAllSync();
        assertEquals(1, records.size());

        ViewHistory record = records.get(0);
        assertEquals(update.getPageTitle(), record.getPageTitle());
        assertEquals(update.isPinned(), record.isPinned());
        return result;
    }

    private static void checkFieldEquals(ViewHistory viewHistory, ViewHistory updated) {
        assertEquals(viewHistory.getLogin(), updated.getLogin());
        assertEquals(viewHistory.getUrl(), updated.getUrl());
        assertEquals(viewHistory.getPageTitle(), updated.getPageTitle());
        assertEquals(viewHistory.isPinned(), updated.isPinned());
        assertNotNull(updated.getId());
    }

    private static ViewHistory newSimple(boolean pinned) {
        return new ViewHistory("login", "/?project=x&cluster=y&service=z", "title", pinned);
    }

}
