package ru.yandex.solomon.core.db.dao.kikimr;

import java.util.List;

import com.google.common.collect.ImmutableList;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import static java.util.Arrays.asList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.listFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;


/**
 * @author snoop
 */
public class KikimrDaoSupportTest {

    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Test
    public void toTsvRegularExpressionForEmpty() {
        List<String> strings = List.of();

        String tsv = toTsv(strings);

        assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("")));
        assertFalse(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("one")));
    }

    @Test
    public void toTsvRegularExpressionForSingleItem() {
        List<String> strings = List.of("one");

        String tsv = toTsv(strings);

        assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("")));

        assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("one")));
        assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("ONE")));
        assertFalse(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("on")));

        assertFalse(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("two")));
        assertFalse(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("TWO")));
    }

    @Test
    public void toTsvRegularExpressionForSeveralItems() {
        List<String> strings = List.of("one", "two", "three");

        String tsv = toTsv(strings);

        assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("")));

        for (String str : strings) {
            assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression(str)));
            assertTrue(tsv.matches(KikimrDaoSupport.toTsvRegularExpression(str.toUpperCase())));
            assertFalse(tsv.matches(KikimrDaoSupport.toTsvRegularExpression(str.substring(0, 2))));
        }

        assertFalse(tsv.matches(KikimrDaoSupport.toTsvRegularExpression("four")));
    }

    @Test
    public void toTsvCollection() {
        List<String> strings = asList("one", "two", "three");
        assertEquals(strings, listFromTsv(toTsv(strings)));

        strings = ImmutableList.of("one");
        assertEquals(strings, listFromTsv(toTsv(strings)));

        strings = asList("one", "two", "three\tfour");
        List<String> expectedStrings = asList("one", "two", "three four");
        assertEquals(expectedStrings, listFromTsv(toTsv(strings, true)));

        thrown.expect(NullPointerException.class);
        strings = asList("one", null, "three");
        toTsv(strings);
    }
}
