package ru.yandex.solomon.core.db.dao.memory;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.core.db.dao.ProjectFlagsDao;

/**
 * @author Vladimir Gordiychuk
 */
public class InMemoryProjectFlagsDao implements ProjectFlagsDao {
    private final ConcurrentMap<Key, Boolean> flagByKey = new ConcurrentHashMap<>();
    private volatile Throwable error;

    public void setFailure(Throwable e) {
        this.error = e;
    }

    @Override
    public CompletableFuture<List<Record>> findAll() {
        return CompletableFuture.supplyAsync(() -> {
            failIfNecessary();
            return flagByKey.entrySet()
                    .stream()
                    .map(entry -> {
                        var key = entry.getKey();
                        return new Record(key.flag, key.projectId, entry.getValue());
                    })
                    .collect(Collectors.toList());
        });
    }

    @Override
    public CompletableFuture<Void> upsert(Record record) {
        return CompletableFuture.supplyAsync(() -> {
            failIfNecessary();
            var key = new Key(record.flag, record.projectId);
            flagByKey.put(key, record.value);
            return null;
        });
    }

    @Override
    public CompletableFuture<Void> deleteOne(String flag, String projectId) {
        return CompletableFuture.supplyAsync(() -> {
            failIfNecessary();
            flagByKey.remove(new Key(flag, projectId));
            return null;
        });
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return CompletableFuture.supplyAsync(() -> {
            failIfNecessary();
            return null;
        });
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return CompletableFuture.supplyAsync(() -> {
            failIfNecessary();
            return null;
        });
    }

    private void failIfNecessary() {
        var copy = error;
        if (copy == null) {
            return;
        }
        Throwables.throwIfUnchecked(copy);
        throw new RuntimeException(copy);
    }

    private static class Key extends DefaultObject {
        private final String flag;
        private final String projectId;

        public Key(String flag, String projectId) {
            this.flag = flag;
            this.projectId = projectId;
        }
    }
}
