package ru.yandex.solomon.core.db.model;

import java.io.UncheckedIOException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.ServiceMetricConf.AggrRule;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class ServiceMetricConfTest {

    @Test
    public void toJsonAndBack() {
        {
            var expect = ServiceMetricConf.of(new AggrRule[]{AggrRule.of("host=*", "host=cluster", null)}, false);
            var actual = fromJson(toJson(expect), ServiceMetricConf.class);
            assertEquals(expect, actual);
        }
        {
            var expect = ServiceMetricConf.of(new AggrRule[]{AggrRule.of("host=*", "host=cluster", null)}, true);
            var actual = fromJson(toJson(expect), ServiceMetricConf.class);
            assertEquals(expect, actual);
        }
        {
            var expect = ServiceMetricConf.of(new AggrRule[]{
                    AggrRule.of("host=*", "host=cluster", null),
                    AggrRule.of("host=*", "host={{DC}}", null),
                    AggrRule.of("host=*, clientId=*", "host=cluster, clientId=total", null),
            }, false);
            var actual = fromJson(toJson(expect), ServiceMetricConf.class);
            assertEquals(expect, actual);
        }
        {
            var expect = ServiceMetricConf.of(new AggrRule[]{
                    AggrRule.of("host=*, name=*.size", "host=cluster", MetricAggregation.LAST),
            }, true);
            var actual = fromJson(toJson(expect), ServiceMetricConf.class);
            assertEquals(expect, actual);
        }
    }

    @Test
    public void parseEmpty() {
        var expect = ServiceMetricConf.of(new AggrRule[0], false);
        var actual = fromJson("""
                    {"aggrRules":[],"rawDataMemOnly":false}
                    """, ServiceMetricConf.class);
        assertEquals(expect, actual);
    }

    @Test
    public void parseMemOnly() {
        var expect = ServiceMetricConf.of(new AggrRule[0], false);
        var actual = fromJson("""
                    {"aggrRules":[],"rawDataMemOnly":false,"empty":true}
                    """, ServiceMetricConf.class);
        assertEquals(expect, actual);
    }

    @Test
    public void parseCommonAggr() {
        var expected = ServiceMetricConf.of(new AggrRule[]{
                AggrRule.of("host=*", "host=cluster", null)
        }, false);
        var actual = fromJson("""
                    {"aggrRules":[{"cond":["host=*"],"target":["host=cluster"]}],"rawDataMemOnly":false}
                    """, ServiceMetricConf.class);
        assertEquals(expected, actual);
    }

    @Test
    public void parseMultipleConditionAndTarget() {
        var expected = ServiceMetricConf.of(new AggrRule[]{
                AggrRule.of("host=*", "host={{DC}}", null),
                AggrRule.of("host=*", "host=cluster", null),
        }, false);
        var actual = fromJson("""
                    {
                        "aggrRules": [
                            {
                                "cond": [
                                    "host=*"
                                ],
                                "target": [
                                    "host={{DC}}"
                                ]
                            },
                            {
                                "cond": [
                                    "host=*"
                                ],
                                "target": [
                                    "host=cluster"
                                ]
                            }
                        ],
                        "rawDataMemOnly": false
                    }
                    """, ServiceMetricConf.class);
        assertEquals(expected, actual);
    }

    @Test
    public void parseAggregation() {
        var expect = ServiceMetricConf.of(new AggrRule[]{
                AggrRule.of("host=*, name=*.size", "host=cluster", MetricAggregation.LAST),
        }, true);
        var actual = fromJson("""
                    {
                        "aggrRules": [
                            {
                                "cond": [
                                    "host=*",
                                    "name=*.size"
                                ],
                                "target": [
                                    "host=cluster"
                                ],
                                "aggr": "LAST"
                            }
                        ],
                        "rawDataMemOnly": true,
                        "empty": false
                    }
                    """, ServiceMetricConf.class);
        assertEquals(expect, actual);
    }

    public static String toJson(Object value) {
        if (value == null) {
            return null;
        }
        try {
            var objectMapper = new ObjectMapper();
            return objectMapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static <T> T fromJson(String jsonText, Class<T> valueClazz) {
        try {
            var objectMapper = new ObjectMapper();
            return objectMapper.readValue(jsonText, valueClazz);
        } catch (java.io.IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
