package ru.yandex.solomon.conf.db3;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.conf.db3.ydb.YdbCloudFavoritesDao;
import ru.yandex.solomon.conf.db3.ydb.YdbCloudServiceDashboardsDao;
import ru.yandex.solomon.conf.db3.ydb.YdbMonitoringFavoritesDao;
import ru.yandex.solomon.conf.db3.ydb.YdbQuickLinksDao;
import ru.yandex.solomon.core.db.YdbClientsContext;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;

import static ru.yandex.solomon.core.db.dao.ConfigDaoContext.createObjectMapper;


/**
 * @author Oleg Baryshnikov
 */
@Configuration
@Import({
    YdbClientsContext.class,
})
public class ConfigV3DaoContext {

    public static final String CONFIG_DIR = "Config";
    public static final String VERSION = "V3";

    public static final String CLOUD_SERVICE_DASHBOARD_TABLE = "CloudServiceDashboard";
    public static final String FAVORITE_TABLE = "Favorite";
    public static final String CLOUD_FAVORITE_TABLE = "CloudFavorite";
    public static final String QUICK_LINKS_TABLE = "QuickLinks";

    private final TableClient tableClient;
    @Nullable
    private final MetricRegistry metricRegistry;
    private final String ydbPath;
    private final ObjectMapper objectMapper;

    @Autowired
    public ConfigV3DaoContext(
        TableClient tableClient,
        @Qualifier("configKikimrRootPath") String rootSchema,
        @Nullable MetricRegistry metricRegistry)
    {
        this.tableClient = tableClient;
        this.metricRegistry = metricRegistry;
        this.ydbPath = rootSchema + "/" + CONFIG_DIR + "/" + VERSION + "/";
        this.objectMapper = createObjectMapper();
    }

    @Bean
    public CloudServiceDashboardsDao cloudServiceDashboardsDao() {
        String tablePath = ydbPath + CLOUD_SERVICE_DASHBOARD_TABLE;
        var dao = new YdbCloudServiceDashboardsDao(tableClient, tablePath);
        return measure(dao, CloudServiceDashboardsDao.class);
    }

    @Bean
    public MonitoringFavoritesDao monitoringFavoritesDao() {
        String tablePath = ydbPath + FAVORITE_TABLE;
        var dao = new YdbMonitoringFavoritesDao(tableClient, tablePath);
        return measure(dao, MonitoringFavoritesDao.class);
    }

    @Bean
    public CloudFavoritesDao cloudFavoritesDao() {
        String tablePath = ydbPath + CLOUD_FAVORITE_TABLE;
        var dao = new YdbCloudFavoritesDao(tableClient, tablePath);
        return measure(dao, CloudFavoritesDao.class);
    }

    @Bean
    public QuickLinksDao quickLinksDao() {
        String tablePath = ydbPath + QUICK_LINKS_TABLE;
        var dao = new YdbQuickLinksDao(tableClient, tablePath, objectMapper);
        return measure(dao, QuickLinksDao.class);
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return metricRegistry == null ? dao : DaoMetricsProxy.of(dao, daoClass, metricRegistry);
    }
}
