package ru.yandex.solomon.conf.db3;

import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.core.db.model.AbstractAuditable;
import ru.yandex.solomon.core.db.model.AuditableBuilder;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class QuickLinksRecord extends AbstractAuditable {
    private final String projectId;
    private final LinkItem[] items;

    private QuickLinksRecord(Builder builder) {
        super(builder);
        projectId = builder.projectId;
        items = builder.items;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getProjectId() {
        return projectId;
    }

    public LinkItem[] getItems() {
        return items;
    }

    public static class LinkItem extends DefaultObject {
        private final String title;
        private final String url;
        private final LinkItem[] items;

        @JsonCreator
        public LinkItem(
            @Nullable @JsonProperty("title") String title,
            @Nullable @JsonProperty("url") String url,
            @Nullable @JsonProperty("items") LinkItem[] items)
        {
            this.title = Objects.requireNonNull(title, "link title must not be null");
            this.url = Nullables.orEmpty(url);
            this.items = Nullables.orDefault(items, new LinkItem[0]);
        }

        @JsonProperty
        public String getTitle() {
            return title;
        }

        @JsonProperty
        public String getUrl() {
            return url;
        }

        @JsonProperty
        public LinkItem[] getItems() {
            return items;
        }
    }

    public static final class Builder extends AuditableBuilder<Builder> {
        private String projectId = "";
        private LinkItem[] items = new LinkItem[0];

        private Builder() {
        }

        private Builder(QuickLinksRecord quickLinksRecord) {
            super(quickLinksRecord);
            this.projectId = quickLinksRecord.projectId;
            this.items = quickLinksRecord.items;
        }

        public Builder setProjectId(String projectId) {
            this.projectId = projectId;
            return this;
        }

        public Builder setItems(LinkItem[] items) {
            this.items = items;
            return this;
        }

        public QuickLinksRecord build() {
            return new QuickLinksRecord(this);
        }
    }
}
