package ru.yandex.solomon.conf.db3.memory;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.conf.db3.EntitiesDao;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class InMemoryEntitiesDao implements EntitiesDao {
    private ConcurrentMap<String, Entity> entityById = new ConcurrentHashMap<>();

    public void add(Entity entity) {
        entityById.put(entity.getId(), entity);
    }

    public void addAll(Collection<Entity> entities) {
        for (var entity : entities) {
            add(entity);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return runAsync(() -> {});
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return runAsync(() -> entityById.clear());
    }

    @Override
    public CompletableFuture<Boolean> insert(Entity config) {
        return supplyAsync(() -> {
            validate(config);
            return entityById.putIfAbsent(config.getId(), config) == null;
        });
    }

    private void validate(Entity config) {
        if (config.getLocalId().isEmpty()) {
            return;
        }
        for (Entity value : entityById.values()) {
            if (value.getId().equals(config.getId())) {
                continue;
            }
            if (!value.getLocalId().equals("") && value.getParentId().equals(config.getParentId()) && value.getLocalId().equals(config.getLocalId())) {
                throw new IllegalArgumentException("Name must be uniq in parentId");
            }
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Entity>> list(String parentId, String filterByName, int pageSize, String pageToken) {
        return supplyAsync(() -> {
            var matched = entityById.values()
                    .stream()
                    .filter(config -> config.getParentId().equals(parentId))
                    .filter(config -> StringUtils.containsIgnoreCase(config.getName(), filterByName))
                    .sorted(Comparator.comparing(Entity::getParentId).thenComparing(Entity::getId))
                    .collect(Collectors.toList());

            return toPageResult(matched, pageSize, pageToken);
        });
    }

    @Override
    public CompletableFuture<TokenBasePage<Entity>> listByLocalId(String parentId, String filterByLocalId, int pageSize, String pageToken) {
        return supplyAsync(() -> {
            var matched = entityById.values()
                    .stream()
                    .filter(config -> config.getParentId().equals(parentId))
                    .filter(config -> StringUtils.containsIgnoreCase(config.getLocalId(), filterByLocalId))
                    .sorted(Comparator.comparing(Entity::getParentId).thenComparing(Entity::getLocalId))
                    .collect(Collectors.toList());

            return toPageResult(matched, pageSize, pageToken);
        });
    }

    @Override
    public CompletableFuture<TokenBasePage<Entity>> listAll(String filterByName, int pageSize, String pageToken) {
        return supplyAsync(() -> {
            var matched = entityById.values()
                    .stream()
                    .filter(config -> StringUtils.containsIgnoreCase(config.getName(), filterByName))
                    .sorted(Comparator.comparing(Entity::getParentId).thenComparing(Entity::getId))
                    .collect(Collectors.toList());

            return toPageResult(matched, pageSize, pageToken);
        });
    }

    @Override
    public CompletableFuture<Optional<Entity>> read(String parentId, String entityId) {
        return supplyAsync(() -> Optional.ofNullable(entityById.get(entityId))
                .filter(entity -> entity.getParentId().equals(parentId)));
    }

    @Override
    public CompletableFuture<Optional<Entity>> readById(String entityId) {
        return supplyAsync(() -> Optional.ofNullable(entityById.get(entityId)));
    }

    @Override
    public CompletableFuture<List<Entity>> readAll() {
        return supplyAsync(() -> List.copyOf(entityById.values()));
    }

    @Override
    public CompletableFuture<Optional<Entity>> update(Entity config) {
        return readById(config.getId())
                .thenApply(optional -> {
                    if (optional.isEmpty()) {
                        return Optional.empty();
                    }
                    if (!optional.get().getLocalId().equals(config.getLocalId())) {
                        validate(config);
                    }

                    var prev = optional.get();
                    if (config.getVersion() != -1 && prev.getVersion() != config.getVersion()) {
                        return Optional.empty();
                    }

                    var patched = prev.toBuilder()
                            .setName(config.getName())
                            .setDescription(config.getDescription())
                            .setData(config.getData())
                            .setProto(config.getProto())
                            .setUpdatedAt(config.getUpdatedAt())
                            .setUpdatedBy(config.getUpdatedBy())
                            .setVersion(prev.getVersion() + 1)
                            .setLocalId(config.getLocalId())
                            .build();

                    if (entityById.replace(config.getId(), prev, patched)) {
                        return Optional.of(patched);
                    }

                    return Optional.empty();
                });
    }

    @Override
    public CompletableFuture<Void> upsert(Entity config) {
        return runAsync(() -> {
            validate(config);
            entityById.put(config.getId(), config);
        });
    }

    @Override
    public CompletableFuture<Boolean> delete(String parentId, String entityId) {
        return read(parentId, entityId)
                .thenApply(entity -> {
                    if (entity.isEmpty()) {
                        return false;
                    }

                    return entityById.remove(entityId, entity.get());
                });
    }

    @Override
    public CompletableFuture<Boolean> deleteWithVersion(String parentId, String entityId, int version) {
        return read(parentId, entityId)
                .thenApply(entity -> {
                    if (entity.isEmpty()) {
                        return false;
                    }
                    if (version != -1 && entity.get().getVersion() != version) {
                        return false;
                    }
                    return entityById.remove(entityId, entity.get());
                });
    }

    @Override
    public CompletableFuture<Boolean> exists(String parentId, String entityId) {
        return read(parentId, entityId)
                .thenApply(Optional::isPresent);
    }

    @Override
    public CompletableFuture<Void> deleteByParentId(String parentId) {
        return supplyAsync(() -> {
            entityById.values().removeIf(entity -> entity.getParentId().equals(parentId));
            return null;
        });
    }

    private TokenBasePage<Entity> toPageResult(List<Entity> matched, int pageSize, String pageToken) {
        if (pageSize <= 0) {
            pageSize = 100;
        } else {
            pageSize = Math.min(pageSize, 1000);
        }

        int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);
        int nextOffset = Math.min(offset + pageSize, matched.size());
        var list = matched.subList(offset, nextOffset);
        if (nextOffset >= matched.size()) {
            return new TokenBasePage<>(list, "");
        }
        return new TokenBasePage<>(list, Integer.toString(nextOffset));
    }
}
