package ru.yandex.solomon.conf.db3.ydb;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.ParametersAreNullableByDefault;

import com.google.protobuf.Any;

import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class Entity {
    private final String id;
    private final String localId;
    private final String parentId;
    private final String name;
    private final String description;
    @Deprecated
    private final String data;
    private final Any proto;

    private final long createdAt;
    private final long updatedAt;
    private final String createdBy;
    private final String updatedBy;
    private final ContainerType containerType;
    private final int version;

    private Entity(Builder builder) {
        parentId = Objects.requireNonNull(builder.parentId, "parent id must not be null");
        id = Objects.requireNonNull(builder.id, "id must not be null");
        name = Objects.requireNonNull(builder.name, "name must not be null");
        description = Nullables.orEmpty(builder.description);
        data = Nullables.orEmpty(builder.data);
        localId = Nullables.orEmpty(builder.localId);
        proto = Nullables.orDefault(builder.proto, Any.getDefaultInstance());
        createdAt = builder.createdAt;
        updatedAt = builder.updatedAt;
        createdBy = Nullables.orEmpty(builder.createdBy);
        updatedBy = Nullables.orEmpty(builder.updatedBy);
        containerType = Objects.requireNonNull(builder.containerType, "containerType must not be null");
        version = builder.version;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        Builder builder = new Builder();
        builder.id = this.id;
        builder.localId = this.localId;
        builder.parentId = this.parentId;
        builder.name = this.name;
        builder.description = this.description;
        builder.data = this.data;
        builder.proto = this.proto;
        builder.createdAt = this.createdAt;
        builder.updatedAt = this.updatedAt;
        builder.createdBy = this.createdBy;
        builder.updatedBy = this.updatedBy;
        builder.version = this.version;
        builder.containerType = this.containerType;
        return builder;
    }

    public String getParentId() {
        return parentId;
    }

    public String getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public String getData() {
        return data;
    }

    public Any getProto() {
        return proto;
    }

    public String getLocalId() {
        return localId;
    }

    public boolean hasProto() {
        return proto != Any.getDefaultInstance();
    }

    public ContainerType getContainerType() {
        return containerType;
    }

    public long getCreatedAt() {
        return createdAt;
    }

    public long getUpdatedAt() {
        return updatedAt;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public int getVersion() {
        return version;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        Entity entity = (Entity) o;

        if (createdAt != entity.createdAt) {
            return false;
        }
        if (updatedAt != entity.updatedAt) {
            return false;
        }
        if (version != entity.version) {
            return false;
        }
        if (!parentId.equals(entity.parentId)) {
            return false;
        }
        if (!id.equals(entity.id)) {
            return false;
        }
        if (!name.equals(entity.name)) {
            return false;
        }
        if (!description.equals(entity.description)) {
            return false;
        }
        if (!data.equals(entity.data)) {
            return false;
        }
        if (!proto.equals(entity.proto)) {
            return false;
        }
        if (!localId.equals(entity.localId)) {
            return false;
        }
        if (!containerType.equals(entity.containerType)) {
            return false;
        }
        if (!createdBy.equals(entity.createdBy)) {
            return false;
        }
        return updatedBy.equals(entity.updatedBy);
    }

    @Override
    public int hashCode() {
        int result = parentId.hashCode();
        result = 31 * result + id.hashCode();
        result = 31 * result + name.hashCode();
        result = 31 * result + description.hashCode();
        result = 31 * result + data.hashCode();
        result = 31 * result + proto.hashCode();
        result = 31 * result + containerType.hashCode();
        result = 31 * result + localId.hashCode();
        result = 31 * result + (int) (createdAt ^ (createdAt >>> 32));
        result = 31 * result + (int) (updatedAt ^ (updatedAt >>> 32));
        result = 31 * result + createdBy.hashCode();
        result = 31 * result + updatedBy.hashCode();
        result = 31 * result + version;
        return result;
    }

    @Override
    public String toString() {
        return "EntityV3{" +
                "parentId='" + parentId + '\'' +
                ", id='" + id + '\'' +
                ", name='" + name + '\'' +
                ", description='" + description + '\'' +
                ", data='" + data + '\'' +
                ", proto='" + proto + '\'' +
                ", containerType='" + containerType + '\'' +
                ", localId='" + localId + '\'' +
                ", createdAt=" + createdAt +
                ", updatedAt=" + updatedAt +
                ", createdBy='" + createdBy + '\'' +
                ", updatedBy='" + updatedBy + '\'' +
                ", version=" + version +
                '}';
    }

    @ParametersAreNullableByDefault
    public static final class Builder {
        private String parentId;
        private String id;
        private String localId;
        private String name;
        private String description;
        private String data;
        private Any proto;
        private ContainerType containerType;
        private long createdAt;
        private long updatedAt;
        private String createdBy;
        private String updatedBy;
        private int version;

        private Builder() {
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setLocalId(String localId) {
            this.localId = localId;
            return this;
        }

        public Builder setParentId(String parentId) {
            this.parentId = parentId;
            return this;
        }

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setData(String data) {
            this.data = data;
            return this;
        }

        public Builder setProto(Any data) {
            this.proto = data;
            return this;
        }

        public Builder setCreatedAt(long createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder setUpdatedAt(long updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public Builder setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public Builder setUpdatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
            return this;
        }

        public Builder setVersion(int version) {
            this.version = version;
            return this;
        }

        public Builder setContainerType(ContainerType containerType) {
            this.containerType = containerType;
            return this;
        }

        public Entity build() {
            return new Entity(this);
        }
    }
}
