package ru.yandex.solomon.conf.db3.ydb;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.conf.db3.CloudFavoritesDao;
import ru.yandex.solomon.conf.db3.EntityType;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class YdbCloudFavoritesDao implements CloudFavoritesDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
            YdbMonitoringFavoritesDao.class,
            "favorites",
            Arrays.asList(
                    "create_table",
                    "delete",
                    "exists",
                    "insert",
                    "list",
                    "list_all"
            ));

    private final Table table;
    private final QueryText queryText;

    public YdbCloudFavoritesDao(TableClient tableClient, String tablePath) {
        this.table = new Table(tableClient, tablePath);
        this.queryText = TEMPLATE.build(Map.of("favorites.table.path", tablePath));
    }

    @Override
    public CompletableFuture<List<CloudFavoriteRecord>> list(String login, EntityType entityType) {
        try {
            String query = queryText.query("list");
            Params params = Params.of(
                    "$login", utf8(login),
                    "$type", utf8(entityType.name()));
            return table.queryList(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<CloudFavoriteRecord>> listAll(String login) {
        try {
            String query = queryText.query("list_all");
            Params params = Params.of("$login", utf8(login));
            return table.queryList(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> create(CloudFavoriteRecord record) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, record);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(CloudFavoriteRecord record) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of(
                    "$login", utf8(record.getLogin()),
                    "$type", utf8(record.getType().name()),
                    "$id", utf8(record.getId()));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(CloudFavoriteRecord record) {
        try {
            String query = queryText.query("exists");
            Params params = Params.of(
                    "$login", utf8(record.getLogin()),
                    "$type", utf8(record.getType().name()),
                    "$id", utf8(record.getId()));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static final class Table extends YdbTable<String, CloudFavoriteRecord> {
        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("login", PrimitiveType.utf8())
                    .addNullableColumn("type", PrimitiveType.utf8())
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .setPrimaryKeys("login", "type", "id")
                    .build();
        }

        @Override
        protected String getId(CloudFavoriteRecord config) {
            return config.getId();
        }

        @Override
        protected Params toParams(CloudFavoriteRecord config) {
            return Params.create()
                    .put("$login", utf8(config.getLogin()))
                    .put("$type", utf8(config.getType().name()))
                    .put("$id", utf8(config.getId()));
        }

        @Override
        protected CloudFavoriteRecord mapFull(ResultSetReader r) {
            return new CloudFavoriteRecord(
                    r.getColumn("login").getUtf8(),
                    EntityType.valueOf(r.getColumn("type").getUtf8()),
                    r.getColumn("id").getUtf8());
        }

        @Override
        protected CloudFavoriteRecord mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
