package ru.yandex.solomon.conf.db3.ydb;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.conf.db3.CloudServiceDashboardRecord;
import ru.yandex.solomon.conf.db3.CloudServiceDashboardsDao;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class YdbCloudServiceDashboardsDao implements CloudServiceDashboardsDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
            YdbEntitiesDao.class,
            "cloud_service_dashboard",
            Arrays.asList(
                    "create_table",
                    "delete",
                    "delete_by_service",
                    "exists",
                    "insert",
                    "list",
                    "list_all",
                    "read",
                    "update",
                    "upsert"
            ));

    private final Table table;
    private final QueryText queryText;

    public YdbCloudServiceDashboardsDao(TableClient tableClient, String tablePath) {
        this.table = new Table(tableClient, tablePath);
        this.queryText = TEMPLATE.build(Map.of("cloud_service_dashboard.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Boolean> insert(CloudServiceDashboardRecord config) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, config);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public CompletableFuture<Optional<CloudServiceDashboardRecord>> read(String id) {
        try {
            String query = queryText.query("read");
            Params params = Params.of("$id", utf8(id));
            return table.queryOne(query, params);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<CloudServiceDashboardRecord>> list(
            Set<String> services,
            String filterByName,
            int pageSize,
            String pageToken) {
        try {
            String query = queryText.query("list");

            String servicesRegexp = "(" + String.join("|", services) + ")";

            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$servicesRegexp", utf8(servicesRegexp),
                    "$filterByNameRegexp", utf8(toRegularExpression(filterByName)),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(query, params)
                    .thenApply(result -> {
                        final String nextPageToken;

                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }

                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<CloudServiceDashboardRecord>> listAll(String filterByName, int pageSize, String pageToken) {
        try {
            String query = queryText.query("list_all");

            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$filterByNameRegexp", utf8(toRegularExpression(filterByName)),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(query, params)
                    .thenApply(result -> {
                        final String nextPageToken;

                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }

                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<CloudServiceDashboardRecord>> update(CloudServiceDashboardRecord config) {
        try {
            String query = queryText.query("update");
            return table.updateOne(query, config);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> upsert(CloudServiceDashboardRecord config) {
        try {
            String query = queryText.query("upsert");
            return table.upsertOne(query, config);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(String dashboardId) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$id", utf8(dashboardId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String dashboardId) {
        try {
            String query = queryText.query("exists");
            Params params = Params.of(
                    "$id", utf8(dashboardId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByService(String service) {
        try {
            String query = queryText.query("delete_by_service");
            Params params = Params.of("$service", utf8(service));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }


    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static final class Table extends YdbTable<String, CloudServiceDashboardRecord> {

        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("service", PrimitiveType.utf8())
                    .addNullableColumn("name", PrimitiveType.utf8())
                    .addNullableColumn("description", PrimitiveType.utf8())
                    .addNullableColumn("data", PrimitiveType.utf8())
                    .addNullableColumn("createdAt", PrimitiveType.int64())
                    .addNullableColumn("updatedAt", PrimitiveType.int64())
                    .addNullableColumn("createdBy", PrimitiveType.utf8())
                    .addNullableColumn("updatedBy", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.int32())
                    .setPrimaryKeys("id")
                    .build();
        }

        @Override
        protected String getId(CloudServiceDashboardRecord config) {
            return config.getId();
        }

        @Override
        protected Params toParams(CloudServiceDashboardRecord config) {
            return Params.create()
                    .put("$id", utf8(config.getId()))
                    .put("$service", utf8(config.getService()))
                    .put("$name", utf8(config.getName()))
                    .put("$description", utf8(config.getDescription()))
                    .put("$data", utf8(config.getData()))
                    .put("$createdAt", int64(config.getCreatedAt()))
                    .put("$updatedAt", int64(config.getUpdatedAt()))
                    .put("$createdBy", utf8(config.getCreatedBy()))
                    .put("$updatedBy", utf8(config.getUpdatedBy()))
                    .put("$version", int32(config.getVersion()));
        }

        @Override
        protected CloudServiceDashboardRecord mapFull(ResultSetReader r) {
            return toDashboard(r);
        }

        @Override
        protected CloudServiceDashboardRecord mapPartial(ResultSetReader r) {
            return toDashboard(r);
        }

        private CloudServiceDashboardRecord toDashboard(ResultSetReader r) {
            return CloudServiceDashboardRecord.newBuilder()
                    .setId(r.getColumn("id").getUtf8())
                    .setService(r.getColumn("service").getUtf8())
                    .setName(r.getColumn("name").getUtf8())
                    .setDescription(r.getColumn("description").getUtf8())
                    .setData(r.getColumn("data").getUtf8())
                    .setCreatedAt(r.getColumn("createdAt").getInt64())
                    .setCreatedBy(r.getColumn("createdBy").getUtf8())
                    .setUpdatedAt(r.getColumn("updatedAt").getInt64())
                    .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                    .setVersion(r.getColumn("version").getInt32())
                    .build();
        }
    }
}
