package ru.yandex.solomon.conf.db3.ydb;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.conf.db3.EntitiesDao;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.string;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toRegularExpression;


/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class YdbEntitiesDao implements EntitiesDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
            YdbEntitiesDao.class,
            "entity",
            Arrays.asList(
                    "create_table",
                    "delete",
                    "delete_by_parent_id",
                    "exists",
                    "insert",
                    "list",
                    "list_all",
                    "read",
                    "read_by_id",
                    "update",
                    "upsert"
            ));

    private final EntitiesTable table;
    private final QueryText queryText;

    public YdbEntitiesDao(TableClient tableClient, String tablePath, boolean isCloud) {
        String parentColumn = isCloud ? "folderId" : "projectId";
        this.table = new EntitiesTable(tableClient, tablePath, parentColumn);
        this.queryText = TEMPLATE.build(Map.of("entity.table.path", tablePath, "parentColumn", parentColumn));
    }

    @Override
    public CompletableFuture<Boolean> insert(Entity config) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, config)
                    .exceptionally(this::handleLocalIdException);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Entity>> read(String parentId, String entityId) {
        try {
            String query = queryText.query("read");
            Params params = Params.of(
                "$parentId", utf8(parentId),
                "$id", utf8(entityId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Entity>> readById(String entityId) {
        try {
            String query = queryText.query("read_by_id");
            Params params = Params.of("$id", utf8(entityId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<Entity>> readAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Entity>> list(
        String parentId,
        String filterByName,
        int pageSize,
        String pageToken) {
        return listByFilter(parentId, filterByName, "name", pageSize, pageToken);
    }

    @Override
    public CompletableFuture<TokenBasePage<Entity>> listByLocalId(
            String parentId,
            String filterByLocalId,
            int pageSize,
            String pageToken) {
        return listByFilter(parentId, filterByLocalId, "localId", pageSize, pageToken);
    }

    private CompletableFuture<TokenBasePage<Entity>> listByFilter(String parentId, String filter, String column, int pageSize, String pageToken) {
        try {
            String query = queryText.query("list", Map.of("filter.column", column));

            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$parentId", utf8(parentId),
                    "$filterRegexp", utf8(toRegularExpression(filter)),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(query, params)
                    .thenApply(result -> {
                        final String nextPageToken;

                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }

                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<Entity>> listAll(String filterByName, int pageSize, String pageToken) {
        try {
            String query = queryText.query("list_all");

            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$filterByNameRegexp", utf8(toRegularExpression(filterByName)),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(query, params)
                    .thenApply(result -> {
                        final String nextPageToken;

                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }

                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> upsert(Entity config) {
        try {
            String query = queryText.query("upsert");
            return table.upsertOne(query, config)
                    .exceptionally(this::handleLocalIdException);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Entity>> update(Entity config) {
        try {
            String query = queryText.query("update");
            return table.updateOne(query, config)
                    .exceptionally(this::handleLocalIdException);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(String parentId, String entityId) {
        return deleteWithVersion(parentId, entityId, -1);
    }

    @Override
    public CompletableFuture<Boolean> deleteWithVersion(String parentId, String entityId, int version) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of(
                    "$parentId", utf8(parentId),
                    "$id", utf8(entityId),
                    "$version", int32(version)
            );
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> exists(String parentId, String entityId) {
        try {
            String query = queryText.query("exists");
            Params params = Params.of(
                "$parentId", utf8(parentId),
                "$id", utf8(entityId));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByParentId(String parentId) {
        try {
            String query = queryText.query("delete_by_parent_id");
            Params params = Params.of("$parentId", utf8(parentId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private <T> T handleLocalIdException(Throwable throwable) {
        Throwable cause = CompletableFutures.unwrapCompletionException(throwable);
        if (cause instanceof UnexpectedResultException) {
            if (cause.getMessage().contains("Name must be uniq in parentId")) {
                throw new ConflictException("Name must be uniq in parentId");
            }
        }
        throw new RuntimeException(throwable);
    }

    private static final class EntitiesTable extends YdbTable<String, Entity> {
        private final String parentColumn;

        EntitiesTable(TableClient tableClient, String path, String parentColumn) {
            super(tableClient, path);
            this.parentColumn = parentColumn;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn(parentColumn, PrimitiveType.utf8())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("data", PrimitiveType.utf8())
                .addNullableColumn("localId", PrimitiveType.utf8())
                .addNullableColumn("proto", PrimitiveType.string())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("containerType", PrimitiveType.utf8())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys(parentColumn, "id")
                .addGlobalIndex("globalIdIndex", List.of("id"))
                .build();
        }

        @Override
        protected String getId(Entity config) {
            return config.getId();
        }

        @Override
        protected Params toParams(Entity config) {
            return Params.create()
                .put("$parentId", utf8(config.getParentId()))
                .put("$id", utf8(config.getId()))
                .put("$containerType", utf8(config.getContainerType().name()))
                .put("$name", utf8(config.getName()))
                .put("$description", utf8(config.getDescription()))
                .put("$data", utf8(config.getData()))
                .put("$localId", utf8(config.getLocalId()))
                .put("$proto", string(config.getProto().toByteString()))
                .put("$createdAt", int64(config.getCreatedAt()))
                .put("$updatedAt", int64(config.getUpdatedAt()))
                .put("$createdBy", utf8(config.getCreatedBy()))
                .put("$updatedBy", utf8(config.getUpdatedBy()))
                .put("$version", int32(config.getVersion()));
        }

        @Override
        protected Entity mapFull(ResultSetReader r) {
            return toEntity(r);
        }

        @Override
        protected Entity mapPartial(ResultSetReader r) {
            return toEntity(r);
        }

        private Entity toEntity(ResultSetReader r) {
            var containerTypeStr = r.getColumn("containerType").getUtf8();
            ContainerType containerType = ContainerType.UNKNOWN;
            if (!containerTypeStr.isBlank()) {
                try {
                    containerType = ContainerType.valueOf(containerTypeStr);
                } catch (Throwable ignore) {}
            }
            return Entity.newBuilder()
                .setParentId(r.getColumn(parentColumn).getUtf8())
                .setId(r.getColumn("id").getUtf8())
                .setLocalId(r.getColumn("localId").getUtf8())
                .setContainerType(containerType)
                .setName(r.getColumn("name").getUtf8())
                .setDescription(r.getColumn("description").getUtf8())
                .setData(r.getColumn("data").getUtf8())
                .setProto(any(r, "proto"))
                .setCreatedAt(r.getColumn("createdAt").getInt64())
                .setCreatedBy(r.getColumn("createdBy").getUtf8())
                .setUpdatedAt(r.getColumn("updatedAt").getInt64())
                .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                .setVersion(r.getColumn("version").getInt32())
                .build();
        }

        private static Any any(ResultSetReader r, String columnName) {
            int protoIdx = r.getColumnIndex(columnName);
            if (protoIdx >= 0) {
                var bytes = r.getColumn(protoIdx).getString();
                if (bytes.length == 0) {
                    return Any.getDefaultInstance();
                }
                try {
                    return Any.parseFrom(bytes);
                } catch (InvalidProtocolBufferException e) {
                    throw new RuntimeException("Unable to parse column " + columnName + " at row " + r.getRowCount());
                }
            }
            return Any.getDefaultInstance();
        }
    }
}
