package ru.yandex.solomon.conf.db3.ydb;

import java.util.Arrays;
import java.util.Collections;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.conf.db3.QuickLinksDao;
import ru.yandex.solomon.conf.db3.QuickLinksRecord;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.fromJsonArray;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;


/**
 * @author Oleg Baryshnikov
 */
public class YdbQuickLinksDao implements QuickLinksDao {

    private static final QueryTemplate TEMPLATE = new QueryTemplate(
            YdbQuickLinksDao.class,
            "quick_links", Arrays.asList(
            "read",
            "insert",
            "upsert",
            "delete"
    ));

    private final QuickLinksTable table;
    private final QueryText queryText;

    public YdbQuickLinksDao(TableClient tableClient, String tablePath, ObjectMapper objectMapper) {
        this.table = new QuickLinksTable(tableClient, tablePath, objectMapper);
        this.queryText = TEMPLATE.build(Collections.singletonMap("quick.links.table.path", tablePath));
    }

    @Override
    public CompletableFuture<Optional<QuickLinksRecord>> read(String id) {
        try {
            String query = queryText.query("read");
            Params params = Params.of("$projectId", utf8(id));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> insert(QuickLinksRecord projectMenu) {
        try {
            String query = queryText.query("insert");
            return table.insertOne(query, projectMenu);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<QuickLinksRecord>> upsert(QuickLinksRecord quickLinksRecord) {
        try {
            String query = queryText.query("upsert");
            return table.updateOne(query, quickLinksRecord)
                    .thenApply(newMenu -> {
                        final int oldVersion = quickLinksRecord.getVersion();
                        if (newMenu.isPresent()) {
                            return newMenu.filter(m -> m.getVersion() == oldVersion);
                        }
                        return Optional.of(quickLinksRecord.toBuilder().setVersion(oldVersion + 1).build());
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(String id) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$projectId", utf8(id));
            return table.queryBool(query, params);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    private static final class QuickLinksTable extends YdbTable<String, QuickLinksRecord> {

        private final ObjectMapper objectMapper;

        QuickLinksTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("projectId", PrimitiveType.utf8())
                    .addNullableColumn("items", PrimitiveType.utf8())
                    .addNullableColumn("createdAt", PrimitiveType.int64())
                    .addNullableColumn("updatedAt", PrimitiveType.int64())
                    .addNullableColumn("createdBy", PrimitiveType.utf8())
                    .addNullableColumn("updatedBy", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.int32())
                    .setPrimaryKey("projectId")
                    .build();
        }

        @Override
        protected String getId(QuickLinksRecord projectMenu) {
            return projectMenu.getProjectId();
        }

        @Override
        protected Params toParams(QuickLinksRecord menu) {
            return Params.create()
                    .put("$projectId", utf8(menu.getProjectId()))
                    .put("$items", utf8(toJson(objectMapper, menu.getItems())))
                    .put("$createdAt", int64(menu.getCreatedAtMillis()))
                    .put("$updatedAt", int64(menu.getUpdatedAtMillis()))
                    .put("$createdBy", utf8(menu.getCreatedBy()))
                    .put("$updatedBy", utf8(menu.getUpdatedBy()))
                    .put("$version", int32(menu.getVersion()));
        }

        @Override
        protected QuickLinksRecord mapFull(ResultSetReader r) {
            return QuickLinksRecord.newBuilder()
                    .setProjectId(r.getColumn("projectId").getUtf8())
                    .setItems(fromJsonArray(objectMapper, r.getColumn("items").getUtf8(), QuickLinksRecord.LinkItem.class))
                    .setCreatedAtMillis(r.getColumn("createdAt").getInt64())
                    .setCreatedBy(r.getColumn("createdBy").getUtf8())
                    .setUpdatedAtMillis(r.getColumn("updatedAt").getInt64())
                    .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                    .setVersion(r.getColumn("version").getInt32())
                    .build();
        }

        @Override
        protected QuickLinksRecord mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
