package ru.yandex.solomon.conf.db3;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeFalse;
import static org.junit.Assume.assumeTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;


/**
 * @author Oleg Baryshnikov
 */
@YaExternal
@ParametersAreNonnullByDefault
public abstract class AbstractCloudServiceDashboardsDaoTest {

    protected abstract CloudServiceDashboardsDao getDao();

    @Test
    public void insertOneServiceDashboard() {
        CloudServiceDashboardRecord config = makeSimpleServiceDashboard("compute", "compute", "compute_dashboard");
        assertTrue(insertSync(config));
        Optional<CloudServiceDashboardRecord> fromDb = readSync(config.getId());
        assertTrue(fromDb.isPresent());
        Assert.assertEquals(config, fromDb.get());
    }

    @Test
    public void insertServiceDashboardTwice() {
        CloudServiceDashboardRecord config = makeSimpleServiceDashboard("compute", "compute", "compute_dashboard");
        assertTrue(insertSync(config));
        assertFalse(insertSync(config));
    }

    @Test
    public void insertDifferentDashboardsById() {
        assertTrue(insertSync(makeSimpleServiceDashboard("compute", "compute1", "compute_dashboard1")));
        assertTrue(insertSync(makeSimpleServiceDashboard("compute", "compute2", "compute_dashboard2")));
    }

    @Test
    public void listByParentId() {
        List<CloudServiceDashboardRecord> configs = generateServiceDashboards();
        List<CloudServiceDashboardRecord> expectedConfigs = configs.stream()
                .filter(entity -> "serverless".equals(entity.getService()))
                .sorted(Comparator.comparing(CloudServiceDashboardRecord::getName))
                .collect(Collectors.toList());

        List<CloudServiceDashboardRecord> solomonGraphsFromDb = listByProjectIdSync(Set.of("serverless"), "", 0, "").getItems();
        assertEquals(expectedConfigs, solomonGraphsFromDb);
    }

    @Test
    public void listByParentIdWithDefaultLimit() {
        List<CloudServiceDashboardRecord> entities = IntStream.range(0, 110)
                .mapToObj(num -> makeSimpleServiceDashboard("compute", "entity" + num, "entity " + num))
                .collect(Collectors.toList());

        for (CloudServiceDashboardRecord entity : entities) {
            assumeTrue(insertSync(entity));
        }

        List<CloudServiceDashboardRecord> expectedEntities = entities.stream()
                .sorted(Comparator.comparing(CloudServiceDashboardRecord::getName))
                .limit(100)
                .collect(Collectors.toList());

        List<CloudServiceDashboardRecord> solomonGraphsFromDb = listByProjectIdSync(Set.of("compute"), "", 0, "").getItems();
        assertEquals(expectedEntities, solomonGraphsFromDb);
    }

    @Test
    public void listByParentIdWithFilterByName() {
        List<CloudServiceDashboardRecord> configs = Arrays.asList(
                makeSimpleServiceDashboard("compute", "compute1", "compute"),
                makeSimpleServiceDashboard("serverless", "serverless1", "serverless 1"),
                makeSimpleServiceDashboard("serverless", "serverless10", "serverless 10"),
                makeSimpleServiceDashboard("serverless", "serverless3", "serverless 3"),
                makeSimpleServiceDashboard("serverless", "serverless4", "serverless 4"),
                makeSimpleServiceDashboard("mdb-clickhouse", "ch1", "ch 1"),
                makeSimpleServiceDashboard("mdb-clickhouse", "ch2", "ch 2")
        );
        for (CloudServiceDashboardRecord entity : configs) {
            assumeTrue(insertSync(entity));
        }
        List<CloudServiceDashboardRecord> expectedEntities = List.of(
                makeSimpleServiceDashboard("serverless", "serverless1", "serverless 1"),
                makeSimpleServiceDashboard("serverless", "serverless10", "serverless 10")
        );
        List<CloudServiceDashboardRecord> solomonEntitiesFromDb = listByProjectIdSync(Set.of("serverless"), "1", 0, "").getItems();
        assertEquals(expectedEntities, solomonEntitiesFromDb);
    }

    @Test
    public void listByParentIdWithLimit() {
        List<CloudServiceDashboardRecord> entities = Arrays.asList(
                makeSimpleServiceDashboard("rtmr", "d1", "dash 1"),
                makeSimpleServiceDashboard("rtmr", "d2", "dash 2"),
                makeSimpleServiceDashboard("rtmr", "d3", "dash 3"),
                makeSimpleServiceDashboard("rtmr", "d4", "dash 4"),
                makeSimpleServiceDashboard("solomon", "d5", "dash 5")
        );

        for (CloudServiceDashboardRecord entity : entities) {
            assumeTrue(insertSync(entity));
        }

        {
            List<CloudServiceDashboardRecord> expectedEntities = List.of(
                    makeSimpleServiceDashboard("rtmr", "d1", "dash 1"),
                    makeSimpleServiceDashboard("rtmr", "d2", "dash 2")
            );

            TokenBasePage<CloudServiceDashboardRecord> page = listByProjectIdSync(Set.of("rtmr"), "", 2, "");

            assertEquals(expectedEntities, page.getItems());
            assertEquals("2", page.getNextPageToken());
        }

        {
            List<CloudServiceDashboardRecord> expectedEntities = List.of(
                    makeSimpleServiceDashboard("rtmr", "d3", "dash 3"),
                    makeSimpleServiceDashboard("rtmr", "d4", "dash 4")
            );

            TokenBasePage<CloudServiceDashboardRecord> page = listByProjectIdSync(Set.of("rtmr"), "", 2, "2");

            assertEquals(expectedEntities, page.getItems());
            assertEquals("", page.getNextPageToken());
        }
    }

    @Test
    public void listAll() {
        List<CloudServiceDashboardRecord> entities = generateServiceDashboards();

        {
            TokenBasePage<CloudServiceDashboardRecord> result = listAllSync("", 10, "");
            List<CloudServiceDashboardRecord> expected = entities.stream()
                    .sorted(Comparator.comparing(CloudServiceDashboardRecord::getId))
                    .limit(10)
                    .collect(Collectors.toList());
            assertEquals(expected, result.getItems());
            assertEquals("10", result.getNextPageToken());
        }

        {
            TokenBasePage<CloudServiceDashboardRecord> result = listAllSync("", 10, "10");
            List<CloudServiceDashboardRecord> expected = entities.stream()
                    .sorted(Comparator.comparing(CloudServiceDashboardRecord::getId))
                    .skip(10)
                    .limit(10)
                    .collect(Collectors.toList());
            assertEquals(expected, result.getItems());
            assertEquals("", result.getNextPageToken());
        }

        {
            TokenBasePage<CloudServiceDashboardRecord> result = listAllSync("d3", 10, "");
            List<CloudServiceDashboardRecord> expected = entities.stream()
                    .sorted(Comparator.comparing(CloudServiceDashboardRecord::getId))
                    .filter(entity -> StringUtils.containsIgnoreCase(entity.getName(), "d3"))
                    .limit(10)
                    .collect(Collectors.toList());
            assertEquals(expected, result.getItems());
            assertEquals("", result.getNextPageToken());
        }
    }

    private List<CloudServiceDashboardRecord> generateServiceDashboards() {
        List<String> services = List.of("serverless", "compute", "monitoring");
        List<CloudServiceDashboardRecord> entities = new ArrayList<>(services.size() * 4);

        for (String projectId : services) {
            entities.add(makeSimpleServiceDashboard(projectId, projectId + "_d1", "entity 1"));
            entities.add(makeSimpleServiceDashboard(projectId, projectId + "_d2", "entity 2"));
            entities.add(makeSimpleServiceDashboard(projectId, projectId + "_d3", "d3"));
            entities.add(makeSimpleServiceDashboard(projectId, projectId + "_d4", "entity 4"));
        }

        for (CloudServiceDashboardRecord entity : entities) {
            assumeTrue(insertSync(entity));
        }
        return entities;
    }

    @Test
    public void partialUpdate() {
        CloudServiceDashboardRecord entity = CloudServiceDashboardRecord.newBuilder()
                .setService("compute")
                .setId("compute_dashboard")
                .setName("compute_dashboard")
                .setDescription("description")
                .setData("data")
                .setCreatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setCreatedBy("user1")
                .setUpdatedBy("user1")
                .build();

        CloudServiceDashboardRecord entityToUpdate = CloudServiceDashboardRecord.newBuilder()
                .setService("compute")
                .setId("compute_dashboard")
                .setName("compute_dashboard (updated)")
                .setDescription("description (updated)")
                .setData("other data")
                .setCreatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setCreatedBy("user2")
                .setUpdatedBy("user2")
                .setVersion(0)
                .build();

        assumeTrue(insertSync(entity));

        Optional<CloudServiceDashboardRecord> updatedServiceDashboardOpt = updatePartialSync(entityToUpdate);
        assertTrue(updatedServiceDashboardOpt.isPresent());
        CloudServiceDashboardRecord updatedServiceDashboard = updatedServiceDashboardOpt.get();
        CloudServiceDashboardRecord expectedServiceDashboard = entityToUpdate.toBuilder()
                .setCreatedAt(entity.getCreatedAt())
                .setCreatedBy(entity.getCreatedBy())
                .setVersion(entity.getVersion() + 1)
                .build();
        Assert.assertEquals(expectedServiceDashboard, updatedServiceDashboard);
    }

    @Test
    public void upsert() {
        CloudServiceDashboardRecord config = CloudServiceDashboardRecord.newBuilder()
                .setService("serverless")
                .setId("serverless_dashboard")
                .setName("Serverless Dashboard")
                .setDescription("description")
                .setData("data")
                .setCreatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setCreatedBy("user1")
                .setUpdatedBy("user1")
                .build();

        CloudServiceDashboardRecord configToUpsert = CloudServiceDashboardRecord.newBuilder()
                .setService("serverless")
                .setId("serverless_dashboard")
                .setName("Serverless dashboard (upserted)")
                .setDescription("description (updated)")
                .setData("other data")
                .setCreatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setCreatedBy("user2")
                .setUpdatedBy("user2")
                .setVersion(2)
                .build();

        assumeTrue(insertSync(config));

        upsertSync(configToUpsert);

        Optional<CloudServiceDashboardRecord> updatedServiceDashboardOpt = readSync(configToUpsert.getId());
        assertTrue(updatedServiceDashboardOpt.isPresent());

        CloudServiceDashboardRecord updatedServiceDashboard = updatedServiceDashboardOpt.get();
        Assert.assertEquals(configToUpsert, updatedServiceDashboard);
    }

    @Test
    public void deleteOne() {
        CloudServiceDashboardRecord config = makeSimpleServiceDashboard("mdb", "mdb", "mdb");

        assumeTrue(insertSync(config));

        assertTrue(deleteSync(config.getId()));
        assumeFalse(readSync(config.getId()).isPresent());

        assertFalse(deleteSync( "mdb"));
        assertFalse(deleteSync("not_mdb"));
    }

    @Test
    public void deleteByParentId() {
        CloudServiceDashboardRecord configA = makeSimpleServiceDashboard("service1", "configA", "configA");
        CloudServiceDashboardRecord configB = makeSimpleServiceDashboard("service1", "configB", "configB");
        CloudServiceDashboardRecord configX = makeSimpleServiceDashboard("service2", "configX", "configX");
        CloudServiceDashboardRecord configC = makeSimpleServiceDashboard("service2", "configC", "configC");
        assumeTrue(insertSync(configA));
        assumeTrue(insertSync(configB));
        assumeTrue(insertSync(configX));
        assumeTrue(insertSync(configC));

        join(getDao().deleteByService("service1"));

        assertFalse(existsSync(configA.getId()));
        assertFalse(existsSync( configB.getId()));
        assertTrue(existsSync( configX.getId()));
        assertTrue(existsSync( configC.getId()));
    }

    @Test
    public void exists() {
        CloudServiceDashboardRecord config = makeSimpleServiceDashboard("service", "config", "config");

        assumeTrue(insertSync(config));

        assertTrue(existsSync("config"));
        assertFalse(existsSync("config2"));
    }

    private static CloudServiceDashboardRecord makeSimpleServiceDashboard(String service, String id, String name) {
        return CloudServiceDashboardRecord.newBuilder()
                .setService(service)
                .setId(id)
                .setName(name)
                .setDescription("description")
                .setData("data")
                .setCreatedAt(Instant.parse("2019-11-20T13:28:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:28:00Z").toEpochMilli())
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .build();
    }

    private boolean insertSync(CloudServiceDashboardRecord graph) {
        return join(getDao().insert(graph));
    }

    private Optional<CloudServiceDashboardRecord> readSync(String id) {
        return join(getDao().read(id));
    }

    private boolean deleteSync(String id) {
        return join(getDao().delete(id));
    }

    private boolean existsSync(String id) {
        return join(getDao().exists(id));
    }

    private TokenBasePage<CloudServiceDashboardRecord> listByProjectIdSync(Set<String> services, String filterByName, int pageSize, String pageToken) {
        return join(getDao().list(services, filterByName, pageSize, pageToken));
    }

    private TokenBasePage<CloudServiceDashboardRecord> listAllSync(String filterByName, int pageSize, String pageToken) {
        return join(getDao().listAll(filterByName, pageSize, pageToken));
    }

    private Optional<CloudServiceDashboardRecord> updatePartialSync(CloudServiceDashboardRecord config) {
        return join(getDao().update(config));
    }

    private void upsertSync(CloudServiceDashboardRecord config) {
        join(getDao().upsert(config));
    }
}
