package ru.yandex.solomon.conf.db3;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import com.google.protobuf.Any;
import com.google.protobuf.StringValue;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeFalse;
import static org.junit.Assume.assumeTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;


/**
 * @author Oleg Baryshnikov
 */
@YaExternal
@ParametersAreNonnullByDefault
public abstract class AbstractEntitiesDaoTest {

    protected abstract EntitiesDao getDao();

    @Test
    public void insertOneEntity() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph");
        assertTrue(insertSync(graph));
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());
        Assert.assertEquals(graph, fromDb.get());
    }

    @Test
    public void insertOneEntity_old() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setProto(null)
                .build();
        assertTrue(insertSync(graph));
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());
        assertFalse(fromDb.get().hasProto());
        Assert.assertEquals(graph, fromDb.get());
    }

    @Test
    public void insertEntityTwice() {
        Entity graph = makeSimpleEntity("project", "graph", "graph");
        assertTrue(insertSync(graph));
        assertFalse(insertSync(graph));
    }

    @Test
    public void insertDifferentEntriesById() {
        assertTrue(insertSync(makeSimpleEntity("project", "graph1", "graph")));
        assertTrue(insertSync(makeSimpleEntity("project", "graph2", "graph")));
    }

    @Test
    public void listByParentId() {
        List<Entity> entities = generateEntities();
        List<Entity> expectedEntities = entities.stream()
                .filter(entity -> "solomon".equals(entity.getParentId()))
                .sorted(Comparator.comparing(Entity::getName))
                .collect(Collectors.toList());

        List<Entity> solomonGraphsFromDb = listByProjectIdSync("solomon", "", 0, "").getItems();
        solomonGraphsFromDb.sort(Comparator.comparing(Entity::getName));
        assertEquals(expectedEntities, solomonGraphsFromDb);
    }

    @Test
    public void listByLocalId() {
        List<Entity> entities = generateEntities().stream()
                .map(entity -> entity.toBuilder().setLocalId(UUID.randomUUID().toString()).setId(UUID.randomUUID().toString()).build())
                .collect(Collectors.toList());
        for (Entity entity : entities) {
            assertTrue(insertSync(entity));
        }
        List<Entity> expectedEntities = List.of(entities.get(2));

        List<Entity> solomonGraphsFromDb = listByProjectIdLocalIdSync(entities.get(2).getParentId(), entities.get(2).getLocalId(), 0, "").getItems();
        assertEquals(expectedEntities, solomonGraphsFromDb);
    }

    @Test
    public void listByParentIdWithDefaultLimit() {
        List<Entity> entities = IntStream.range(0, 110)
                .mapToObj(num -> makeSimpleEntity("solomon", "entity" + num, "entity " + num))
                .collect(Collectors.toList());

        for (Entity entity : entities) {
            assumeTrue(insertSync(entity));
        }

        List<Entity> expectedEntities = entities.stream()
                .sorted(Comparator.comparing(Entity::getName))
                .limit(100)
                .collect(Collectors.toList());

        List<Entity> solomonGraphsFromDb = listByProjectIdSync("solomon", "", 0, "").getItems();
        assertEquals(expectedEntities, solomonGraphsFromDb);
    }

    @Test
    public void listByParentIdWithFilterByName() {
        List<Entity> entities = Arrays.asList(
                makeSimpleEntity("solomon", "graph1", "g1"),
                makeSimpleEntity("solomon", "graph2", "graph 2"),
                makeSimpleEntity("solomon", "graph3", "g3"),
                makeSimpleEntity("solomon", "graph4", "g4"),
                makeSimpleEntity("solomon", "dash", "graph"),
                makeSimpleEntity("yt", "graph5", "graph 5"),
                makeSimpleEntity("yt", "dash1", "dash 1")
        );
        for (Entity entity : entities) {
            assumeTrue(insertSync(entity));
        }
        List<Entity> expectedEntities = List.of(
                makeSimpleEntity("solomon", "dash", "graph"),
                makeSimpleEntity("solomon", "graph2", "graph 2")
        );
        List<Entity> solomonEntitiesFromDb = listByProjectIdSync("solomon", "graph", 0, "").getItems();
        assertEquals(expectedEntities, solomonEntitiesFromDb);
    }

    @Test
    public void listByParentIdWithLimit() {
        List<Entity> entities = Arrays.asList(
                makeSimpleEntity("rtmr", "d1", "dash 1"),
                makeSimpleEntity("rtmr", "d2", "dash 2"),
                makeSimpleEntity("rtmr", "d3", "dash 3"),
                makeSimpleEntity("rtmr", "d4", "dash 4"),
                makeSimpleEntity("solomon", "d5", "dash 5")
        );

        for (Entity entity : entities) {
            assumeTrue(insertSync(entity));
        }

        {
            List<Entity> expectedEntities = List.of(
                    makeSimpleEntity("rtmr", "d1", "dash 1"),
                    makeSimpleEntity("rtmr", "d2", "dash 2")
            );

            TokenBasePage<Entity> page = listByProjectIdSync("rtmr", "", 2, "");

            assertEquals(expectedEntities, page.getItems());
            assertEquals("2", page.getNextPageToken());
        }

        {
            List<Entity> expectedEntities = List.of(
                    makeSimpleEntity("rtmr", "d3", "dash 3"),
                    makeSimpleEntity("rtmr", "d4", "dash 4")
            );

            TokenBasePage<Entity> page = listByProjectIdSync("rtmr", "", 2, "2");

            assertEquals(expectedEntities, page.getItems());
            assertEquals("", page.getNextPageToken());
        }
    }

    @Test
    public void listAll() {
        List<Entity> entities = generateEntities();

        {
            TokenBasePage<Entity> result = listAllSync("", 10, "");
            List<Entity> expected = entities.stream()
                    .sorted(Comparator.comparing(Entity::getId))
                    .limit(10)
                    .collect(Collectors.toList());
            assertEquals(expected, result.getItems());
            assertEquals("10", result.getNextPageToken());
        }

        {
            TokenBasePage<Entity> result = listAllSync("", 10, "10");
            List<Entity> expected = entities.stream()
                    .sorted(Comparator.comparing(Entity::getId))
                    .skip(10)
                    .limit(10)
                    .collect(Collectors.toList());
            assertEquals(expected, result.getItems());
            assertEquals("", result.getNextPageToken());
        }

        {
            TokenBasePage<Entity> result = listAllSync("d3", 10, "");
            List<Entity> expected = entities.stream()
                    .sorted(Comparator.comparing(Entity::getId))
                    .filter(entity -> StringUtils.containsIgnoreCase(entity.getName(), "d3"))
                    .limit(10)
                    .collect(Collectors.toList());
            assertEquals(expected, result.getItems());
            assertEquals("", result.getNextPageToken());
        }
    }

    private List<Entity> generateEntities() {
        List<String> projectIds = List.of("solomon", "rtmr", "yt");
        List<Entity> entities = new ArrayList<>(projectIds.size() * 4);

        for (String projectId : projectIds) {
            entities.add(makeSimpleEntity(projectId, projectId + "_d1", "entity 1"));
            entities.add(makeSimpleEntity(projectId, projectId + "_d2", "entity 2"));
            entities.add(makeSimpleEntity(projectId, projectId + "_d3", "d3"));
            entities.add(makeSimpleEntity(projectId, projectId + "_d4", "entity 4"));
        }

        for (Entity entity : entities) {
            assumeTrue(insertSync(entity));
        }
        return entities;
    }

    @Test
    public void partialUpdate() {
        Entity entity = Entity.newBuilder()
                .setParentId("project")
                .setId("entity")
                .setContainerType(ContainerType.PROJECT)
                .setName("Entity")
                .setDescription("description")
                .setData("{}")
                .setCreatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setCreatedBy("user1")
                .setCreatedBy("user1")
                .build();

        Entity entityToUpdate = Entity.newBuilder()
                .setParentId("project")
                .setId("entity")
                .setContainerType(ContainerType.PROJECT)
                .setName("Entity (updated)")
                .setDescription("description (updated)")
                .setData("{\"key\":\"value\"}")
                .setCreatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setCreatedBy("user2")
                .setUpdatedBy("user2")
                .setVersion(0)
                .build();

        assumeTrue(insertSync(entity));

        Optional<Entity> updatedEntityOpt = updatePartialSync(entityToUpdate);
        assertTrue(updatedEntityOpt.isPresent());

        Entity updatedEntity = updatedEntityOpt.get();
        Entity expectedEntity = entityToUpdate.toBuilder()
                .setCreatedAt(entity.getCreatedAt())
                .setCreatedBy(entity.getCreatedBy())
                .setVersion(entity.getVersion() + 1)
                .build();
        Assert.assertEquals(expectedEntity, updatedEntity);
    }

    @Test
    public void updateWithoutVersion() {
        Entity entity = Entity.newBuilder()
                .setParentId("project")
                .setId("entity")
                .setContainerType(ContainerType.PROJECT)
                .setName("Entity")
                .setDescription("description")
                .setData("{}")
                .setCreatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setCreatedBy("user1")
                .setCreatedBy("user1")
                .build();

        Entity entityToUpdate = Entity.newBuilder()
                .setParentId("project")
                .setId("entity")
                .setContainerType(ContainerType.PROJECT)
                .setName("Entity (updated)")
                .setDescription("description (updated)")
                .setData("{\"key\":\"value\"}")
                .setCreatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setCreatedBy("user2")
                .setUpdatedBy("user2")
                .setVersion(-1)
                .build();

        assumeTrue(insertSync(entity));

        // Update entity 3 times
        for (int i = 0; i < 3; ++i) {
            updatePartialSync(entityToUpdate);
        }

        // Entity update №4
        Optional<Entity> updatedEntityOpt = updatePartialSync(entityToUpdate);
        assertTrue(updatedEntityOpt.isPresent());

        Entity updatedEntity = updatedEntityOpt.get();
        Entity expectedEntity = entityToUpdate.toBuilder()
                .setCreatedAt(entity.getCreatedAt())
                .setCreatedBy(entity.getCreatedBy())
                .setVersion(4)
                .build();
        Assert.assertEquals(expectedEntity, updatedEntity);
    }

    @Test
    public void upsert() {
        Entity entity = Entity.newBuilder()
                .setParentId("project")
                .setId("entity")
                .setContainerType(ContainerType.PROJECT)
                .setName("Entity")
                .setDescription("description")
                .setData("{}")
                .setCreatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:06:00Z").toEpochMilli())
                .setCreatedBy("user1")
                .setCreatedBy("user1")
                .build();

        Entity entityToUpsert = Entity.newBuilder()
                .setParentId("project")
                .setContainerType(ContainerType.PROJECT)
                .setId("entity")
                .setName("Entity (upserted)")
                .setDescription("description (updated)")
                .setData("{\"key\":\"value\"}")
                .setCreatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setCreatedBy("user2")
                .setUpdatedBy("user2")
                .setVersion(2)
                .build();

        assumeTrue(insertSync(entity));

        upsertSync(entityToUpsert);

        Optional<Entity> updatedEntityOpt = readSync(entityToUpsert);
        assertTrue(updatedEntityOpt.isPresent());

        Entity updatedEntity = updatedEntityOpt.get();
        Assert.assertEquals(entityToUpsert, updatedEntity);
    }

    @Test
    public void deleteOne() {
        Entity entity = makeSimpleEntity("project", "entity", "entity");

        assumeTrue(insertSync(entity));

        assertTrue(deleteSync(entity.getParentId(), entity.getId()));
        assumeFalse(readSync(entity).isPresent());

        assertFalse(deleteSync("project", "graph_2"));
        assertFalse(deleteSync("project", "graph"));

        entity = makeSimpleEntity("project", "graph", "graph");
        assumeTrue(insertSync(entity));
        assertFalse(deleteSync("project", "xyz"));
        Entity found = readSync(entity).orElseThrow(AssertionError::new);
        assertTrue(EqualsBuilder.reflectionEquals(entity, found));
        assertTrue(deleteSync(entity.getParentId(), entity.getId()));
        assumeFalse(readSync(entity).isPresent());
    }

    @Test
    public void deleteWithVersion() {
        Entity entity = Entity.newBuilder()
                .setParentId("project")
                .setId("entity")
                .setContainerType(ContainerType.PROJECT)
                .setName("Entity (upserted)")
                .setDescription("description (updated)")
                .setData("{\"key\":\"value\"}")
                .setCreatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:08:00Z").toEpochMilli())
                .setCreatedBy("user2")
                .setUpdatedBy("user2")
                .setVersion(2)
                .build();

        assumeTrue(insertSync(entity));
        assertFalse(deleteSync(entity.getParentId(), entity.getId(), 222));
        assertTrue(deleteSync(entity.getParentId(), entity.getId(), 2));
        assumeFalse(readSync(entity).isPresent());

        assumeTrue(insertSync(entity));
        assertTrue(deleteSync(entity.getParentId(), entity.getId(), -1));
        assumeFalse(readSync(entity).isPresent());
    }

    @Test
    public void deleteByParentId() {
        Entity graphA = makeSimpleEntity("project1", "graphA", "graphA");
        Entity dashB = makeSimpleEntity("project1", "graphB", "graphB");
        Entity graphX = makeSimpleEntity("project2", "graphX", "graphX");
        Entity graphC = makeSimpleEntity("project2", "graphC", "graphC");
        assumeTrue(insertSync(graphA));
        assumeTrue(insertSync(dashB));
        assumeTrue(insertSync(graphX));
        assumeTrue(insertSync(graphC));

        join(getDao().deleteByParentId("project1"));

        assertFalse(existsSync("project1", graphA.getId()));
        assertFalse(existsSync("project1", dashB.getId()));
        assertTrue(existsSync("project2", graphX.getId()));
        assertTrue(existsSync("project2", graphC.getId()));
    }

    @Test
    public void exists() {
        Entity graph = makeSimpleEntity("project", "graph", "graph");

        assumeTrue(insertSync(graph));

        assertTrue(existsSync("project", "graph"));
        assertFalse(existsSync("project", "graph2"));
        assertTrue(existsSync(graph.getParentId(), graph.getId()));
        assertFalse(existsSync("project2", "graph"));
    }

    @Test
    public void readById() {
        Entity graph = makeSimpleEntity("project", "graph", "graph");
        assertTrue(insertSync(graph));
        Entity found = readByIdSync(graph).orElseThrow(AssertionError::new);
        assertTrue(EqualsBuilder.reflectionEquals(graph, found));
    }

    private static Entity makeSimpleEntity(String parentId, String id, String name) {
        return Entity.newBuilder()
                .setParentId(parentId)
                .setId(id)
                .setContainerType(ContainerType.PROJECT)
                .setName(name)
                .setDescription("description")
                .setData("{}")
                .setProto(Any.pack(StringValue.of("some value here")))
                .setCreatedAt(Instant.parse("2019-11-20T13:28:00Z").toEpochMilli())
                .setUpdatedAt(Instant.parse("2019-11-20T13:28:00Z").toEpochMilli())
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .build();
    }

    @Test
    public void insertUniqLocalId_without_local() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph");
        assertTrue(insertSync(graph));
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());

        Entity graph2 = makeSimpleEntity("solomon2", "graph2", "graph");
        assertTrue(insertSync(graph2));
        fromDb = readSync(graph2);
        assertTrue(fromDb.isPresent());
    }

    @Test
    public void updateUniqLocalId_without_local() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph");
        assertTrue(insertSync(graph));

        Entity graph2 = makeSimpleEntity("solomon2", "graph2", "graph");
        assertTrue(insertSync(graph2));

        updatePartialSync(graph.toBuilder().setDescription("123").build());
        Optional<Entity> fromDb = readSync(graph);
        assertEquals("123", fromDb.get().getDescription());

        updatePartialSync(graph2.toBuilder().setDescription("123").build());
        fromDb = readSync(graph2);
        assertEquals("123", fromDb.get().getDescription());
    }

    @Test
    public void upsertUniqLocalId_without_local() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph");
        upsertSync(graph);
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());

        Entity graph2 = makeSimpleEntity("solomon2", "graph2", "graph");
        upsertSync(graph2);
        fromDb = readSync(graph2);
        assertTrue(fromDb.isPresent());
    }

    @Test
    public void insertUniqLocalId_differentParent() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        assertTrue(insertSync(graph));
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());

        Entity graph2 = makeSimpleEntity("solomon2", "graph2", "graph").toBuilder()
                .setLocalId("id")
                .build();
        assertTrue(insertSync(graph2));
        fromDb = readSync(graph2);
        assertTrue(fromDb.isPresent());

        Entity graph3 = makeSimpleEntity("solomon2", "graph3", "graph").toBuilder()
                .setLocalId("id3")
                .build();
        assertTrue(insertSync(graph3));
    }

    @Test
    public void updateUniqLocalId_differentParent() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        assertTrue(insertSync(graph));

        Entity graph2 = makeSimpleEntity("solomon2", "graph2", "graph").toBuilder()
                .setLocalId("id")
                .build();
        assertTrue(insertSync(graph2));

        Entity graph3 = makeSimpleEntity("solomon2", "graph3", "graph").toBuilder()
                .setLocalId("id3")
                .build();
        assertTrue(insertSync(graph3));

        updatePartialSync(graph.toBuilder().setDescription("123").build());
        Optional<Entity> fromDb = readSync(graph);
        assertEquals("123", fromDb.get().getDescription());

        updatePartialSync(graph2.toBuilder().setDescription("123").build());
        fromDb = readSync(graph2);
        assertEquals("123", fromDb.get().getDescription());

        updatePartialSync(graph.toBuilder().setName("id3").setVersion(-1).build());
        fromDb = readSync(graph);
        assertEquals("id3", fromDb.get().getName());
    }

    @Test
    public void upsertUniqLocalId_differentParent() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        upsertSync(graph);
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());

        Entity graph2 = makeSimpleEntity("solomon2", "graph2", "graph").toBuilder()
                .setLocalId("id")
                .build();
        upsertSync(graph2);
        fromDb = readSync(graph2);
        assertTrue(fromDb.isPresent());
    }

    @Test
    public void insertUniqLocalId_sameParent() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        assertTrue(insertSync(graph));
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());

        Entity graph2 = makeSimpleEntity("solomon", "graph2", "graph").toBuilder()
                .setLocalId("id")
                .build();
        try {
            assertTrue(insertSync(graph2));
        } catch (CompletionException e) {
            assertTrue(Throwables.getRootCause(e).getMessage().contains("Name must be uniq in parentId"));
        }
        fromDb = readSync(graph2);
        assertTrue(fromDb.isEmpty());
    }

    @Test
    public void upsertUniqLocalId_sameParent() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        upsertSync(graph);
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());

        Entity graph2 = makeSimpleEntity("solomon", "graph2", "graph").toBuilder()
                .setLocalId("id")
                .build();
        try {
            upsertSync(graph2);
        } catch (CompletionException e) {
            assertTrue(Throwables.getRootCause(e).getMessage().contains("Name must be uniq in parentId"));
        }
        fromDb = readSync(graph2);
        assertTrue(fromDb.isEmpty());
    }

    @Test
    public void updateUniqLocalId_sameParent() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        assertTrue(insertSync(graph));

        Entity graph2 = makeSimpleEntity("solomon", "graph2", "graph").toBuilder()
                .setLocalId("id2")
                .build();
        assertTrue(insertSync(graph2));

        updatePartialSync(graph.toBuilder().setName("id3").build());
        Optional<Entity> fromDb = readSync(graph);
        assertEquals("id3", fromDb.get().getName());

        try {
            updatePartialSync(graph2.toBuilder().setName("id3").build());
        } catch (CompletionException e) {
            assertTrue(Throwables.getRootCause(e).getMessage().contains("Name must be uniq in parentId"));
        }
        fromDb = readSync(graph2);
        assertEquals(fromDb.get().getLocalId(), "id2");
    }

    @Test
    public void upsertUniqLocalId_sameId() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph").toBuilder()
                .setLocalId("id")
                .build();
        upsertSync(graph);
        upsertSync(graph = graph.toBuilder().setName("123").build());
        upsertSync(graph = graph.toBuilder().setLocalId("id2").build());
        Optional<Entity> fromDb = readSync(graph);
        assertTrue(fromDb.isPresent());
        assertEquals(fromDb.get().getName(), "123");
        assertEquals(fromDb.get().getLocalId(), "id2");
    }

    @Test
    public void updateUniqLocalId_sameId() {
        Entity graph = makeSimpleEntity("solomon", "graph", "graph");
        assertTrue(insertSync(graph));
        updatePartialSync(graph = graph.toBuilder().setName("name-name").build());
        updatePartialSync(graph = graph.toBuilder().setLocalId("id3").setVersion(-1).build());
        updatePartialSync(graph = graph.toBuilder().setLocalId("id4").setVersion(-1).build());
        Optional<Entity> fromDb = readSync(graph);
        assertEquals(graph.getName(), fromDb.get().getName());
        assertEquals(graph.getLocalId(), fromDb.get().getLocalId());
        updatePartialSync(graph = graph.toBuilder().setLocalId("").setVersion(-1).build());
        fromDb = readSync(graph);
        assertEquals(graph.getLocalId(), fromDb.get().getLocalId());
    }

    private boolean insertSync(Entity graph) {
        return join(getDao().insert(graph));
    }

    private Optional<Entity> readSync(Entity graph) {
        return join(getDao().read(graph.getParentId(), graph.getId()));
    }

    private Optional<Entity> readByIdSync(Entity graph) {
        return join(getDao().readById(graph.getId()));
    }

    private boolean deleteSync(String parentId, String entityId) {
        return join(getDao().delete(parentId, entityId));
    }

    private boolean deleteSync(String parentId, String entityId, int version) {
        return join(getDao().deleteWithVersion(parentId, entityId, version));
    }

    private boolean existsSync(String parentId, String entityId) {
        return join(getDao().exists(parentId, entityId));
    }

    private TokenBasePage<Entity> listByProjectIdSync(String parentId, String filterByName, int pageSize, String pageToken) {
        return join(getDao().list(parentId, filterByName, pageSize, pageToken));
    }

    private TokenBasePage<Entity> listByProjectIdLocalIdSync(String parentId, String filter, int pageSize, String pageToken) {
        return join(getDao().listByLocalId(parentId, filter, pageSize, pageToken));
    }

    private TokenBasePage<Entity> listAllSync(String filterByName, int pageSize, String pageToken) {
        return join(getDao().listAll(filterByName, pageSize, pageToken));
    }

    private Optional<Entity> updatePartialSync(Entity config) {
        return join(getDao().update(config));
    }

    private void upsertSync(Entity config) {
        join(getDao().upsert(config));
    }
}
