package ru.yandex.solomon.conf.db3;

import java.time.Instant;
import java.util.Optional;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Test;

import ru.yandex.solomon.core.db.dao.DaoTestFixture;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.decrement;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.equalsByString;
import static ru.yandex.solomon.core.db.dao.DaoTestFixture.increment;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public abstract class AbstractQuickLinksDaoTest {

    public abstract QuickLinksDao getDao();

    @Test
    public void insert() {
        QuickLinksRecord.LinkItem[] items = new QuickLinksRecord.LinkItem[]{
                new QuickLinksRecord.LinkItem("Item 1", "/projects/solomon/explorer", null),
                new QuickLinksRecord.LinkItem(
                        "Item 2",
                        null,
                        new QuickLinksRecord.LinkItem[]{
                                new QuickLinksRecord.LinkItem("Subitem 1", "/projects/kikimr/explorer", null),
                                new QuickLinksRecord.LinkItem("Subitem 2", "/projects/yt/explorer", null),
                                new QuickLinksRecord.LinkItem("Subitem 3", "/projects/rtmr/explorer", null)
                        }
                ),
        };

        QuickLinksRecord quickLinksRecord = QuickLinksRecord.newBuilder()
                .setProjectId("solomon")
                .setItems(items)
                .setCreatedAt(Instant.EPOCH)
                .setUpdatedAt(Instant.EPOCH)
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .setVersion(0)
                .build();

        Optional<QuickLinksRecord> quickLinksUpsertedOpt = upsertSync(quickLinksRecord);
        assertTrue(quickLinksUpsertedOpt.isPresent());

        QuickLinksRecord quickLinksUpserted = quickLinksUpsertedOpt.get();
        assertEquals("solomon", quickLinksUpserted.getProjectId());
        assertArrayEquals(items, quickLinksUpserted.getItems());
        assertEquals(1, quickLinksUpserted.getVersion());
        assertEquals(Instant.EPOCH, quickLinksUpserted.getUpdatedAt());
        assertEquals("user", quickLinksUpserted.getUpdatedBy());

        assertFalse(insertSync(quickLinksRecord));

        quickLinksRecord = QuickLinksRecord.newBuilder()
                .setProjectId("other")
                .setItems(items)
                .setCreatedAt(Instant.EPOCH)
                .setUpdatedAt(Instant.EPOCH)
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .setVersion(0)
                .build();

        assertTrue(insertSync(quickLinksRecord));
        QuickLinksRecord found = readSync(quickLinksRecord.getProjectId()).orElseThrow(AssertionError::new);
        DaoTestFixture.equalsByString(quickLinksRecord, found);
    }

    @Test
    public void deleteById() {
        QuickLinksRecord quickLinks1 = QuickLinksRecord.newBuilder()
                .setProjectId("solomon")
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .build();

        QuickLinksRecord quickLinks2 = QuickLinksRecord.newBuilder()
                .setProjectId("junk")
                .setCreatedBy("user")
                .setUpdatedBy("user")
                .build();

        assertTrue(upsertSync(quickLinks1).isPresent());
        assertTrue(upsertSync(quickLinks2).isPresent());

        deleteSync("solomon");

        assertFalse(readSync("solomon").isPresent());
        assertTrue(readSync("junk").isPresent());
    }

    @Test
    public void insertThenFind() {
        QuickLinksRecord.Builder builder = quickLinks();
        QuickLinksRecord quickLinks = builder.build();
        Optional<QuickLinksRecord> newQuickLinksRecord = upsertSync(quickLinks);
        QuickLinksRecord actualMenu = newQuickLinksRecord.orElseThrow(AssertionError::new);

        increment(builder);
        QuickLinksRecord expectedMenu = builder.build();
        equalsByString(actualMenu, expectedMenu);

        QuickLinksRecord foundMenu = readSync(quickLinks.getProjectId()).orElseThrow(AssertionError::new);
        equalsByString(foundMenu, expectedMenu);
    }

    @Test
    public void updateThenFind() {
        QuickLinksRecord.Builder builder = quickLinks();
        QuickLinksRecord quickLinks = builder.build();
        upsertSync(quickLinks).orElseThrow(AssertionError::new);

        QuickLinksRecord.LinkItem[] menuUpdate = { quickLinksItem("Simple", new QuickLinksRecord.LinkItem[0]) };
        increment(builder).setItems(menuUpdate).setUpdatedBy("x");
        upsertSync(builder.build());

        QuickLinksRecord foundMenu = readSync(quickLinks.getProjectId()).orElseThrow(AssertionError::new);
        QuickLinksRecord expectedMenu = increment(builder).build();
        equalsByString(foundMenu, expectedMenu);

        assertFalse(upsertSync(decrement(builder).setUpdatedBy("yyy").build()).isPresent());
        foundMenu = readSync(quickLinks.getProjectId()).orElseThrow(AssertionError::new);
        equalsByString(foundMenu, expectedMenu);
    }

    @Test
    public void extraDeleteById() {
        QuickLinksRecord.Builder builder = quickLinks();
        QuickLinksRecord menuA = upsertSync(builder.setProjectId("project1").build()).orElseThrow(AssertionError::new);
        QuickLinksRecord menuB = upsertSync(builder.setProjectId("project2").build()).orElseThrow(AssertionError::new);
        QuickLinksRecord menuC = upsertSync(builder.setProjectId("project3").build()).orElseThrow(AssertionError::new);

        deleteSync("MenuD");

        readSync(menuA.getProjectId()).orElseThrow(AssertionError::new);
        readSync(menuB.getProjectId()).orElseThrow(AssertionError::new);
        readSync(menuC.getProjectId()).orElseThrow(AssertionError::new);

        deleteSync(menuB.getProjectId());
        readSync(menuA.getProjectId()).orElseThrow(AssertionError::new);
        assertFalse(readSync(menuB.getProjectId()).isPresent());
        readSync(menuC.getProjectId()).orElseThrow(AssertionError::new);
    }

    private boolean insertSync(QuickLinksRecord quickLinksRecord) {
        return join(getDao().insert(quickLinksRecord));
    }

    private Optional<QuickLinksRecord> upsertSync(QuickLinksRecord quickLinksRecord) {
        return join(getDao().upsert(quickLinksRecord));
    }

    private Optional<QuickLinksRecord> readSync(String id) {
        return join(getDao().read(id));
    }

    private void deleteSync(String id) {
        join(getDao().delete(id));
    }

    private static QuickLinksRecord.Builder quickLinks() {
        Instant now = Instant.ofEpochMilli(System.currentTimeMillis());
        return QuickLinksRecord
                .newBuilder()
                .setProjectId("solomon")
                .setItems(quickLinksItems())
                .setCreatedAt(now)
                .setCreatedBy("user")
                .setUpdatedAt(now)
                .setUpdatedBy("user")
                .setVersion(1);
    }

    private static QuickLinksRecord.LinkItem[] quickLinksItems() {
        return new QuickLinksRecord.LinkItem[]{
                quickLinksItem("A", 0),
                quickLinksItem("B", 1),
                quickLinksItem("C", 2) };
    }

    private static QuickLinksRecord.LinkItem quickLinksItem(String title, int childrenCount) {
        QuickLinksRecord.LinkItem[] children = IntStream
                .rangeClosed(1, childrenCount)
                .boxed()
                .map(index -> quickLinksItem(title + "." + index, childrenCount - 1))
                .toArray(QuickLinksRecord.LinkItem[]::new);
        return quickLinksItem(title, children);
    }

    private static QuickLinksRecord.LinkItem quickLinksItem(String title, QuickLinksRecord.LinkItem[] children) {
        return new QuickLinksRecord.LinkItem(title, "http://localhost/", children);
    }

}
