package ru.yandex.solomon.config;

import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.DoubleConsumer;
import java.util.function.IntConsumer;
import java.util.function.LongConsumer;

import javax.annotation.Nullable;

import ru.yandex.solomon.config.protobuf.DataSize;
import ru.yandex.solomon.config.protobuf.DataUnit;
import ru.yandex.solomon.config.protobuf.Time;

/**
 * @author Vladimir Gordiychuk
 */
public final class OptionalSet {
    private OptionalSet() {
    }

    public static <T> void setNonNull(Consumer<T> consumer, @Nullable T obj) {
        if (obj != null) {
            consumer.accept(obj);
        }
    }

    public static void setString(Consumer<String> consumer, String value) {
        if (!"".equals(value)) {
            consumer.accept(value);
        }
    }

    public static void setInt(IntConsumer consumer, int value) {
        if (value != 0) {
            consumer.accept(value);
        }
    }

    public static void setLong(LongConsumer consumer, long value) {
        if (value != 0) {
            consumer.accept(value);
        }
    }

    public static void setDouble(DoubleConsumer consumer, double value) {
        if (value != 0d) {
            consumer.accept(value);
        }
    }

    public static void setTime(BiConsumer<Long, TimeUnit> consumer, Time time) {
        if (time != Time.getDefaultInstance()) {
            consumer.accept(time.getValue(), TimeUnitConverter.protoToUnit(time.getUnit()));
        }
    }

    public static void setTimeMillis(IntConsumer consumer, Time time) {
        if (time != Time.getDefaultInstance()) {
            TimeUnit unit = TimeUnitConverter.protoToUnit(time.getUnit());
            consumer.accept(Math.toIntExact(unit.toMillis(time.getValue())));
        }
    }

    public static void setTimeOrDefaultMillis(IntConsumer consumer, Time time, Time defaultTime) {
        if (time != Time.getDefaultInstance()) {
            time = defaultTime;
        }

        TimeUnit unit = TimeUnitConverter.protoToUnit(time.getUnit());
        consumer.accept(Math.toIntExact(unit.toMillis(time.getValue())));
    }

    public static void setBytesSize(IntConsumer consumer, DataSize dataSize) {
        if (dataSize == DataSize.getDefaultInstance()) {
            return;
        }

        long size = DataSizeConverter.convert(dataSize.getValue(), dataSize.getUnit(), DataUnit.BYTES);
        if ((long)((int) size) != size) {
            throw new IllegalStateException("Int overflow: " + dataSize);
        }

        consumer.accept((int) size);
    }
}
