package ru.yandex.solomon.config;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;

import ru.yandex.misc.ExceptionUtils;

/**
 * @author alexlovkov
 */
public class SolomonConfigs {

    private static final String CONFIG_ARG = "--config";
    private static final String SECRETS_ARG = "--secrets";

    public static String getConfigFileFromArgs(String... args) {
        String argValue = findArgValue(CONFIG_ARG, args);
        Preconditions.checkNotNull(argValue, "argument %s not specified", CONFIG_ARG);
        return argValue;
    }

    public static Optional<String> getSecretsFileFromArgs(String... args) {
        String argValue = findArgValue(SECRETS_ARG, args);
        return Optional.ofNullable(argValue);
    }

    @Nullable
    static String findArgValue(String argName, String... args) {
        String configFile = null;
        for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if (arg.startsWith(argName)) {
                if (argName.equals(arg)) {
                    Preconditions.checkState(i < args.length - 1, "argument %s has no value", argName);
                    return args[i + 1];
                }
                configFile = arg.substring(argName.length() + 1);
            }
        }
        return configFile;
    }

    public static <T extends Message> T parseConfig(String configFile, T instance) {
        try {
            @SuppressWarnings("TypeParameterQualifier")
            T.Builder builder = instance.toBuilder();
            try (BufferedReader reader = Files.newBufferedReader(Paths.get(configFile))) {
                TextFormat.merge(reader, builder);
                mergeIncludes(builder, instance);
            }
            return (T) builder.build();
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    public static <T extends Message> T parseConfigWithoutInclude(String configFile, T instance) {
        try {
            @SuppressWarnings("TypeParameterQualifier")
            T.Builder builder = instance.toBuilder();
            try (BufferedReader reader = Files.newBufferedReader(Paths.get(configFile))) {
                TextFormat.merge(reader, builder);
            }
            return (T) builder.build();
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    @SuppressWarnings("TypeParameterQualifier")
    private static <T extends Message.Builder> T mergeIncludes(T builder, Message instance) {
        Descriptors.Descriptor descriptor = instance.getDescriptorForType();
        Descriptors.FieldDescriptor includeDescriptor = descriptor.findFieldByName("Include");
        if (includeDescriptor == null) {
            return builder;
        }

        if (!includeDescriptor.isRepeated()) {
            return builder;
        }

        if (includeDescriptor.getType() != Descriptors.FieldDescriptor.Type.STRING) {
            return builder;
        }

        List<String> files = (List<String>) builder.getField(includeDescriptor);
        for (String file: files) {
            try (BufferedReader reader = Files.newBufferedReader(Paths.get(file))) {
                Message.Builder includedBuilder = instance.toBuilder();
                TextFormat.merge(reader, includedBuilder);
                builder.mergeFrom(includedBuilder.build());
            } catch (IOException e) {
                throw ExceptionUtils.throwException(e);
            }
        }
        return builder;
    }
}
