package ru.yandex.solomon.config.netty;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import io.netty.util.concurrent.DefaultEventExecutorGroup;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.concurrent.EventExecutorGroup;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.RejectedExecutionHandler;
import io.netty.util.concurrent.SingleThreadEventExecutor;
import io.netty.util.concurrent.ThreadProperties;


/**
 * @author Sergey Polovko
 */
public class NettyGeneralExecutor extends DefaultEventExecutorGroup {

    private final int maximumPoolSize;
    private final AtomicLong submittedTasks = new AtomicLong();
    private final AtomicLong completedTasks = new AtomicLong();

    public NettyGeneralExecutor(int nThreads, ThreadFactory threadFactory) {
        super(nThreads, threadFactory);
        this.maximumPoolSize = nThreads;
    }

    public int getPoolSize() {
        return executorCount();
    }

    public int getMaximumPoolSize() {
        return maximumPoolSize;
    }

    public long getSubmittedTasks() {
        return submittedTasks.get();
    }

    public long getCompletedTasks() {
        return completedTasks.get();
    }

    public int getActiveCount() {
        int count = 0;
        for (EventExecutor executor : this) {
            if (((EventExecutorImpl) executor).isActive()) {
                count++;
            }
        }
        return count;
    }

    @Override
    protected EventExecutor newChild(Executor executor, Object... args) throws Exception {
        return new EventExecutorImpl(this, executor, (Integer) args[0], (RejectedExecutionHandler) args[1]);
    }

    @Override
    public Future<?> submit(Runnable task) {
        submittedTasks.incrementAndGet();
        return super.submit(task);
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        submittedTasks.incrementAndGet();
        return super.submit(task, result);
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
        submittedTasks.incrementAndGet();
        return super.submit(task);
    }

    @Override
    public <T> List<java.util.concurrent.Future<T>> invokeAll(
        Collection<? extends Callable<T>> tasks) throws InterruptedException
    {
        submittedTasks.addAndGet(tasks.size());
        return super.invokeAll(tasks);
    }

    @Override
    public <T> List<java.util.concurrent.Future<T>> invokeAll(
        Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException
    {
        submittedTasks.addAndGet(tasks.size());
        return super.invokeAll(tasks, timeout, unit);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        submittedTasks.addAndGet(tasks.size());
        return super.invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(
        Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException,
        ExecutionException, TimeoutException
    {
        submittedTasks.addAndGet(tasks.size());
        return super.invokeAny(tasks, timeout, unit);
    }

    @Override
    public void execute(Runnable command) {
        submittedTasks.incrementAndGet();
        super.execute(command);
    }

    /**
     * EVENT EXECUTOR IMPL
     */
    private final class EventExecutorImpl extends SingleThreadEventExecutor {
        EventExecutorImpl(
            EventExecutorGroup parent,
            Executor executor,
            int maxPendingTasks,
            RejectedExecutionHandler rejectedExecutionHandler)
        {
            super(parent, executor, true, maxPendingTasks, rejectedExecutionHandler);
        }

        @Override
        protected void run() {
            do {
                Runnable task = takeTask();
                if (task != null) {
                    task.run();
                    completedTasks.incrementAndGet();
                    updateLastExecutionTime();
                }
            } while (!confirmShutdown());
        }

        public boolean isActive() {
            ThreadProperties tp = threadProperties();
            switch (tp.state()) {
                case NEW:
                case RUNNABLE:
                    return true;
                default:
                    return false;
            }
        }
    }
}
