package ru.yandex.solomon.config.thread;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import io.netty.channel.EventLoopGroup;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.config.netty.NettyGeneralExecutor;
import ru.yandex.solomon.config.protobuf.TThreadPoolConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolsConfig;
import ru.yandex.solomon.util.NettyUtils;
import ru.yandex.solomon.util.concurrent.ForkJoinPools;
import ru.yandex.solomon.util.concurrent.ThreadUtils;

/**
 * @author alexlovkov
 */
@Component
public class LazyThreadPoolProvider implements ThreadPoolProvider {

    private final TThreadPoolsConfig threadPoolsConfig;
    private final Map<String, TThreadPoolConfig> configByName;
    private final Map<String, ExecutorService> executorByName;
    private final EventLoopGroup ioExecutor;
    private final ScheduledExecutorService scheduledExecutorService;

    @Autowired
    public LazyThreadPoolProvider(TThreadPoolsConfig threadPoolsConfig) {
        this.threadPoolsConfig = threadPoolsConfig;
        this.configByName = threadPoolsConfig
            .getThreadPoolsList()
            .stream()
            .collect(Collectors.toMap(TThreadPoolConfig::getName, Function.identity()));
        this.executorByName = new ConcurrentHashMap<>(threadPoolsConfig.getThreadPoolsCount());
        this.ioExecutor = NettyUtils.createEventLoopGroup(SystemThreadPools.IO.getName(), getThreads(SystemThreadPools.IO));
        this.scheduledExecutorService = Executors.newScheduledThreadPool(
            getThreads(SystemThreadPools.SCHEDULER),
            ThreadUtils.newThreadFactory(SystemThreadPools.SCHEDULER.getName()));
    }

    @Override
    public Set<String> getKnownNames() {
        return ImmutableSet.copyOf(configByName.keySet());
    }

    @Override
    public ExecutorService getExecutorService(String threadPoolName, String config) {
        if (threadPoolName.isEmpty()) {
            throw new IllegalArgumentException("not found parameter in config:" + config);
        }
        ExecutorService executorService = executorByName.get(threadPoolName);
        if (executorService != null) {
            return executorService;
        }
        TThreadPoolConfig threadPool = configByName.get(threadPoolName);
        if (threadPool == null) {
            throw new IllegalArgumentException("not found pool with name=" + threadPoolName + " from config:" + config);
        }
        return executorByName.computeIfAbsent(threadPool.getName(), ignore -> createThreadPool(threadPool));
    }

    private ExecutorService createThreadPool(TThreadPoolConfig threadPoolConf) {
        String name = threadPoolConf.getName();
        int threads = threadPoolConf.getThreads();

        switch (threadPoolConf.getThreadPoolType()) {
            case GENERAL:
                return Executors.newFixedThreadPool(threads, ThreadUtils.newThreadFactory(name));

            case FORK_JOIN:
                return ForkJoinPools.newPool(threads, name);

            case NETTY:
                return new NettyGeneralExecutor(threads, ThreadUtils.newThreadFactory(name));

            default:
                throw new IllegalArgumentException("unknown thread pool type: " + threadPoolConf.getThreadPoolType());
        }
    }

    @Override
    public EventLoopGroup getIOExecutor() {
        return ioExecutor;
    }

    @Override
    public ScheduledExecutorService getSchedulerExecutorService() {
        return scheduledExecutorService;
    }

    private int getThreads(SystemThreadPools pool) {
        TThreadPoolConfig threadPool =
            threadPoolsConfig.getThreadPoolsList().stream()
                .filter(e -> e.getName().equals(pool.getName())).findFirst()
                .orElseThrow(() -> new IllegalArgumentException(
                    "not found thread-pool in ThreadPoolsConfig with name=" + pool.getName()));
        return threadPool.getThreads();
    }

    @Override
    public void close() {
        ioExecutor.shutdownGracefully();
        scheduledExecutorService.shutdownNow();
        executorByName.values().forEach(ExecutorService::shutdownNow);
    }
}
