package ru.yandex.solomon.config;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.solomon.config.protobuf.DataUnit;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.config.DataSizeConverter.convert;

/**
 * @author Vladimir Gordiychuk
 */
public class DataSizeConverterTest {

    @Test
    public void toSameSize() throws Exception {
        convertToSame(10, DataUnit.BYTES);
        convertToSame(11, DataUnit.KILOBYTES);
        convertToSame(12, DataUnit.MEGABYTES);
        convertToSame(13, DataUnit.GIGABYTES);
    }

    @Test
    public void toBytesFromHigh() throws Exception {
        assertThat(convert(1, DataUnit.KILOBYTES, DataUnit.BYTES), equalTo((long) 1024));
        assertThat(convert(2, DataUnit.KILOBYTES, DataUnit.BYTES), equalTo((long) 2 * 1024));
        assertThat(convert(1, DataUnit.MEGABYTES, DataUnit.BYTES), equalTo((long) 1024 * 1024));
        assertThat(convert(5, DataUnit.MEGABYTES, DataUnit.BYTES), equalTo((long) 5 * 1024 * 1024));
        assertThat(convert(5, DataUnit.GIGABYTES, DataUnit.BYTES), equalTo((long) 5 * 1024 * 1024 * 1024));
    }

    @Test
    public void toKilobytesFromHigh() throws Exception {
        assertThat(convert(1, DataUnit.MEGABYTES, DataUnit.KILOBYTES), equalTo((long) 1024));
        assertThat(convert(2, DataUnit.MEGABYTES, DataUnit.KILOBYTES), equalTo((long) 2 * 1024));
        assertThat(convert(1, DataUnit.GIGABYTES, DataUnit.KILOBYTES), equalTo((long) 1024 * 1024));
        assertThat(convert(5, DataUnit.GIGABYTES, DataUnit.KILOBYTES), equalTo((long) 5 * 1024 * 1024));
    }

    @Test
    public void toMegabytesFromLow() throws Exception {
        assertThat(convert(1024, DataUnit.KILOBYTES, DataUnit.MEGABYTES), equalTo((long) 1));
        assertThat(convert(5 * 1024, DataUnit.KILOBYTES, DataUnit.MEGABYTES), equalTo((long) 5));
    }

    @Test
    public void bidirectional() throws Exception {
        List<DataUnit> source = Stream.of(DataUnit.values())
                .filter(unit -> unit != DataUnit.UNRECOGNIZED)
                .sorted(Comparator.comparing(DataUnit::getNumber, Comparator.reverseOrder()))
                .collect(Collectors.toList());

        int i = 0;
        for (int index = 0; index < source.size(); index++) {
            DataUnit from = source.get(index);
            long expect = 5;

            for (int toIndex = index; toIndex < source.size(); toIndex++) {
                DataUnit to = source.get(index);
                long converted = convert(expect, from, to);
                long convertedBack = convert(converted, to, from);

                assertThat("Failed convert from " + from + " to " + to + " and back",
                        convertedBack, equalTo(expect)
                );
            }
        }
    }

    private void convertToSame(long value, DataUnit unit) {
        long result = convert(value, unit, unit);
        assertThat(result, equalTo(value));
    }
}
