package ru.yandex.solomon.config;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

import com.google.common.io.Files;
import com.google.protobuf.TextFormat;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolsConfig;
import ru.yandex.solomon.config.protobuf.alert.TAlertingConfig;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonConfigsTest {
    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();

    @Test
    public void findArgValue() {
        String gatewayConf = SolomonConfigs.findArgValue("--config", "arg1", "--config", "gateway.conf", "arg4");
        Assert.assertEquals("gateway.conf", gatewayConf);

        String alertingConf = SolomonConfigs.findArgValue("--config", "arg1", "--config=alerting.conf", "arg4");
        Assert.assertEquals("alerting.conf", alertingConf);

        try {
            SolomonConfigs.findArgValue("--config", "arg1", "--config");
            Assert.fail("expected exception was not thrown");
        } catch (Exception e) {
            Assert.assertEquals("argument --config has no value", e.getMessage());
        }
    }

    @Test
    public void getConfigAndSecretFiles() {
        String[] args = {"arg1", "--secrets", "gateway.secrets", "arg4", "--config", "gateway.conf"};
        String gatewayConf = SolomonConfigs.getConfigFileFromArgs(args);
        Optional<String> gatewaySecrets = SolomonConfigs.getSecretsFileFromArgs(args);

        Assert.assertEquals("gateway.conf", gatewayConf);
        Assert.assertTrue(gatewaySecrets.isPresent());
        Assert.assertEquals("gateway.secrets", gatewaySecrets.get());
    }

    @Test
    public void parseConfigNoIncludes() throws IOException {
        TKikimrClientConfig.Builder kikimrClientConfig = TKikimrClientConfig.newBuilder();
        kikimrClientConfig.setSchemaRoot("/Root/Solomon/Prestable");
        kikimrClientConfig.getGrpcConfigBuilder()
            .addAddresses("conductor_group://kikimr_man_slice_18:2135");

        TAlertingConfig expected = TAlertingConfig.newBuilder()
                .setKikimrConfig(kikimrClientConfig.build())
                .build();

        File cfgFile = tmp.newFile();
        Files.write(TextFormat.printToString(expected), cfgFile, StandardCharsets.UTF_8);

        TAlertingConfig result = SolomonConfigs.parseConfig(cfgFile.getAbsolutePath(), TAlertingConfig.getDefaultInstance());
        assertThat(result, equalTo(expected));
    }

    @Test
    public void parseConfigWithIncludes() throws IOException {
        // Storage
        TKikimrClientConfig.Builder kikimrClientConfig = TKikimrClientConfig.newBuilder();
        kikimrClientConfig.setSchemaRoot("/Root/Solomon/Prestable");
        kikimrClientConfig.getGrpcConfigBuilder()
            .addAddresses("conductor_group://kikimr_man_slice_18:2135");

        TAlertingConfig storage = TAlertingConfig.newBuilder()
                .setKikimrConfig(kikimrClientConfig.build())
                .build();
        File storageCfg = tmp.newFile();
        Files.write(TextFormat.printToString(storage), storageCfg, StandardCharsets.UTF_8);

        // ThreadPool
        TAlertingConfig threadPool = TAlertingConfig.newBuilder()
                .setThreadPoolsConfig(TThreadPoolsConfig.newBuilder()
                        .addThreadPools(TThreadPoolConfig.newBuilder()
                                .setName("My thread pool")
                                .setThreads(42)
                                .build())
                        .build())
                .build();
        File threadPoolsCfg = tmp.newFile();
        Files.write(TextFormat.printToString(threadPool), threadPoolsCfg, StandardCharsets.UTF_8);

        // Initial config
        TAlertingConfig cfg = TAlertingConfig.newBuilder()
                .addInclude(storageCfg.getAbsolutePath())
                .addInclude(threadPoolsCfg.getAbsolutePath())
                .build();
        File initCfg = tmp.newFile();
        Files.write(TextFormat.printToString(cfg), initCfg, StandardCharsets.UTF_8);

        TAlertingConfig expected = TAlertingConfig.newBuilder()
                .addAllInclude(cfg.getIncludeList())
                .setKikimrConfig(storage.getKikimrConfig())
                .setThreadPoolsConfig(threadPool.getThreadPoolsConfig())
                .build();

        TAlertingConfig result = SolomonConfigs.parseConfig(initCfg.getAbsolutePath(), TAlertingConfig.getDefaultInstance());
        assertThat(result, equalTo(expected));
    }
}
