package ru.yandex.solomon.coremon.client;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.LongSupplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;
import io.grpc.Status;

import ru.yandex.cluster.discovery.ClusterDiscoveryImpl;
import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.FilterDiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsResponse;
import ru.yandex.monitoring.coremon.TCreateShardRequest;
import ru.yandex.monitoring.coremon.TCreateShardResponse;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TInitShardRequest;
import ru.yandex.monitoring.coremon.TPulledDataRequest;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.monitoring.coremon.TReloadShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardResponse;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.coremon.client.grpc.CoremonHostClient;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.util.UnknownShardLocation;
import ru.yandex.solomon.util.client.ClientFutures;

import static com.google.common.base.Preconditions.checkState;
import static java.util.concurrent.CompletableFuture.failedFuture;


/**
 * @author Sergey Polovko
 */
public class CoremonClientImpl implements CoremonClient {

    private static final LongSupplier EXTRA_AWAIT_MILLIS = () -> 3_000;

    private final Map<String, CoremonClusterState> clusterById;

    public CoremonClientImpl(
        String clientId,
        TGrpcClientConfig config,
        ThreadPoolProvider threadPoolProvider,
        ClusterMapper clusterMapper,
        ClientOptionsFactory factory)
    {
        var options = factory.newBuilder("CoremonClientConfig", config)
            .setClientId(clientId)
            .build();

        var discovery = DiscoveryService.async();
        var timer = threadPoolProvider.getSchedulerExecutorService();
        var executor = threadPoolProvider.getExecutorService("CpuLowPriority", "");
        Map<String, CoremonClusterState> clusterById = new LinkedHashMap<>();
        for (var clusterId : clusterMapper.knownClusterIds()) {
            var filterDiscovery = new FilterDiscoveryService(discovery, address -> {
                return Objects.equals(clusterId, clusterMapper.byFqdnOrNull(address.getHost()));
            });

            var clusterDiscovery = new ClusterDiscoveryImpl<>(
                    hostAndPort -> new CoremonHostClient(hostAndPort, options),
                    config.getAddressesList(),
                    filterDiscovery,
                    timer,
                    executor,
                    TimeUnit.HOURS.toMillis(1));

            clusterById.put(clusterId, new CoremonClusterState(clusterId, executor, timer, clusterDiscovery));
        }
        this.clusterById = clusterById;
    }

    @Override
    public CompletableFuture<TDataProcessResponse> processPulledData(String host, TPulledDataRequest request) {
        CoremonHostClient client = clientByHostOrNull(host);
        if (client == null) {
            return unknownHost(host);
        }
        return client.processPulledData(request);
    }

    @Override
    public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
        CoremonHostClient client = clientByHostOrNull(host);
        if (client == null) {
            return unknownHost(host);
        }
        return client.processPushedData(request);
    }

    @Override
    public CompletableFuture<List<String>> initShard(String projectId, String shardId, int numId) {
        TInitShardRequest request = TInitShardRequest.newBuilder()
            .setProjectId(projectId)
            .setShardId(shardId)
            .setNumId(numId)
            .setTtl(3)
            .build();
        List<CompletableFuture<String>> futures = clusterById.values().stream()
            .map(c -> c.initShard(request))
            .collect(Collectors.toList());
        return CompletableFutures.allOf(futures)
            .thenApply(l -> l); // convert ListF -> List
    }

    @Override
    public CompletableFuture<ShardInfo> createShard(
            String projectId,
            String clusterName,
            String serviceName,
            String createdBy)
    {
        TCreateShardRequest request = TCreateShardRequest.newBuilder()
            .setProjectId(projectId)
            .setClusterName(clusterName)
            .setServiceName(serviceName)
            .setCreatedBy(createdBy)
            .setTtl(3)
            .build();

        var futures = clusterById.values().stream()
            .map(c -> c.createShard(request))
            .collect(Collectors.toList());

        return ClientFutures.allOrAnyOkOf(futures, EXTRA_AWAIT_MILLIS)
            .thenApply(responses -> {
                String shardId = "";
                int numId = 0;
                var hosts = new ArrayList<String>(responses.size());
                for (TCreateShardResponse response : responses) {
                    if (!shardId.isEmpty()) {
                        checkState(
                                shardId.equals(response.getShardId()),
                                "different shard ids: %s != %s",
                                shardId, response.getShardId());
                        checkState(
                                numId == response.getNumId(),
                                "different num ids: %d != %d",
                                numId, response.getNumId());
                    }
                    shardId = response.getShardId();
                    numId = response.getNumId();
                    hosts.add(response.getAssignedToHost());
                }
                return new ShardInfo(shardId, numId, ImmutableList.copyOf(hosts));
            });
    }

    @Override
    public CompletableFuture<Void> removeShard(String projectId, String shardId, int numId) {
        TRemoveShardRequest request = TRemoveShardRequest.newBuilder()
            .setProjectId(projectId)
            .setShardId(shardId)
            .setNumId(numId)
            .build();

        var futures = new ArrayList<CompletableFuture<TRemoveShardResponse>>(clusterById.size());
        for (var cluster : clusterById.values()) {
            futures.add(cluster.removeShard(request));
        }

        return CompletableFutures.allOfVoid(futures);
    }

    @Override
    public CompletableFuture<TRemoveShardResponse> removeShard(String clusterId, TRemoveShardRequest request) {
        var cluster = clusterById.get(clusterId);
        if (cluster == null) {
            return clusterNotFound(clusterId);
        }

        return cluster.removeShard(request);
    }

    @Override
    public CompletableFuture<Void> reloadShard(String host, String projectId, String shardId) {
        CoremonHostClient client = clientByHostOrNull(host);
        if (client == null) {
            return unknownHost(host);
        }
        TReloadShardRequest request = TReloadShardRequest.newBuilder()
            .setProjectId(projectId)
            .setShardId(shardId)
            .build();
        return client.reloadShard(request).thenAccept(r -> {});
    }

    @Override
    public CompletableFuture<DeleteMetricsResponse> deleteMetrics(String clusterId, DeleteMetricsRequest request) {
        var cluster = clusterById.get(clusterId);
        if (cluster == null) {
            return clusterNotFound(clusterId);
        }

        return cluster.deleteMetrics(request);
    }

    @Override
    public CompletableFuture<Task> getTask(String clusterId, GetTaskRequest request) {
        var cluster = clusterById.get(clusterId);
        if (cluster == null) {
            return clusterNotFound(clusterId);
        }

        return cluster.getTask(request);
    }

    @Override
    public ImmutableList<String> shardHosts(int shardId) {
        var b = ImmutableList.<String>builder();
        for (var cluster : clusterById.values()) {
            String host = cluster.shardHostOrNull(shardId);
            if (host != null) {
                b.add(host);
            }
        }
        var result = b.build();
        if (result.isEmpty()) {
            throw new UnknownShardLocation(shardId);
        }
        return result;
    }

    @Override
    public Set<String> clusterIds() {
        return Collections.unmodifiableSet(clusterById.keySet());
    }

    @Nullable
    private CoremonHostClient clientByHostOrNull(String host) {
        for (var cluster : clusterById.values()) {
            var client = cluster.clientByFqdn(host);
            if (client != null) {
                return client;
            }
        }
        return null;
    }

    private static <T> CompletableFuture<T> unknownHost(String host) {
        return failedFuture(new IllegalArgumentException("unknown host: " + host));
    }

    private static <T> CompletableFuture<T> clusterNotFound(String clusterId) {
        return failedFuture(
            Status.NOT_FOUND.withDescription("cluster with id " + clusterId + " not found").asRuntimeException());
    }
}
