package ru.yandex.solomon.coremon.client;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.common.collect.ImmutableList;
import com.google.protobuf.Any;
import io.grpc.Status;
import io.grpc.protobuf.StatusProto;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.coremon.api.task.RemoveShardParams;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsResponse;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TPulledDataRequest;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.monitoring.coremon.TRemoveShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardResponse;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;

import static java.util.concurrent.CompletableFuture.failedFuture;
import static java.util.stream.Collectors.toUnmodifiableList;

/**
 * @author Sergey Polovko
 */
public class CoremonClientStub implements CoremonClient {
    public volatile Supplier<CompletableFuture<?>> beforeSupplier;
    private final ConcurrentMap<String, Cluster> clusterById = new ConcurrentHashMap<>();

    @Override
    public CompletableFuture<TDataProcessResponse> processPulledData(String host, TPulledDataRequest request) {
        return unsupported("processPulledData");
    }

    @Override
    public CompletableFuture<TDataProcessResponse> processPushedData(String host, TPushedDataRequest request) {
        return unsupported("processPushedData");
    }

    @Override
    public CompletableFuture<List<String>> initShard(String projectId, String shardId, int numId) {
        return unsupported("initShard");
    }

    @Override
    public CompletableFuture<ShardInfo> createShard(
            String projectId,
            String clusterName,
            String serviceName,
            String createdBy)
    {
        return unsupported("createShard");
    }

    @Override
    public CompletableFuture<Void> removeShard(String projectId, String shardId, int numId) {
        return async(() -> {
            var req = TRemoveShardRequest.newBuilder()
                    .setNumId(numId)
                    .setShardId(shardId)
                    .setProjectId(projectId)
                    .build();

            clusterById.values().forEach(cluster -> cluster.removeShard(req));
        });
    }

    @Override
    public CompletableFuture<TRemoveShardResponse> removeShard(String clusterId, TRemoveShardRequest request) {
        return async(() -> getClusterOrThrow(clusterId).removeShard(request));
    }

    @Override
    public CompletableFuture<Void> reloadShard(String host, String projectId, String shardId) {
        return unsupported("reloadShard");
    }

    @Override
    public CompletableFuture<DeleteMetricsResponse> deleteMetrics(String clusterId, DeleteMetricsRequest request) {
        return async(() -> getClusterOrThrow(clusterId).deleteMetrics(request));
    }

    @Override
    public CompletableFuture<Task> getTask(String clusterId, GetTaskRequest request) {
        return async(() -> getClusterOrThrow(clusterId).getTask(request));
    }

    @Override
    public ImmutableList<String> shardHosts(int shardId) {
        return ImmutableList.of();
    }

    @Override
    public Set<String> clusterIds() {
        return clusterById.keySet();
    }

    public void addCluster(String clusterId) {
        clusterById.put(clusterId, new Cluster());
    }

    public void putTask(String clusterId, Task task) {
        getClusterOrThrow(clusterId).taskById.put(task.getId(), task);
    }

    public Task taskById(String clusterId, String taskId) {
        return getClusterOrThrow(clusterId).taskById.get(taskId);
    }

    public Task removeTaskById(String clusterId, String taskId) {
        return getClusterOrThrow(clusterId).taskById.remove(taskId);
    }

    public Collection<Task> tasks() {
        return clusterById.values().stream()
            .flatMap(Cluster::tasks)
            .collect(toUnmodifiableList());
    }

    public Collection<Task> tasks(String clusterId) {
        return clusterById.get(clusterId).tasks().collect(toUnmodifiableList());
    }

    public void beforeCluster(String clusterId, Runnable before) {
        getClusterOrThrow(clusterId).before = before;
    }

    private Cluster getClusterOrThrow(String clusterId) {
        var cluster = clusterById.get(clusterId);
        if (cluster == null) {
            throw Status.NOT_FOUND.withDescription("cluster '" + clusterId + "' not found").asRuntimeException();
        }
        return cluster;
    }

    private CompletableFuture<Void> async(Runnable fn) {
        return before().thenRunAsync(fn);
    }

    private <T> CompletableFuture<T> async(Supplier<T> fn) {
        return before().thenApplyAsync(ignore -> fn.get());
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return CompletableFuture.completedFuture(null);
        }

        return copy.get();
    }

    private static <T> CompletableFuture<T> unsupported(String name) {
        return failedFuture(new UnsupportedOperationException("method " + name + " not supported"));
    }

    private static class Cluster {
        private volatile Runnable before = () -> {
        };

        private final ConcurrentMap<String, Task> taskById = new ConcurrentHashMap<>();

        TRemoveShardResponse removeShard(TRemoveShardRequest request) {
            before.run();

            var params = Any.pack(RemoveShardParams.newBuilder()
                    .setNumId(request.getNumId())
                    .setProjectId(request.getProjectId())
                    .setShardId(request.getShardId())
                    .build());

            var task = Task.newBuilder()
                    .setId("remove_shard_" + Integer.toUnsignedLong(request.getNumId()))
                    .setType("remove_shard")
                    .setExecuteAt(System.currentTimeMillis())
                    .setParams(params)
                    .build();

            taskById.putIfAbsent(task.getId(), task);
            return TRemoveShardResponse.newBuilder()
                    .setTaskId(task.getId())
                    .build();
        }

        DeleteMetricsResponse deleteMetrics(DeleteMetricsRequest request) {
            before.run();

            var phase = request.getPhase().name().toLowerCase();
            var taskId = "%s_%s_%d".formatted(
                request.getOperationId(),
                phase,
                Integer.toUnsignedLong(request.getNumId()));

            if (request.getInterrupt()) {
                var task = taskById.computeIfPresent(taskId, (id, t) -> {
                    if (t.getState() == Task.State.COMPLETED) {
                        return t;
                    }

                    var oldStatus = StatusProto.toStatusException(t.getStatus()).getStatus().toString();
                    return t.toBuilder()
                        .setState(Task.State.COMPLETED)
                        .setStatus(
                            StatusProto.fromStatusAndTrailers(
                                Status.CANCELLED.withDescription(oldStatus),
                                null))
                        .build();
                });

                return DeleteMetricsResponse.newBuilder()
                    .setTaskId(task == null ? "" : task.getId())
                    .build();
            }

            var params = Any.pack(
                DeleteMetricsParams.newBuilder()
                    .setOperationId(request.getOperationId())
                    .setNumId(request.getNumId())
                    .setSelectors(request.getSelectors())
                    .setCreatedAt(request.getCreatedAt())
                    .build());

            var task = Task.newBuilder()
                .setId(taskId)
                .setType("delete_metrics_" + phase)
                .setExecuteAt(System.currentTimeMillis())
                .setParams(params)
                .setState(Task.State.RUNNING)
                .build();

            taskById.putIfAbsent(task.getId(), task);
            return DeleteMetricsResponse.newBuilder()
                .setTaskId(task.getId())
                .build();
        }

        Task getTask(GetTaskRequest request) {
            before.run();

            var task = taskById.get(request.getId());
            if (task == null) {
                throw Status.NOT_FOUND.withDescription("not found task").asRuntimeException();
            }
            return task;
        }

        Stream<Task> tasks() {
            return taskById.values().stream();
        }
    }
}
