package ru.yandex.solomon.coremon.client;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import io.grpc.Status;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsResponse;
import ru.yandex.monitoring.coremon.TCreateShardRequest;
import ru.yandex.monitoring.coremon.TCreateShardResponse;
import ru.yandex.monitoring.coremon.TInitShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardResponse;
import ru.yandex.monitoring.coremon.TShardAssignmentsRequest;
import ru.yandex.solomon.coremon.client.grpc.CoremonHostClient;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.collection.Nullables;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Sergey Polovko
 */
final class CoremonClusterState implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CoremonClusterState.class);

    private static final TShardAssignmentsRequest SHARDS_ASSIGNMENTS_REQUEST = TShardAssignmentsRequest.newBuilder()
        .setTtl(3)
        .build();

    private final String clusterId;
    private final ClusterDiscovery<CoremonHostClient> discovery;

    // shard num id -> fqdn
    private final AtomicReference<Int2ObjectMap<String>> shardsMap = new AtomicReference<>(Int2ObjectMaps.emptyMap());
    @Nullable
    private volatile CoremonHostClient leaderClient;

    private final PingActorRunner actor;

    CoremonClusterState(String clusterId, Executor executor, ScheduledExecutorService timer, ClusterDiscovery<CoremonHostClient> discovery) {
        this.clusterId = clusterId;
        this.discovery = discovery;
        this.actor = PingActorRunner.newBuilder()
                .operation("refresh_coremon_shard_assignments_for_" + clusterId)
                .executor(executor)
                .timer(timer)
                .pingInterval(Duration.ofSeconds(5))
                .backoffMaxDelay(Duration.ofSeconds(30))
                .onPing(this::updateClusterState)
                .build();
        actor.forcePing();
    }

    CompletableFuture<String> initShard(TInitShardRequest request) {
        String knownHost = shardsMap.get().get(request.getNumId());
        if (knownHost != null) {
            return completedFuture(knownHost);
        }
        var leader = leaderClient;
        if (leader == null) {
            return leaderUnavailable();
        }
        return leader.initShard(request)
            .thenApply(response -> {
                updateLeaderClient(response.getLeaderHost());

                String host = response.getAssignedToHost();
                rememberShardHost(request.getNumId(), host);
                return host;
            });
    }

    CompletableFuture<TCreateShardResponse> createShard(TCreateShardRequest request) {
        var leader = leaderClient;
        if (leader == null) {
            return leaderUnavailable();
        }
        return leader.createShard(request)
            .thenApply(response -> {
                updateLeaderClient(response.getLeaderHost());

                String host = response.getAssignedToHost();
                rememberShardHost(response.getNumId(), host);
                return response;
            });
    }

    CompletableFuture<TRemoveShardResponse> removeShard(TRemoveShardRequest request) {
        var client = Nullables.orDefault(shardHostClientOrNull(request.getNumId()), leaderClient);
        if (client == null) {
            return leaderUnavailable();
        }

        return client.removeShard(request);
    }

    CompletableFuture<DeleteMetricsResponse> deleteMetrics(DeleteMetricsRequest request) {
        var client = Nullables.orDefault(shardHostClientOrNull(request.getNumId()), leaderClient);
        if (client == null) {
            return leaderUnavailable();
        }

        return client.deleteMetrics(request);
    }

    CompletableFuture<Task> getTask(GetTaskRequest request) {
        var node = randomHost();
        if (node == null) {
            return leaderUnavailable();
        }
        return node.getTask(request);
    }

    String shardHostOrNull(int shardId) {
        Int2ObjectMap<String> shardsMap = this.shardsMap.get();
        var host = shardsMap.get(shardId);
        if (host == null) {
            actor.forcePing();
        }
        return host;
    }

    CoremonHostClient shardHostClientOrNull(int shardId) {
        var fqdn = shardHostOrNull(shardId);
        if (fqdn == null) {
            return null;
        }

        return clientByFqdn(fqdn);
    }

    @Nullable
    CoremonHostClient clientByFqdn(String fqdn) {
        if (discovery.getNodes().contains(fqdn)) {
            return discovery.getTransportByNode(fqdn);
        }
        return null;
    }

    private CompletableFuture<?> updateClusterState(int attempt) {
        if (leaderClient == null) {
            leaderClient = randomHost();
        }

        var leader = leaderClient;
        if (leader == null) {
            return leaderUnavailable();
        }

        return leader.getShardAssignments(SHARDS_ASSIGNMENTS_REQUEST)
                .thenAccept(response -> {
                    // TODO: dynamically expand clients map using last hosts list from response
                    updateLeaderClient(response.getLeaderHost());
                    Int2ObjectMap<String> shardsMap = ShardAssignmentsSerializer.fromResponse(response);
                    this.shardsMap.set(shardsMap);
                    logger.info("get {} shard assignments on cluster {}", shardsMap.size(), clusterId);
                }).whenComplete((ignore, e) -> {
                    if (e != null) {
                        this.leaderClient = randomHost();
                    }
                });
    }

    @Nullable
    private CoremonHostClient randomHost() {
        CoremonHostClient host = null;
        for (int index = 0; index < 5; index++) {
            host = Nullables.orDefault(discovery.getTransport(), host);
            if (host != null && host.isReady()) {
                return host;
            }
        }
        return Nullables.orDefault(host, leaderClient);
    }

    @ManagerMethod
    private List<String> readyHosts() {
        return discovery.getNodes()
                .stream()
                .filter(fqdn -> discovery.getTransportByNode(fqdn).isReady())
                .collect(Collectors.toList());
    }

    private void updateLeaderClient(String leaderHost) {
        if (!discovery.hasNode(leaderHost)) {
            return;
        }

        leaderClient = discovery.getTransportByNode(leaderHost);
    }

    private void rememberShardHost(int shardId, String host) {
        Int2ObjectMap<String> oldValue, newValue;
        do {
            oldValue = shardsMap.get();
            if (oldValue.containsKey(shardId)) {
                return;
            }
            newValue = new Int2ObjectOpenHashMap<>(oldValue);
            newValue.put(shardId, host);
        } while (!shardsMap.compareAndSet(oldValue, newValue));
    }

    private <T> CompletableFuture<T> leaderUnavailable() {
        return failedFuture(Status.UNAVAILABLE.withDescription("unknown leader at: " + clusterId).asRuntimeException());
    }

    @Override
    public void close() {
        actor.close();
    }
}
