package ru.yandex.solomon.coremon.client;

import java.util.Arrays;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;

import ru.yandex.monitoring.coremon.TShardAssignmentsResponse;
import ru.yandex.solomon.util.host.HostUtils;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Sergey Polovko
 */
public final class ShardAssignmentsSerializer {
    private ShardAssignmentsSerializer() {}


    public static TShardAssignmentsResponse toResponse(String[] hosts, Int2ObjectMap<String> shard2Host) {
        var host2Id = new Object2IntOpenHashMap<String>(hosts.length);
        host2Id.defaultReturnValue(-1);
        for (int i = 0; i < hosts.length; i++) {
            host2Id.put(hosts[i], i);
        }

        var shardIds = new IntArrayList(shard2Host.size());
        var hostIds = new IntArrayList(shard2Host.size());

        for (Int2ObjectMap.Entry<String> e : shard2Host.int2ObjectEntrySet()) {
            int shardId = e.getIntKey();
            int hostId = host2Id.getInt(e.getValue());
            if (hostId == -1) {
                // skip shard assigned to unknown host
                continue;
            }
            shardIds.add(shardId);
            hostIds.add(hostId);
        }

        return TShardAssignmentsResponse.newBuilder()
            .setLeaderHost(HostUtils.getFqdn())
            .addAllHosts(Arrays.asList(hosts))
            .addAllHostIds(hostIds)
            .addAllShardIds(shardIds)
            .build();
    }

    public static Int2ObjectMap<String> fromResponse(TShardAssignmentsResponse response) {
        checkArgument(
            response.getShardIdsCount() == response.getHostIdsCount(),
            "mismatched shardIdsCount(%d) and hostIdsCount(%d)",
            response.getShardIdsCount(),
            response.getHostIdsCount());

        final int count = response.getHostIdsCount();
        final var map = new Int2ObjectOpenHashMap<String>(count);

        for (int i = 0; i < count; i++) {
            final int shardId = response.getShardIds(i);
            final int hostId = response.getHostIds(i);
            final String host = response.getHosts(hostId);
            map.put(shardId, host);
        }

        return map;
    }
}
