package ru.yandex.solomon.coremon.client.grpc;

import java.util.concurrent.CompletableFuture;

import com.google.common.net.HostAndPort;

import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monitoring.coremon.CoremonServiceGrpc;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsResponse;
import ru.yandex.monitoring.coremon.TCreateShardRequest;
import ru.yandex.monitoring.coremon.TCreateShardResponse;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TInitShardRequest;
import ru.yandex.monitoring.coremon.TInitShardResponse;
import ru.yandex.monitoring.coremon.TPulledDataRequest;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.monitoring.coremon.TReloadShardRequest;
import ru.yandex.monitoring.coremon.TReloadShardResponse;
import ru.yandex.monitoring.coremon.TRemoveShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardResponse;
import ru.yandex.monitoring.coremon.TShardAssignmentsRequest;
import ru.yandex.monitoring.coremon.TShardAssignmentsResponse;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.scheduler.proto.TaskSchedulerServiceGrpc;

/**
 * @author Sergey Polovko
 */
public class CoremonHostClient implements AutoCloseable {

    private final String fqdn;
    private final GrpcTransport transport;

    public CoremonHostClient(HostAndPort address, GrpcClientOptions options) {
        this.fqdn = address.getHost();
        this.transport = new GrpcTransport(address, options);
    }

    public String getFqdn() {
        return fqdn;
    }

    public boolean isReady() {
        return transport.isConnected();
    }

    public CompletableFuture<TDataProcessResponse> processPulledData(TPulledDataRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getProcessPulledDataMethod(), request);
    }

    public CompletableFuture<TDataProcessResponse> processPushedData(TPushedDataRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getProcessPushedDataMethod(), request);
    }

    public CompletableFuture<TInitShardResponse> initShard(TInitShardRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getInitShardMethod(), request);
    }

    public CompletableFuture<TCreateShardResponse> createShard(TCreateShardRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getCreateShardMethod(), request);
    }

    public CompletableFuture<TRemoveShardResponse> removeShard(TRemoveShardRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getRemoveShardMethod(), request);
    }

    public CompletableFuture<TReloadShardResponse> reloadShard(TReloadShardRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getReloadShardMethod(), request);
    }

    public CompletableFuture<TShardAssignmentsResponse> getShardAssignments(TShardAssignmentsRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getGetShardAssignmentsMethod(), request);
    }

    public CompletableFuture<DeleteMetricsResponse> deleteMetrics(DeleteMetricsRequest request) {
        return transport.unaryCall(CoremonServiceGrpc.getDeleteMetricsMethod(), request);
    }

    public CompletableFuture<Task> getTask(GetTaskRequest request) {
        return transport.unaryCall(TaskSchedulerServiceGrpc.getGetTaskMethod(), request);
    }

    @Override
    public void close() {
        transport.close();
    }

    @Override
    public String toString() {
        return "{fqdn=" + fqdn + ", ready=" + transport.isReady() + ", connected=" + transport.isConnected() + "}";
    }
}
