package ru.yandex.solomon.coremon.client;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.net.HostAndPort;
import io.grpc.Server;
import io.grpc.inprocess.InProcessServerBuilder;
import io.grpc.stub.StreamObserver;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.cluster.discovery.ClusterDiscoveryStub;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.InProcessChannelFactory;
import ru.yandex.monitoring.coremon.CoremonServiceGrpc;
import ru.yandex.monitoring.coremon.TShardAssignmentsRequest;
import ru.yandex.monitoring.coremon.TShardAssignmentsResponse;
import ru.yandex.solomon.coremon.client.grpc.CoremonHostClient;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class CoremonClusterStateTest {
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .withLookingForStuckThread(true)
            .build();

    private ManualClock clock;
    private ScheduledExecutorService timer;
    private Int2ObjectMap<String> shard2Host;
    private volatile Peer leader;
    private Peer alice;
    private Peer bob;
    private CoremonClusterState state;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);

        alice = new Peer("alice");
        bob = new Peer("bob");

        shard2Host = new Int2ObjectOpenHashMap<>();

        alice.start();
        bob.start();
    }

    @After
    public void tearDown() {
        alice.stop();
        bob.stop();
        timer.shutdownNow();
        if (state != null) {
            state.close();
        }
    }

    @Test
    public void resolveHostByNumId() throws InterruptedException {
        shard2Host.put(42, alice.name);
        shard2Host.put(54, bob.name);

        leader = alice;
        state = createState();
        awaitSync(leader, 2);

        assertEquals(alice.name, state.shardHostOrNull(42));
        assertEquals(bob.name, state.shardHostOrNull(54));
        assertNull(state.shardHostOrNull(33));
    }

    @Test
    public void firstNodeUnavailable() throws InterruptedException {
        shard2Host.put(42, alice.name);
        shard2Host.put(54, bob.name);

        leader = bob;
        alice.stop();
        state = createState();
        awaitSync(leader, 2);

        assertEquals(alice.name, state.shardHostOrNull(42));
        assertEquals(bob.name, state.shardHostOrNull(54));
        assertNull(state.shardHostOrNull(33));
    }

    @Test
    public void secondNodeUnavailable() throws InterruptedException {
        shard2Host.put(42, alice.name);
        shard2Host.put(54, bob.name);

        leader = alice;
        bob.stop();
        state = createState();
        awaitSync(leader, 2);

        assertEquals(alice.name, state.shardHostOrNull(42));
        assertEquals(bob.name, state.shardHostOrNull(54));
        assertNull(state.shardHostOrNull(33));
    }

    @Test
    public void leaderBecomeUnavailable() throws InterruptedException {
        shard2Host.put(42, alice.name);
        shard2Host.put(54, bob.name);

        leader = alice;
        state = createState();
        awaitSync(leader, 2);

        assertEquals(alice.name, state.shardHostOrNull(42));
        assertEquals(bob.name, state.shardHostOrNull(54));

        alice.stop();
        leader = bob;

        shard2Host.put(33, bob.name);

        awaitSync(leader, 2);

        assertEquals(alice.name, state.shardHostOrNull(42));
        assertEquals(bob.name, state.shardHostOrNull(54));
        assertEquals(bob.name, state.shardHostOrNull(33));
    }

    private void awaitSync(Peer peer, int count) throws InterruptedException {
        for (int index = 0; index < count; index++) {
            var sync = peer.sync.get();
            do {
                clock.passedTime(10, TimeUnit.SECONDS);
            } while (!sync.await(1, TimeUnit.MILLISECONDS));
        }
    }

    private CoremonClusterState createState() {
        var discovery = new ClusterDiscoveryStub<CoremonHostClient>();
        for (var peer : List.of(alice, bob)) {
            discovery.add(peer.fqdn().getHost(), new CoremonHostClient(peer.fqdn(), DefaultClientOptions.newBuilder()
                    .setChannelFactory(new InProcessChannelFactory())
                    .build()));
        }
        return new CoremonClusterState("test", ForkJoinPool.commonPool(), timer, discovery);
    }

    private class Peer extends CoremonServiceGrpc.CoremonServiceImplBase {
        private final String name;
        private final Server server;
        private final AtomicReference<CountDownLatch> sync = new AtomicReference<>(new CountDownLatch(1));

        public Peer(String name) {
            this.name = name;
            this.server = InProcessServerBuilder.forName(name)
                    .addService(this)
                    .build();
        }

        public void start() throws IOException {
            server.start();
        }

        private HostAndPort fqdn() {
            return HostAndPort.fromHost(name);
        }

        public void stop() {
            server.shutdownNow();
        }

        @Override
        public void getShardAssignments(TShardAssignmentsRequest request, StreamObserver<TShardAssignmentsResponse> responseObserver) {
            var copy = sync.get();
            try {
                if (this != leader) {
                    leader.getShardAssignments(request, responseObserver);
                } else {
                    responseObserver.onNext(assignments());
                    responseObserver.onCompleted();
                }
            } finally {
                sync.compareAndSet(copy, new CountDownLatch(1));
                copy.countDown();
            }
        }

        private TShardAssignmentsResponse assignments() {
            String[] hosts = {alice.name, bob.name};
            return ShardAssignmentsSerializer.toResponse(hosts, shard2Host)
                    .toBuilder()
                    .setLeaderHost(leader.name)
                    .build();
        }
    }
}
